/*
    CaPriCe for Palm OS - Amstrad CPC 464/664/6128 emulator for Palm devices
    Copyright (C) 2009  Frdric Coste

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <PalmOS.h>
#include <ByteOrderUtils.h>

#include "..\NativeCode\Native_CPC.h"
#include "Caprice.h"
#include "Sound.h"
#include "Math.h"
#include "Trace.h"


//===================
// PATCH begin
#ifdef _PATCH_ENABLE

#endif /* _PATCH_ENABLE */
// PATCH end
//===================


#undef FORCE_SNDSTREAM_NORMAL
//#define FORCE_SNDSTREAM_NORMAL

#undef FORCE_SNDSTREAM_EXTENDED
//#define FORCE_SNDSTREAM_EXTENDED


#define SOUND_MUTE_VOLUME           0

#define SOUND_VOLUME_STEPS          100
#define SOUND_VOLUME_CHANGE_STEP    2

#define SOUND_SAMPLE_GAIN           4096 // 1024 = 100% Palm Sample Gain
#define SOUND_SCALE_VOLUME(vol)     ((tLong)vol /* From 0 to 100 */ * SOUND_SAMPLE_GAIN / SOUND_VOLUME_STEPS)

#define SoundExtendedMinVersion     sysMakeROMVersion(5,4,0,sysROMStageDevelopment,0)


#if SND_16BITS == 0
#define SND_STREAM_BITS  sndUInt8
#else
#define SND_STREAM_BITS  sndInt16Little
#endif

#if SND_STEREO == 0
#define SND_STREAM_STEREO  sndMono
#else /* SND_STEREO */
#define SND_STREAM_STEREO  sndStereo
#endif /* SND_STEREO */


static SndStreamRef sndStream = NULL;

#ifdef SNDSTREAM_NATIVE
static MemHandle SoundCallbackH = NULL;
static MemPtr SoundCallbackP = NULL;
#endif /* SNDSTREAM_NATIVE */

// Amplitude table (c)Hacker KAY
static const tUShort Amplitudes_AY[16] =
{
   0, 836, 1212, 1773, 2619, 3875, 5397, 8823,
   10392, 16706, 23339, 29292, 36969, 46421, 55195, 65535
};


static Err SoundCallback(void *userDataP,
                         SndStreamRef stream,
                         void *bufferP,
                         UInt32 FrameCount);
static Err SoundVariableCallback(void *userDataP,
                                 SndStreamRef stream,
                                 void *bufferP,
                                 UInt32* bufferSizeP);


Err SoundStart(tUChar* contextP)
/***********************************************************************
 *
 *  SoundStart
 *
 ***********************************************************************/
#undef SOUNDSTART_TRACE_ENABLED
//#define SOUNDSTART_TRACE_ENABLED

#ifdef SOUNDSTART_TRACE_ENABLED
#  define SOUNDSTART_TRACE_SHOW_INT(value) TRACE_SHOW_INT("SoundStart", value)
#else /* SOUNDSTART_TRACE_ENABLED */
#  define SOUNDSTART_TRACE_SHOW_INT(value)
#endif /* SOUNDSTART_TRACE_ENABLED */
{
tPSG* PSG;
UInt32 romVersion;
tULong buffersize;
Err Error;
tNativeCPC* NativeCPC = (tNativeCPC*)(contextP + CONTEXT_OFFSET_NATIVECPC);
tSoundCallbackParam* SoundCbParamP = (tSoundCallbackParam*)(contextP + CONTEXT_OFFSET_SOUND_CB_PARAM);

#ifndef __RELEASE__
  if (NativeCPC == NULL)
    ErrDisplay("NativeCPC == NULL");
  if (SoundCbParamP == NULL)
    ErrDisplay("SoundCbParamP == NULL");
#endif /* __RELEASE__ */

  PSG = (tPSG*)EndianSwap32(NativeCPC->PSG);

  SOUNDSTART_TRACE_SHOW_INT(1);


  //
  // Prepare callback parameters
  //

#ifndef __RELEASE__
  if (PSG == NULL)
    ErrDisplay("PSG == NULL");
  if (PSG->pbSndBuffer == NULL)
    ErrDisplay("PSG->pbSndBuffer == NULL");
  if (PSG->snd_bufferptr == NULL)
    ErrDisplay("PSG->snd_bufferptr == NULL");
#endif /* __RELEASE__ */
  
#ifndef SNDSTREAM_NATIVE

  SoundCbParamP->SoundBufferStartP = (tUChar*)EndianSwap32(PSG->pbSndBuffer);
  SoundCbParamP->LastPosP = SoundCbParamP->SoundBufferStartP;
  SoundCbParamP->SoundBufferEndP = (tUChar*)EndianSwap32(PSG->pbSndBufferEnd);
  SoundCbParamP->CurrentPosPP = (tUChar**)(&PSG->snd_bufferptr);
  SoundCbParamP->CurrentSizeP = (tULong*)(&PSG->FilledBufferSize);
  
#else /* SNDSTREAM_NATIVE */

  SoundCbParamP->SoundBufferStartP = PSG->pbSndBuffer;
  SoundCbParamP->LastPosP = SoundCbParamP->SoundBufferStartP;
  SoundCbParamP->SoundBufferEndP = PSG->pbSndBufferEnd;
  SoundCbParamP->CurrentPosPP = (tUChar**)EndianSwap32(&PSG->snd_bufferptr);
  SoundCbParamP->CurrentSizeP = (tULong*)EndianSwap32(&PSG->FilledBufferSize);
  
  // Sound callback native routine
  SoundCallbackH = DmGetResource('ARMC', Native_SoundCallback);

#ifndef __RELEASE__
  if (SoundCallbackH == NULL)
    ErrDisplay("SoundCallbackH == NULL");
#endif /* __RELEASE__ */

  SoundCallbackP = MemHandleLock(SoundCallbackH);

#ifdef __RELEASE__
  if (SoundCallbackP == NULL)
    ErrDisplay("SoundCallbackP == NULL");
#endif /* __RELEASE__ */

#endif /* SNDSTREAM_NATIVE */

  SoundCbParamP->BufferRead = 0;
  
#ifndef __RELEASE__
  SoundCbParamP->DebugCount = 0;
  SoundCbParamP->DebugSamples = 0;
#endif /* __RELEASE__ */


  //  
  // NORMAL streaming
  //
#if !defined(FORCE_SNDSTREAM_NORMAL) && !defined(FORCE_SNDSTREAM_EXTENDED)
  // Retreive ROM version
  FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
  if (romVersion < SoundExtendedMinVersion)
#else /* !defined(FORCE_SNDSTREAM_NORMAL) && !defined(FORCE_SNDSTREAM_EXTENDED) */
#ifdef FORCE_SNDSTREAM_NORMAL
  if (1)
#else /* FORCE_SNDSTREAM_NORMAL */
  if (0)
#endif /* FORCE_SNDSTREAM_NORMAL */
#endif /* !defined(FORCE_SNDSTREAM_NORMAL) && !defined(FORCE_SNDSTREAM_EXTENDED) */
  {
    // Initialise audio stream
    Error = SndStreamCreate(&sndStream,
                            sndOutput,
                            SND_FREQ /*Hz*/,
                            SND_STREAM_BITS,
                            SND_STREAM_STEREO,
                            SoundCallback,
                            SoundCbParamP,
                            SND_STREAM_SAMPLES,
                            false /*68k callback*/);
  }
  //
  // EXTENDED streaming
  //
  else
  {
#ifndef SNDSTREAM_NATIVE
    Error = SndStreamCreateExtended(&sndStream,
                                    sndOutput,
                                    sndFormatPCM,
                                    SND_FREQ /*Hz*/,
                                    SND_STREAM_BITS,
                                    SND_STREAM_STEREO,
                                    SoundVariableCallback,
                                    SoundCbParamP,
                                    SND_STREAM_SAMPLES,
                                    false /*68k callback*/);
#else /* ! SNDSTREAM_NATIVE */
    Error = SndStreamCreateExtended(&sndStream,
                                    sndOutput,
                                    sndFormatPCM,
                                    SND_FREQ /*Hz*/,
                                    SND_STREAM_BITS,
                                    SND_STREAM_STEREO,
                                    SoundCallbackP,
                                    SoundCbParamP,
                                    SND_STREAM_SAMPLES,
                                    true /*ARM callback*/);
#endif /* ! SNDSTREAM_NATIVE */
  }

  if (Error == errNone)
  {
#ifndef SNDSTREAM_NATIVE
    SoundCbParamP->StreamRef = sndStream;
#else /* ! SNDSTREAM_NATIVE */
    SoundCbParamP->StreamRef = EndianSwap32(sndStream);
#endif /* ! SNDSTREAM_NATIVE */
  }
  else // Creation problem
  {
  	// Disable Sound
  	prefP->SoundEnabled = 0;
  	prefP->PreferencesChanged = 1;
  	
#ifndef __RELEASE__
    ErrDisplay("SndStreamCreate() != errNone");
#endif /* __RELEASE__ */
  }
  
  return Error;
}
/*----------------------------------------------------------------------------*/


tVoid SoundStop(tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  SoundStop
 *
 ***********************************************************************/
{
	SoundPause(NativeCPC);
	
  // Librer le flux sonore
  if (sndStream != NULL)
  {
    SndStreamDelete(sndStream);
    sndStream = NULL;
  }

#ifdef SNDSTREAM_NATIVE
  if (SoundCallbackH != NULL)
  {
    MemHandleUnlock(SoundCallbackH);
    DmReleaseResource(SoundCallbackH);
  }
#endif /* SNDSTREAM_NATIVE */
}
/*----------------------------------------------------------------------------*/


tVoid SoundPlay(tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  SoundPlay
 *
 ***********************************************************************/
#undef SOUNDPLAY_TRACE_ENABLED
//#define SOUNDPLAY_TRACE_ENABLED

#ifdef SOUNDPLAY_TRACE_ENABLED
#  define SOUNDPLAY_TRACE_SHOW_INT(value) TRACE_SHOW_INT("SoundPlay", value)
#else /* SOUNDPLAY_TRACE_ENABLED */
#  define SOUNDPLAY_TRACE_SHOW_INT(value)
#endif /* SOUNDPLAY_TRACE_ENABLED */
{
tPSG* PSG = (tPSG*)EndianSwap32(NativeCPC->PSG);

  if (sndStream == NULL)
    return;
    
  if (prefP->SoundEnabled == 0)
    return;

  if (SystemHalt)
    return;

#ifndef __RELEASE__
  if (NativeCPC == NULL)
    ErrDisplay("NativeCPC == NULL");
#endif /* __RELEASE__ */

  SOUNDPLAY_TRACE_SHOW_INT(1);

  PSG->snd_enabled = EndianSwap32(1);

  SOUNDPLAY_TRACE_SHOW_INT(2);

  if (prefP->SoundSystemVolume == 0)
  {
    SndStreamSetVolume(sndStream,
                       SOUND_SCALE_VOLUME(prefP->SoundVolume));
  }
  else
  {
    SndStreamSetVolume(sndStream,
                       sndGameVolume);
  }

  SOUNDPLAY_TRACE_SHOW_INT(3);

  SndStreamStart(sndStream);

  SOUNDPLAY_TRACE_SHOW_INT(4);
}
/*----------------------------------------------------------------------------*/


tVoid SoundPause(tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  SoundPause
 *
 ***********************************************************************/
{
  if (sndStream == NULL)
    return;

#ifndef __RELEASE__
  if (NativeCPC == NULL)
    ErrDisplay("NativeCPC == NULL");
#endif /* __RELEASE__ */

  ((tPSG*)EndianSwap32(NativeCPC->PSG))->snd_enabled = 0;

  if (prefP->SoundEnabled == 1)
  {
    SndStreamSetVolume(sndStream,
                       SOUND_MUTE_VOLUME);
    SndStreamStop(sndStream);
  }
}
/*----------------------------------------------------------------------------*/


tVoid Sound_Calculate_Level_Tables(tULong Param)
/***********************************************************************
 *
 *  Sound_Calculate_Level_Tables
 *
 ***********************************************************************/
{
tDouble k;
tDouble Log2 = LOG_2;
tNativeCPC* NativeCPC;
tPSG* PSG;
tLong i;
tLong b;
tLong l;
tLong r;
tLong Index_A;
tLong Index_B;
tLong Index_C;
tLong Index_AR;
tLong Index_BR;
tLong Index_CR;
tLong* Level_AL;
tLong* Level_AR;
tLong* Level_BL;
tLong* Level_BR;
tLong* Level_CL;
tLong* Level_CR;
tLong* Level_PP;
tLong LevelTape;

  NativeCPC = (tNativeCPC*)Param;
  
#ifndef __RELEASE__
  if (NativeCPC == NULL)
    ErrDisplay("NativeCPC == NULL");
#endif /* __RELEASE__ */

  Index_A  = AUDIO_INDEX_AL;
  Index_B  = AUDIO_INDEX_BL;
  Index_C  = AUDIO_INDEX_CL;
  Index_AR = AUDIO_INDEX_AR;
  Index_BR = AUDIO_INDEX_BR;
  Index_CR = AUDIO_INDEX_CR;

  // PNO to 68k
  PSG = (tPSG*)EndianSwap32(NativeCPC->PSG);
  Level_AR = (tLong*)EndianSwap32(PSG->Level_AR);
  Level_AL = (tLong*)EndianSwap32(PSG->Level_AL);
  Level_BR = (tLong*)EndianSwap32(PSG->Level_BR);
  Level_BL = (tLong*)EndianSwap32(PSG->Level_BL);
  Level_CR = (tLong*)EndianSwap32(PSG->Level_CR);
  Level_CL = (tLong*)EndianSwap32(PSG->Level_CL);
  Level_PP = (tLong*)EndianSwap32(PSG->Level_PP);

  l = Index_A + Index_B + Index_C;
  r = Index_AR + Index_BR + Index_CR;

#if SND_STEREO == 1
  if (l < r)
  {
    l = r;
  }
#else
  l += r;
  Index_A += Index_AR;
  Index_B += Index_BR;
  Index_C += Index_CR;
#endif
  
#if SND_16BITS == 0
  r = 127;
#else
  r = 32767;
#endif

  if (l == 0) l++; // Avoid division by 0
  l = 255 * r / l;

  for (i = 0; i < 16; i++)
  {
    b = (tLong)rint(Index_A / 255.0 * Amplitudes_AY[i]);
    b = (tLong)rint(b / 65535.0 * l);
    Level_AL[i * 2] = b;
    Level_AL[(i * 2) + 1] = b;
    b = (tLong)rint(Index_AR / 255.0 * Amplitudes_AY[i]);
    b = (tLong)rint(b / 65535.0 * l);
    Level_AR[i * 2] = b;
    Level_AR[(i * 2) + 1] = b;
    b = (tLong)rint(Index_B / 255.0 * Amplitudes_AY[i]);
    b = (tLong)rint(b / 65535.0 * l);
    Level_BL[i * 2] = b;
    Level_BL[(i * 2) + 1] = b;
    b = (tLong)rint(Index_BR / 255.0 * Amplitudes_AY[i]);
    b = (tLong)rint(b / 65535.0 * l);
    Level_BR[i * 2] = b;
    Level_BR[(i * 2) + 1] = b;
    b = (tLong)rint(Index_C / 255.0 * Amplitudes_AY[i]);
    b = (tLong)rint(b / 65535.0 * l);
    Level_CL[i * 2] = b;
    Level_CL[(i * 2) + 1] = b;
    b = (tLong)rint(Index_CR / 255.0 * Amplitudes_AY[i]);
    b = (tLong)rint(b / 65535.0 * l);
    Level_CR[i * 2] = b;
    Level_CR[(i * 2) + 1] = b;
  }

  k = exp(AUDIO_SAMPLE_VOLUME * Log2 / AUDIO_PREAMP_MAX) - 1;
  
  for (i = 0; i < 32; i++)
  {
    Level_AL[i] = (tLong)EndianSwap32(rint(Level_AL[i] * k));
    Level_AR[i] = (tLong)EndianSwap32(rint(Level_AR[i] * k));
    Level_BL[i] = (tLong)EndianSwap32(rint(Level_BL[i] * k));
    Level_BR[i] = (tLong)EndianSwap32(rint(Level_BR[i] * k));
    Level_CL[i] = (tLong)EndianSwap32(rint(Level_CL[i] * k));
    Level_CR[i] = (tLong)EndianSwap32(rint(Level_CR[i] * k));
  }

#if SND_16BITS == 0 // 8 bits per sample?
  LevelTape = -(tLong)rint((TAPE_VOLUME / 2) * k);
#else
  LevelTape = -(tLong)rint((TAPE_VOLUME * 128) * k);
#endif
  PSG->LevelTape  = (tLong)EndianSwap32(LevelTape);

   
  for (i = 0, b = 255; i < 256; i++) // calculate the 256 levels of the Digiblaster/Soundplayer
  {
    Level_PP[i] = (tLong)EndianSwap32(-rint((b << 8) * l / 65535.0 * k));
    b--;
  }
}
/*----------------------------------------------------------------------------*/


static Err SoundCallback(void *userDataP,
                         SndStreamRef stream,
                         void *bufferP,
                         UInt32 FrameCount)
/***********************************************************************
 *
 *  SoundCallback
 *
 ***********************************************************************/
{
tSoundCallbackParam* paramP = (tSoundCallbackParam*)userDataP;
tULong FilledBufferSize;
UInt32 BufferSize;

#ifndef __RELEASE__
  if (paramP == NULL)
    ErrDisplay("paramP == NULL");
  if (paramP->SoundBufferStartP == NULL)
    ErrDisplay("paramP->SoundBufferP == NULL");
  if (stream != paramP->StreamRef)
    ErrDisplay("stream != paramP->StreamRef");
#endif /* __RELEASE__ */

  // Get and Reset Current Size
  FilledBufferSize = EndianSwap32(*paramP->CurrentSizeP);

  BufferSize = FrameCount * SND_SAMPLE_SIZE;

  // If buffer is not fullfilled yet
  if (FilledBufferSize < BufferSize)
  {
    MemSet(bufferP,
           BufferSize,
           SND_SAMPLE_ORIGIN);
  	return errNone;
  }

  *paramP->CurrentSizeP = 0;

  // Copy requested amount to data
  MemMove(bufferP,
          paramP->SoundBufferStartP,
          BufferSize);

  // #### Hack into the undocumented Buffer structure to update the fDataUsed field
  //((UInt32*)bufferP)[2] = BufferSize;

  paramP->BufferRead = 1;

#ifndef __RELEASE__
  // Debug info
  paramP->DebugCount++;
  paramP->DebugSamples += FrameCount;
  paramP->DebugMoveSizeCount += BufferSize;
  paramP->DebugFrameCount = FrameCount;
#endif /* __RELEASE__ */
  
  return errNone;
}
/*----------------------------------------------------------------------------*/


static Err SoundVariableCallback(void *userDataP,
                                 SndStreamRef stream,
                                 void *bufferP,
                                 UInt32* bufferSizeP)
/***********************************************************************
 *
 *  SoundVariableCallback
 *
 ***********************************************************************/
{
tSoundCallbackParam* paramP = (tSoundCallbackParam*)userDataP;
tUChar* CurrentPosP;
tULong FilledBufferSize;

#ifndef __RELEASE__
  if (paramP == NULL)
    ErrDisplay("paramP == NULL");
  if (paramP->SoundBufferStartP == NULL)
    ErrDisplay("paramP->SoundBufferP == NULL");
  if (stream != paramP->StreamRef)
    ErrDisplay("stream != paramP->StreamRef");
#endif /* __RELEASE__ */

  // Get Current Pos
  CurrentPosP = (tUChar*)EndianSwap32(*paramP->CurrentPosPP);

  // Get and Reset Current Size
  FilledBufferSize = EndianSwap32(*paramP->CurrentSizeP);
  *paramP->CurrentSizeP = 0;
  
  // Buffer fully filled
  if ( (paramP->LastPosP >= CurrentPosP) && FilledBufferSize )
  {
  	*bufferSizeP = (tULong)paramP->SoundBufferEndP - (tULong)paramP->LastPosP + 1;
  	CurrentPosP = paramP->SoundBufferStartP;
    paramP->BufferRead = 1;
  }
  else
  {
  	*bufferSizeP = (tULong)CurrentPosP - (tULong)paramP->LastPosP;
  }

  // Copy requested amount to data
  MemMove(bufferP,
          paramP->LastPosP,
          *bufferSizeP);

  // Update Last position
  paramP->LastPosP = CurrentPosP;


#ifndef __RELEASE__
  // Debug info
  paramP->DebugCount++;
  paramP->DebugTotalCount++;
  paramP->DebugSamples += *bufferSizeP / SND_SAMPLE_SIZE;
  paramP->DebugMoveSizeCount += *bufferSizeP;
#endif /* __RELEASE__ */

  return errNone;
}
/*----------------------------------------------------------------------------*/


Boolean IsBufferRead(tUChar* contextP)
/***********************************************************************
 *
 *  IsBufferRead
 *
 ***********************************************************************/
{
tSoundCallbackParam* SoundCbParamP = (tSoundCallbackParam*)(contextP + CONTEXT_OFFSET_SOUND_CB_PARAM);

  if (!SoundCbParamP->BufferRead)
    return false;
    
  SoundCbParamP->BufferRead = 0;

  return true;
}
/*----------------------------------------------------------------------------*/


tVoid SoundIncreaseVolume(tUChar step)
/***********************************************************************
 *
 *  SoundIncreaseVolume
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (prefP == NULL)
    ErrDisplay("prefP == NULL");
#endif /* __RELEASE__ */

  prefP->SoundVolume += step;
  if (prefP->SoundVolume > SOUND_VOLUME_STEPS)
  {
    prefP->SoundVolume = SOUND_VOLUME_STEPS;
  }

  SndStreamSetVolume(sndStream,
                     SOUND_SCALE_VOLUME(prefP->SoundVolume));

  // Ask for Preferences save
  prefP->PreferencesChanged = 1;
}
/*----------------------------------------------------------------------------*/

 
tVoid SoundDecreaseVolume(tUChar step)
/***********************************************************************
 *
 *  SoundDecreaseVolume
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (prefP == NULL)
    ErrDisplay("prefP == NULL");
#endif /* __RELEASE__ */

  if (prefP->SoundVolume <= step)
  {
    prefP->SoundVolume = 0;
  }
  else
  {
    prefP->SoundVolume -= step;
  }

  SndStreamSetVolume(sndStream,
                     SOUND_SCALE_VOLUME(prefP->SoundVolume));

  // Ask for Preferences save
  prefP->PreferencesChanged = 1;
}
/*----------------------------------------------------------------------------*/


#ifdef _DEBUG

void ResetSoundDebugInfo(tUChar* contextP)
/***********************************************************************
 *
 *  ResetSoundDebugInfo
 *
 ***********************************************************************/
{
tSoundCallbackParam* SoundCbParamP = (tSoundCallbackParam*)(contextP + CONTEXT_OFFSET_SOUND_CB_PARAM);

  SoundCbParamP->DebugCount = 0;
  SoundCbParamP->DebugSamples = 0;
}
/*----------------------------------------------------------------------------*/

tULong GetSoundDebugInfo_Count(tUChar* contextP)
/***********************************************************************
 *
 *  GetSoundDebugInfo_Count
 *
 ***********************************************************************/
{
tSoundCallbackParam* SoundCbParamP = (tSoundCallbackParam*)(contextP + CONTEXT_OFFSET_SOUND_CB_PARAM);

  return SoundCbParamP->DebugCount;
}

tULong GetSoundDebugInfo_Samples(tUChar* contextP)
/***********************************************************************
 *
 *  GetSoundDebugInfo_Samples
 *
 ***********************************************************************/
{
tSoundCallbackParam* SoundCbParamP = (tSoundCallbackParam*)(contextP + CONTEXT_OFFSET_SOUND_CB_PARAM);

  return SoundCbParamP->DebugSamples;
}

#endif /* _DEBUG */
