/*
    CaPriCe for Palm OS - Amstrad CPC 464/664/6128 emulator for Palm devices
    Copyright (C) 2009  Frdric Coste

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <PalmOS.h>
#include <PceNativeCall.h>
#include <FeatureMgr.h>
#include <ByteOrderUtils.h>

#include "Resources.h"
#include "Routines.h"
#include "CaPriCe.h"
#include "Trace.h"
#include "..\NativeCode\Native_CRC32.h"


SysAppInfoPtr SysGetAppInfo(SysAppInfoPtr *uiAppPP,
                            SysAppInfoPtr
                            *actionCodeAppPP)
              SYS_TRAP(sysTrapSysGetAppInfo);


#define memNewChunkFlagAllowLarge     0x1000



Err MemPtrNewLarge(UInt32 size,
                   void **newPtr)
/***********************************************************************
 *
 *  MemPtrNewLarge
 *
 ***********************************************************************/
{
  SysAppInfoPtr blankAppInfoP;
  UInt16 ownerID;
  
  ownerID = SysGetAppInfo(&blankAppInfoP,
                          &blankAppInfoP)->memOwnerID;
  
  *newPtr = MemChunkNew(0,
                        size,
                        ownerID | memNewChunkFlagNonMovable | memNewChunkFlagAllowLarge);
  
  if (*newPtr == NULL) return memErrNotEnoughSpace;
  
  return errNone;
}
/*----------------------------------------------------------------------------*/


UInt32 PceNativeCallResource(DmResID resID,
#ifdef __SIMU__
                             char* DLLEntryPointP,
#endif /* __SIMU__ */
                             void* userDataP)
/***********************************************************************
 *
 *  PceNativeCallResource
 *
 ***********************************************************************/
{
#ifdef __SIMU__

  // running on Simulator; call the DLL
  return PceNativeCall((NativeFuncType*)DLLEntryPointP,
                       userDataP);
                       
#else /* __SIMU */

MemHandle armH;
MemPtr    armP;
UInt32    result;

  // running on ARM; call the actual ARM resource
  armH = DmGetResource('ARMC',
                       resID);
  armP = MemHandleLock(armH);
    
  if (!armP)
  {
    FrmCustomAlert(CustomTraceAlert,
                   "PceNativeResourceCall",
                   "armP Null",
                   ".");
  }

  result = PceNativeCall(armP,
                         userDataP);
      
  MemHandleUnlock(armH);
  DmReleaseResource(armH);

  return result;
    
#endif /* __SIMU__ */
}
/*----------------------------------------------------------------------------*/


UInt32 PceNativeCallMemPtr(MemPtr resP,
#ifdef __SIMU__
                           char* DLLEntryPointP,
#endif /* __SIMU__ */
                           void* userDataP)
/***********************************************************************
 *
 *  PceNativeCallMemPtr
 *
 ***********************************************************************/
{
#ifdef __SIMU__ 

  // running on Simulator; call the DLL
  return (PceNativeCall((NativeFuncType*)DLLEntryPointP,
                        userDataP));

#else /* __SIMU__ */
                        
  return (PceNativeCall(resP,
                        userDataP));

#endif /* __SIMU__ */
}
/*----------------------------------------------------------------------------*/


MemPtr DmCreateLargeResource(UInt32 AppCreator,
                             UInt32 resFtrNum,
                             DmResType type,
                             DmResID FirstID,
                             tULong FinalSize)
/***********************************************************************
 *
 *  DmCreateLargeResource
 *
 ***********************************************************************/
{
MemPtr    ftrP;
MemHandle resH;
MemPtr    resP;
tUChar*   addressP;
tULong    offset = 0;
tULong    size;

  // Free feature just in case we exited without freeing it last time
  if (FtrGet(AppCreator,
             resFtrNum,
             (UInt32*)&addressP) == 0)
  {
    FtrPtrFree(AppCreator, resFtrNum);
  }

  // Allocate feature memory
  if (FtrPtrNew(AppCreator,
                resFtrNum,
                FinalSize,
                (void **)&ftrP))
  {
    return NULL;
  }

  do
  {
    resH = DmGetResource(type,
                         FirstID++);
    if (resH == NULL)
    {
      FtrPtrFree(AppCreator,
                 resFtrNum);
      return NULL;
    }

    resP = MemHandleLock(resH);
    if (resP != NULL)
    {
      size = MemHandleSize(resH);
      DmWrite(ftrP,
              offset,
              resP,
              size);
      offset += size;
      
      MemHandleUnlock(resH);
    }
    
    DmReleaseResource(resH);
  }
  while (offset < FinalSize);
  
  return ftrP;
}
/*----------------------------------------------------------------------------*/


tULong ComputeCRC32(tUChar* memP,
                    tULong size)
/***********************************************************************
 *
 *  ComputeCRC32
 *
 ***********************************************************************/
{
tULong crc32;
tCRC32Param* paramP;
MemHandle tableH;

#ifndef __RELEASE__
  if (memP == NULL)
    ErrDisplay("memP == NULL");
  if (size == 0)
    ErrDisplay("size == 0");
#endif /* __RELEASE__ */

  paramP = MemPtrNew(sizeof(tCRC32Param));

#ifndef __RELEASE__
  if (paramP == NULL)
    ErrDisplay("paramP == NULL");
#endif /* __RELEASE__ */

  if (paramP == NULL)
    return 0;

  // Prepare operation
  paramP->bufferP = memP;
  paramP->size = size;

  // CRC32 Table
  tableH = DmGetResource(TABLE_ID_TYPE,
                         TABLE_ID_CRC32);
  paramP->crc32tableP = (tULong*)MemHandleLock(tableH);

#ifndef __RELEASE__
  if (tableH == NULL)
    ErrDisplay("tableH == NULL");
  if (paramP->crc32tableP == NULL)
    ErrDisplay("param.crc32tableP == NULL");
#endif /* __RELEASE__ */

  // Native CRC32 routine call
  crc32 = PceNativeCallResource(Native_CRC32,
#ifdef __SIMU__
                                "Native_CRC32.dll\0PNOMain", 
#endif /* __SIMU__ */
                                paramP);

  MemHandleUnlock(tableH);
  DmReleaseResource(tableH);

  MemPtrFree(paramP);

  return crc32;
}
/*----------------------------------------------------------------------------*/


void ReindexPointersArea(tReindexOperationParameters* paramP)
/***********************************************************************
 *
 *  ReindexPointersArea
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (paramP == NULL)
    ErrDisplay("paramP == NULL");
#endif /* __RELEASE__ */

  while (paramP->areaSize)
  {
    // Get pointer address
    UInt32 ptr = paramP->native ? EndianSwap32(*paramP->srcAreaP) : 
                                  (UInt32)*paramP->srcAreaP;
    
    // If pointer is inside source context range
    if ( (ptr >= paramP->srcContextBase) &&
         (ptr < (paramP->srcContextBase+paramP->contextSize)) )
    {
      // Adapt pointer to destination context
      ptr -= paramP->srcContextBase;
      ptr += paramP->dstContextBase;
      
      // Update destination area
      *paramP->dstAreaP = paramP->native ? (void*)EndianSwap32(ptr) : (void*)ptr;
    }
      
    // Next pointer
    paramP->srcAreaP++;
    paramP->dstAreaP++;
    paramP->areaSize -= sizeof(tUChar*);
  }
}
/*----------------------------------------------------------------------------*/


#ifdef _TESTU

static Err TestU_ReindexPointersArea_1(tUChar NoTest) SECTION_ROUTINES;

Err Routines_PerformTestU(void)
/***********************************************************************
 *
 *  Routines_PerformTestU
 *
 ***********************************************************************/
{
Err Result = errNone;
tUChar NoTest = 1;

  /* 1 */ if (Result == errNone) Result = TestU_ReindexPointersArea_1(NoTest++);

  return (Result);
}
/*----------------------------------------------------------------------------*/

static Err TestU_ReindexPointersArea_1(tUChar NoTest)
/***********************************************************************
 *
 *  TestU_ReindexPointersArea_1
 *
 ***********************************************************************/
{
UInt32 area[5];
UInt32 SrcIndex;
UInt32 DstIndex;
Err Result = errNone;
tReindexOperationParameters param;

  //
  // Test 1
  //
  param.srcContextBase = 0x10000000;
  param.dstContextBase = 0x20000000;
  area[0]  = 0x011111FF;
  area[1]  = 0x111111EE;
  area[2]  = 0x155555DD;
  area[3]  = 0x1AAAAACC;
  area[4]  = 0x333333BB;
  param.srcAreaP = (void**)area;
  param.dstAreaP = (void**)area;
  param.areaSize = sizeof(area[0]) * 4;
  param.contextSize = 0x10000000;
  param.native = cFalse;
  ReindexPointersArea(&param);
  if ( (area[0] != 0x011111FF) ||
       (area[1] != 0x211111EE) ||
       (area[2] != 0x255555DD) ||
       (area[3] != 0x2AAAAACC) ||
       (area[4] != 0x333333BB) )
  {
    Result=appErrorClass+NoTest;
  }

  //
  // Test 2
  //
  param.srcContextBase = 0x10000000;
  param.dstContextBase = 0x20000000;
  area[0]  = 0xFF111101;
  area[1]  = 0xEE111111;
  area[2]  = 0xDD555515;
  area[3]  = 0xCCAAAA1A;
  area[4]  = 0xBB333333;
  param.srcAreaP = (void**)area;
  param.dstAreaP = (void**)area;
  param.areaSize = sizeof(area[0]) * 4;
  param.contextSize = 0x10000000;
  param.native = cTrue;
  ReindexPointersArea(&param);
  if ( (area[0] != 0xFF111101) ||
       (area[1] != 0xEE111121) ||
       (area[2] != 0xDD555525) ||
       (area[3] != 0xCCAAAA2A) ||
       (area[4] != 0xBB333333) )
  {
    Result=appErrorClass+NoTest;
  }
  
  return (Result);
}
/*----------------------------------------------------------------------------*/

#endif /* _TESTU */
