/*
    CaPriCe for Palm OS - Amstrad CPC 464/664/6128 emulator for Palm devices
    Copyright (C) 2009  Frdric Coste

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <PalmOS.h>

#include "CaPriCe.h"
#include "Prefs.h"
#include "Keyboard.h"
#include "Display.h"


#define appPrefID                 0x01

#define appPrefOldVersionNum_1    0x01
#define appPrefOldVersionNum_2    0x02
#define appPrefOldVersionNum_3    0x03
#define appPrefOldVersionNum_4    0x04
#define appPrefOldVersionNum_5    0x05
#define appPrefOldVersionNum_6    0x06
#define appPrefOldVersionNum_7    0x07
#define appPrefOldVersionNum_8    0x08
#define appPrefOldVersionNum_9    0x09
#define appPrefVersionNum         appPrefOldVersionNum_9


// Default values
// Version 1
#define DEFAULT_PREF_CPCMODEL             CPC_MODEL_464
#define DEFAULT_PREF_CPCJUMPERS           CPC_DEFAULT_JUMPERS
#define DEFAULT_PREF_SCREENTYPE           0 // Color
#define DEFAULT_PREF_CPCTRUESPEED         0 // Full speed
// Version 2
#define DEFAULT_PREF_SCREENINTENSITY      10
#define DEFAULT_PREF_SOUNDENABLED         0
#define DEFAULT_PREF_SOUNDVOLUME          40
#define DEFAULT_PREF_SOUNDSYSTEMVOLUME    1   // 0 = Own volume, 1 = System games volume (default)
// Version 3
#define DEFAULT_PREF_DISPLAY320X480       0   // 0 = 320x320 (default), 1 = 320x480
#define DEFAULT_PREF_AUTOCONTEXTSAVE      0   // 0 = Manual save (default), 1 = Automatic save
#define DEFAULT_PREF_CUSTOMCOLORUSE       0   // 0 = Do not use Custom colors (default), 1 = Use custom colors
#define DEFAULT_PREF_CUSTOMCOLORINDEX     0
#define DEFAULT_PREF_MODE2ANTIALIAS       0   // 0 = Mode 2 Aliased (default), 1 = Mode 2 Anti-aliased (not implemented)
// Version 4
#define DEFAULT_PREF_OS_DISPLAY           1   // 0 = OSD Inactive, 1 = OSD Ative (default)
#define DEFAULT_PREF_OS_ROCKER            0   // 0 = OSR Inactive (default), 1 = OSR Active
#define DEFAULT_PREF_CENTERKEY_INDEX      0
#define DEFAULT_PREF_HARDKEY_FIRE         0
#define DEFAULT_PREF_CHANGED              1   // Preferences is changed by default
// Version 5
#define DEFAULT_PREF_AUTOOFF_DISABLE      1   // 0 = AutoOff Enable, 1 = AutoOff Disable (default)
#define DEFAULT_PREF_DISPLAY_OFFSET_X     OFFSCREEN_OFFSET_X
#define DEFAULT_PREF_DISPLAY_OFFSET_Y     OFFSCREEN_OFFSET_Y
#define DEFAULT_PREF_HARDKEY_1_INDEX      HARDKEY_NOT_USED_INDEX
#define DEFAULT_PREF_HARDKEY_2_INDEX      HARDKEY_NOT_USED_INDEX
#define DEFAULT_PREF_HARDKEY_3_INDEX      HARDKEY_NOT_USED_INDEX
#define DEFAULT_PREF_HARDKEY_4_INDEX      HARDKEY_NOT_USED_INDEX
// Version 6
#define DEFAULT_PREF_USE_PARADOS          0
#define DEFAULT_PREF_USE_64K_MEM_EXT      0
#define DEFAULT_PREF_NIGHTMODE_ACTIVE     0
// Version 7
#define DEFAULT_PREF_CPC_KEYCODE_A        76  // Key code for Joystick 0 Fire 2
#define DEFAULT_PREF_RENDERING            0   // Fastest
// Version 8
#define DEFAULT_PREF_CPC_KEYCODE_B        21  // Key code for Shift
#define DEFAULT_PREF_CPC_KEYCODE_C        23  // Key code for Control
#define DEFAULT_PREF_CPC_KEYCODE_D        47  // Key code for Space Bar
#define DEFAULT_AUTOSTART_ENABLE          0   // Disable
// Version 9
#define DEFAULT_PREF_OS_LIGHTPEN          0
#define DEFAULT_PREF_OS_MAGNUMGUN         0
#define DEFAULT_PREF_OS_GUNSTICK          0
#define DEFAULT_PREF_OS_WESTPHASER        0



Err LirePreferences(tPreferences** prefPP)
/***********************************************************************
 *
 *  LirePreferences
 *
 ***********************************************************************/
#undef LIREPREFERENCES_NO_PREF_FOUND
//#define LIREPREFERENCES_NO_PREF_FOUND
#undef LIREPREFERENCES_BAD_PREF_SIZE
//#define LIREPREFERENCES_BAD_PREF_SIZE
#undef LIREPREFERENCES_PREF_IS_VERSION_8
//#define LIREPREFERENCES_PREF_IS_VERSION_8
#undef LIREPREFERENCES_PREF_IS_VERSION_7
//#define LIREPREFERENCES_PREF_IS_VERSION_7
#undef LIREPREFERENCES_PREF_IS_VERSION_6
//#define LIREPREFERENCES_PREF_IS_VERSION_6
#undef LIREPREFERENCES_PREF_IS_VERSION_5
//#define LIREPREFERENCES_PREF_IS_VERSION_5
#undef LIREPREFERENCES_PREF_IS_VERSION_4
//#define LIREPREFERENCES_PREF_IS_VERSION_4
#undef LIREPREFERENCES_PREF_IS_VERSION_3
//#define LIREPREFERENCES_PREF_IS_VERSION_3
#undef LIREPREFERENCES_PREF_IS_VERSION_2
//#define LIREPREFERENCES_PREF_IS_VERSION_2
#undef LIREPREFERENCES_PREF_IS_VERSION_1
//#define LIREPREFERENCES_PREF_IS_VERSION_1
{
tPreferences* prefP;
UInt16 prefsize;
Int16 prefversion;
UInt8 State;

  prefP = (tPreferences*)MemPtrNew(sizeof(tPreferences));
  if (prefP == NULL)
    return memErrNotEnoughSpace;
    
  *prefPP = prefP;
  
  prefsize = MemPtrSize(prefP);
  prefversion = PrefGetAppPreferences(appFileCreator,
                                      appPrefID,
                                      prefP,
                                      &prefsize,
                                      true);

#ifdef LIREPREFERENCES_NO_PREF_FOUND
  prefversion = noPreferenceFound;
#endif /* LIREPREFERENCES_NO_PREF_FOUND */
#ifdef LIREPREFERENCES_BAD_PREF_SIZE
  prefsize = 0;
#endif /* LIREPREFERENCES_BAD_PREF_SIZE */
#ifdef LIREPREFERENCES_PREF_IS_VERSION_8
  prefversion = appPrefOldVersionNum_8;
#endif /* LIREPREFERENCES_PREF_IS_VERSION_8 */
#ifdef LIREPREFERENCES_PREF_IS_VERSION_7
  prefversion = appPrefOldVersionNum_7;
#endif /* LIREPREFERENCES_PREF_IS_VERSION_7 */
#ifdef LIREPREFERENCES_PREF_IS_VERSION_6
  prefversion = appPrefOldVersionNum_6;
#endif /* LIREPREFERENCES_PREF_IS_VERSION_6 */
#ifdef LIREPREFERENCES_PREF_IS_VERSION_5
  prefversion = appPrefOldVersionNum_5;
#endif /* LIREPREFERENCES_PREF_IS_VERSION_5 */
#ifdef LIREPREFERENCES_PREF_IS_VERSION_4
  prefversion = appPrefOldVersionNum_4;
#endif /* LIREPREFERENCES_PREF_IS_VERSION_4 */
#ifdef LIREPREFERENCES_PREF_IS_VERSION_3
  prefversion = appPrefOldVersionNum_3;
#endif /* LIREPREFERENCES_PREF_IS_VERSION_3 */
#ifdef LIREPREFERENCES_PREF_IS_VERSION_2
  prefversion = appPrefOldVersionNum_2;
#endif /* LIREPREFERENCES_PREF_IS_VERSION_2 */
#ifdef LIREPREFERENCES_PREF_IS_VERSION_1
  prefversion = appPrefOldVersionNum_1;
#endif /* LIREPREFERENCES_PREF_IS_VERSION_1 */

  // Preferences is version 8
  if (prefversion == appPrefOldVersionNum_8)
  {
    // Set default values from Version 9
    State = 9;
  }
  // Preferences is version 7
  else if (prefversion == appPrefOldVersionNum_7)
  {
    // Set default values from Version 8
    State = 8;
  }
  // Preferences is version 6
  else if (prefversion == appPrefOldVersionNum_6)
  {
    // Set default values from Version 7
    State = 7;
  }
  // Preferences is version 5
  else if (prefversion == appPrefOldVersionNum_5)
  {
    // Set default values from Version 6
    State = 6;
  }
  // Preferences is version 4
  else if (prefversion == appPrefOldVersionNum_4)
  {
    // Set default values from Version 5
    State = 5;
  }
  // Preferences is version 3
  else if (prefversion == appPrefOldVersionNum_3)
  {
    // Set default values from Version 4
    State = 4;
  }
  // Preferences is version 2
  else if (prefversion == appPrefOldVersionNum_2)
  {
    // Set default values from Version 3
    State = 3;
  }
  // Preferences is version 1
  else if (prefversion == appPrefOldVersionNum_1)
  {
    // Set default values from Version 2
    State = 2;
  }
  else if ( (prefversion == noPreferenceFound) || (prefsize != MemPtrSize(prefP)) )
  {
    // Set default values from Version 1
    State = 1;
  }
  else
  {
    // No default values to set
    prefP->PreferencesChanged = 0;
    State = 0;
  }
  
  switch (State)
  {
  	case 1:
      // Default values Version 1
      prefP->CPCModel = DEFAULT_PREF_CPCMODEL;
      prefP->CPCJumpers = DEFAULT_PREF_CPCJUMPERS;
      prefP->ScreenType = DEFAULT_PREF_SCREENTYPE;
      prefP->CPCTrueSpeed = DEFAULT_PREF_CPCTRUESPEED;
      StrCopy(prefP->FilesPathname, DEFAULT_DISK_PATH);
     
    case 2:
      // Default values Version 2
      prefP->ScreenIntensity = DEFAULT_PREF_SCREENINTENSITY;
      prefP->SoundEnabled = DEFAULT_PREF_SOUNDENABLED;
      prefP->SoundVolume = DEFAULT_PREF_SOUNDVOLUME;
      prefP->SoundSystemVolume = DEFAULT_PREF_SOUNDSYSTEMVOLUME;

    case 3:
      // Default values Version 3
      prefP->Display320x480 = DEFAULT_PREF_DISPLAY320X480;
      prefP->AutomaticContextSave = DEFAULT_PREF_AUTOCONTEXTSAVE;
      prefP->CustomColorUse = DEFAULT_PREF_CUSTOMCOLORUSE;
      MemSet(prefP->CustomColorIndexes, 32, DEFAULT_PREF_CUSTOMCOLORINDEX);
      prefP->Mode2AntiAliasing = DEFAULT_PREF_MODE2ANTIALIAS;

    case 4:
      // Default values Version 4
      prefP->OnScreenDisplayActive = DEFAULT_PREF_OS_DISPLAY;
      prefP->OnScreenRockerActive = DEFAULT_PREF_OS_ROCKER;
      prefP->RockerCenterKeyIndex = DEFAULT_PREF_CENTERKEY_INDEX;
      prefP->JoystickFireHardKey = DEFAULT_PREF_HARDKEY_FIRE;
      prefP->PreferencesChanged = DEFAULT_PREF_CHANGED;

    case 5:
      // Default values Version 5 (v2.0)
      prefP->AutoOffDisable = DEFAULT_PREF_AUTOOFF_DISABLE;
      prefP->ScreenDisplayOffsetX = DEFAULT_PREF_DISPLAY_OFFSET_X;
      prefP->ScreenDisplayOffsetY = DEFAULT_PREF_DISPLAY_OFFSET_Y;
      prefP->HardKey1Index = DEFAULT_PREF_HARDKEY_1_INDEX;
      prefP->HardKey2Index = DEFAULT_PREF_HARDKEY_2_INDEX;
      prefP->HardKey3Index = DEFAULT_PREF_HARDKEY_3_INDEX;
      prefP->HardKey4Index = DEFAULT_PREF_HARDKEY_4_INDEX;

    case 6:
      // Default values Version 6 (v2.1)
      prefP->UseParados = DEFAULT_PREF_USE_PARADOS;
      prefP->Use64kMemoryExtension = DEFAULT_PREF_USE_64K_MEM_EXT;
      prefP->NightModeActive = DEFAULT_PREF_NIGHTMODE_ACTIVE;
      
    case 7:
      // Default values Version 7 (v2.2)
      prefP->CPCKeycodeA = DEFAULT_PREF_CPC_KEYCODE_A;
      prefP->Rendering = DEFAULT_PREF_RENDERING;

    case 8:
      // Default values Version 8 (v2.3)
      prefP->CPCKeycodeB = DEFAULT_PREF_CPC_KEYCODE_B;
      prefP->CPCKeycodeC = DEFAULT_PREF_CPC_KEYCODE_C;
      prefP->CPCKeycodeD = DEFAULT_PREF_CPC_KEYCODE_D;
      prefP->AutoStartEnable = DEFAULT_AUTOSTART_ENABLE;

    case 9:
      // Default values Version 9 (v2.4)
      prefP->OnScreenLightPenActive = DEFAULT_PREF_OS_LIGHTPEN;
      prefP->OnScreenMagnumGunActive = DEFAULT_PREF_OS_MAGNUMGUN;
      prefP->OnScreenGunstickActive = DEFAULT_PREF_OS_GUNSTICK;
      prefP->OnScreenWestPhaserActive = DEFAULT_PREF_OS_WESTPHASER;

    default:
      prefP->NightModeActive = DEFAULT_PREF_NIGHTMODE_ACTIVE; // Night mode always inactive at start
  }
  
#ifdef _ROCKER_ACTIVE
  prefP->OnScreenRockerActive = 1;
  prefP->OnScreenLightPenActive = 0;
  prefP->OnScreenMagnumGunActive = 0;
  prefP->OnScreenGunstickActive = 0;
  prefP->OnScreenWestPhaserActive = 0;
#endif /* _ROCKER_ACTIVE */
#ifdef _LIGHTPEN_ACTIVE
  prefP->OnScreenRockerActive = 0;
  prefP->OnScreenLightPenActive = 1;
  prefP->OnScreenMagnumGunActive = 0;
  prefP->OnScreenGunstickActive = 0;
  prefP->OnScreenWestPhaserActive = 0;
#endif /* _LIGHTPEN_ACTIVE */
#ifdef _MAGNUMGUN_ACTIVE
  prefP->OnScreenRockerActive = 0;
  prefP->OnScreenLightPenActive = 0;
  prefP->OnScreenMagnumGunActive = 1;
  prefP->OnScreenGunstickActive = 0;
  prefP->OnScreenWestPhaserActive = 0;
#endif /* _MAGNUMGUN_ACTIVE */
#ifdef _GUNSTICK_ACTIVE
  prefP->OnScreenRockerActive = 0;
  prefP->OnScreenLightPenActive = 0;
  prefP->OnScreenMagnumGunActive = 0;
  prefP->OnScreenGunstickActive = 1;
  prefP->OnScreenWestPhaserActive = 0;
#endif /* _GUNSTICK_ACTIVE */
#ifdef _WESTPHASER_ACTIVE
  prefP->OnScreenRockerActive = 0;
  prefP->OnScreenLightPenActive = 0;
  prefP->OnScreenMagnumGunActive = 0;
  prefP->OnScreenGunstickActive = 0;
  prefP->OnScreenWestPhaserActive = 1;
#endif /* _WESTPHASER_ACTIVE */
  
  return errNone;
}
/*----------------------------------------------------------------------------*/


void EcrirePreferences(tPreferences* prefP)
/***********************************************************************
 *
 *  EcrirePreferences
 *
 ***********************************************************************/
{
  PrefSetAppPreferences(appFileCreator,
                        appPrefID,
                        appPrefVersionNum,
                        prefP,
                        sizeof(tPreferences),
                        true);
}
/*----------------------------------------------------------------------------*/


void SetHardKeysMask(const tPreferences* prefP,
                     UInt32 originKeyMask)
/***********************************************************************
 *
 *  SetHardKeysMask
 *
 ***********************************************************************/
{
  // Do not generate events for rocker
  emulatorHardKeyMask = originKeyMask & ~(keyBitRockerUp | keyBitRockerDown | keyBitRockerLeft | keyBitRockerRight | keyBitRockerCenter);
  
  // Do not generate Hard Key events for simulating Joystick Fire.
  if (prefP->HardKey1Index == HARDKEY_ROCKER_CENTER_INDEX)
  {
    emulatorHardKeyMask &= ~keyBitHardKey1;
  }
  if (prefP->HardKey2Index == HARDKEY_ROCKER_CENTER_INDEX)
  {
    emulatorHardKeyMask &= ~keyBitHardKey2;
  }
  if (prefP->HardKey3Index == HARDKEY_ROCKER_CENTER_INDEX)
  {
    emulatorHardKeyMask &= ~keyBitHardKey3;
  }
  if (prefP->HardKey4Index == HARDKEY_ROCKER_CENTER_INDEX)
  {
    emulatorHardKeyMask &= ~keyBitHardKey4;
  }

	UpdateRockerCenterKeyMask();
	UpdateHardCPCKeyCodeMask();

  // Set new key mask.
  KeySetMask(emulatorHardKeyMask);
}
/*----------------------------------------------------------------------------*/
