/*
    CaPriCe for Palm OS - Amstrad CPC 464/664/6128 emulator for Palm devices
    Copyright (C) 2009  Frdric Coste

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <PalmOS.h>
#include <ByteOrderUtils.h>

#include "CaPriCe.h"
#include "Keyboard.h"
#include "Display.h"
#include "Resources.h"
#include "Files.h"
#include "Forms.h"
#include "CPC.h"
#include "Sections.h"
#include "Sound.h"
#include "Trace.h"
#include "Prefs.h"


//===================
// PATCH begin
#ifdef _PATCH_ENABLE

#endif /* _PATCH_ENABLE */
// PATCH end
//===================


// Copied from Core/System/CharLatin.h
#define chrPoundSign            0x00A3


#define AUTOTOGGLE_DURATION_INDEX_DEFAULT     2


#define MOD_CPC_SHIFT   (0x01 << 8)
#define MOD_CPC_CTRL    (0x02 << 8)
#define MOD_EMU_KEY     (0x10 << 8)

#define KEY_INVALID               0xff
#define PRESSED_KEY_INVALID       0xffff


static const tUShort cpc_kbd[149] =
{
  // original CPC keyboard
  0x40,                   // CPC_KEY_0
  0x80,                   // CPC_KEY_1
  0x81,                   // CPC_KEY_2
  0x71,                   // CPC_KEY_3
  0x70,                   // CPC_KEY_4
  0x61,                   // CPC_KEY_5
  0x60,                   // CPC_KEY_6
  0x51,                   // CPC_KEY_7
  0x50,                   // CPC_KEY_8
  0x41,                   // CPC_KEY_9
  0x85 | MOD_CPC_SHIFT,   // CPC_KEY_A
  0x66 | MOD_CPC_SHIFT,   // CPC_KEY_B
  0x76 | MOD_CPC_SHIFT,   // CPC_KEY_C
  0x75 | MOD_CPC_SHIFT,   // CPC_KEY_D
  0x72 | MOD_CPC_SHIFT,   // CPC_KEY_E
  0x65 | MOD_CPC_SHIFT,   // CPC_KEY_F
  0x64 | MOD_CPC_SHIFT,   // CPC_KEY_G
  0x54 | MOD_CPC_SHIFT,   // CPC_KEY_H
  0x43 | MOD_CPC_SHIFT,   // CPC_KEY_I
  0x55 | MOD_CPC_SHIFT,   // CPC_KEY_J
  0x45 | MOD_CPC_SHIFT,   // CPC_KEY_K
  0x44 | MOD_CPC_SHIFT,   // CPC_KEY_L
  0x46 | MOD_CPC_SHIFT,   // CPC_KEY_M
  0x56 | MOD_CPC_SHIFT,   // CPC_KEY_N
  0x42 | MOD_CPC_SHIFT,   // CPC_KEY_O
  0x33 | MOD_CPC_SHIFT,   // CPC_KEY_P
  0x83 | MOD_CPC_SHIFT,   // CPC_KEY_Q
  0x62 | MOD_CPC_SHIFT,   // CPC_KEY_R
  0x74 | MOD_CPC_SHIFT,   // CPC_KEY_S
  0x63 | MOD_CPC_SHIFT,   // CPC_KEY_T
  0x52 | MOD_CPC_SHIFT,   // CPC_KEY_U
  0x67 | MOD_CPC_SHIFT,   // CPC_KEY_V
  0x73 | MOD_CPC_SHIFT,   // CPC_KEY_W
  0x77 | MOD_CPC_SHIFT,   // CPC_KEY_X
  0x53 | MOD_CPC_SHIFT,   // CPC_KEY_Y
  0x87 | MOD_CPC_SHIFT,   // CPC_KEY_Z
  0x85,                   // CPC_KEY_a
  0x66,                   // CPC_KEY_b
  0x76,                   // CPC_KEY_c
  0x75,                   // CPC_KEY_d
  0x72,                   // CPC_KEY_e
  0x65,                   // CPC_KEY_f
  0x64,                   // CPC_KEY_g
  0x54,                   // CPC_KEY_h
  0x43,                   // CPC_KEY_i
  0x55,                   // CPC_KEY_j
  0x45,                   // CPC_KEY_k
  0x44,                   // CPC_KEY_l
  0x46,                   // CPC_KEY_m
  0x56,                   // CPC_KEY_n
  0x42,                   // CPC_KEY_o
  0x33,                   // CPC_KEY_p
  0x83,                   // CPC_KEY_q
  0x62,                   // CPC_KEY_r
  0x74,                   // CPC_KEY_s
  0x63,                   // CPC_KEY_t
  0x52,                   // CPC_KEY_u
  0x67,                   // CPC_KEY_v
  0x73,                   // CPC_KEY_w
  0x77,                   // CPC_KEY_x
  0x53,                   // CPC_KEY_y
  0x87,                   // CPC_KEY_z
  0x60 | MOD_CPC_SHIFT,   // CPC_KEY_AMPERSAND
  0x35 | MOD_CPC_SHIFT,   // CPC_KEY_ASTERISK
  0x32,                   // CPC_KEY_AT
  0x26 | MOD_CPC_SHIFT,   // CPC_KEY_BACKQUOTE
  0x26,                   // CPC_KEY_BACKSLASH
  0x86,                   // CPC_KEY_CAPSLOCK
  0x20,                   // CPC_KEY_CLR
  0x35,                   // CPC_KEY_COLON
  0x47,                   // CPC_KEY_COMMA
  0x27,                   // CPC_KEY_CONTROL
  0x11,                   // CPC_KEY_COPY
  0x02 | MOD_CPC_SHIFT,   // CPC_KEY_CPY_DOWN
  0x10 | MOD_CPC_SHIFT,   // CPC_KEY_CPY_LEFT
  0x01 | MOD_CPC_SHIFT,   // CPC_KEY_CPY_RIGHT
  0x00 | MOD_CPC_SHIFT,   // CPC_KEY_CPY_UP
  0x02,                   // CPC_KEY_CUR_DOWN
  0x10,                   // CPC_KEY_CUR_LEFT
  0x01,                   // CPC_KEY_CUR_RIGHT
  0x00,                   // CPC_KEY_CUR_UP
  0x02 | MOD_CPC_CTRL,    // CPC_KEY_CUR_ENDBL
  0x10 | MOD_CPC_CTRL,    // CPC_KEY_CUR_HOMELN
  0x01 | MOD_CPC_CTRL,    // CPC_KEY_CUR_ENDLN
  0x00 | MOD_CPC_CTRL,    // CPC_KEY_CUR_HOMEBL
  0x81 | MOD_CPC_SHIFT,   // CPC_KEY_DBLQUOTE
  0x97,                   // CPC_KEY_DEL
  0x70 | MOD_CPC_SHIFT,   // CPC_KEY_DOLLAR
  0x06,                   // CPC_KEY_ENTER
  0x31 | MOD_CPC_SHIFT,   // CPC_KEY_EQUAL
  0x82,                   // CPC_KEY_ESC
  0x80 | MOD_CPC_SHIFT,   // CPC_KEY_EXCLAMATN
  0x17,                   // CPC_KEY_F0
  0x15,                   // CPC_KEY_F1
  0x16,                   // CPC_KEY_F2
  0x05,                   // CPC_KEY_F3
  0x24,                   // CPC_KEY_F4
  0x14,                   // CPC_KEY_F5
  0x04,                   // CPC_KEY_F6
  0x12,                   // CPC_KEY_F7
  0x13,                   // CPC_KEY_F8
  0x03,                   // CPC_KEY_F9
  0x07,                   // CPC_KEY_FPERIOD
  0x37 | MOD_CPC_SHIFT,   // CPC_KEY_GREATER
  0x71 | MOD_CPC_SHIFT,   // CPC_KEY_HASH
  0x21,                   // CPC_KEY_LBRACKET
  0x21 | MOD_CPC_SHIFT,   // CPC_KEY_LCBRACE
  0x50 | MOD_CPC_SHIFT,   // CPC_KEY_LEFTPAREN
  0x47 | MOD_CPC_SHIFT,   // CPC_KEY_LESS
  0x25,                   // CPC_KEY_LSHIFT
  0x31,                   // CPC_KEY_MINUS
  0x61 | MOD_CPC_SHIFT,   // CPC_KEY_PERCENT
  0x37,                   // CPC_KEY_PERIOD
  0x32 | MOD_CPC_SHIFT,   // CPC_KEY_PIPE
  0x34 | MOD_CPC_SHIFT,   // CPC_KEY_PLUS
  0x30 | MOD_CPC_SHIFT,   // CPC_KEY_POUND
  0x30,                   // CPC_KEY_POWER
  0x36 | MOD_CPC_SHIFT,   // CPC_KEY_QUESTION
  0x51 | MOD_CPC_SHIFT,   // CPC_KEY_QUOTE
  0x23,                   // CPC_KEY_RBRACKET
  0x23 | MOD_CPC_SHIFT,   // CPC_KEY_RCBRACE
  0x22,                   // CPC_KEY_RETURN
  0x41 | MOD_CPC_SHIFT,   // CPC_KEY_RIGHTPAREN
  0x25,                   // CPC_KEY_RSHIFT
  0x34,                   // CPC_KEY_SEMICOLON
  0x36,                   // CPC_KEY_SLASH
  0x57,                   // CPC_KEY_SPACE
  0x84,                   // CPC_KEY_TAB
  0x40 | MOD_CPC_SHIFT,   // CPC_KEY_UNDERSCORE
  0x90,                   // CPC_KEY_J0_UP
  0x91,                   // CPC_KEY_J0_DOWN
  0x92,                   // CPC_KEY_J0_LEFT
  0x93,                   // CPC_KEY_J0_RIGHT
  0x94,                   // CPC_KEY_J0_FIRE1
  0x95,                   // CPC_KEY_J0_FIRE2
  0x60,                   // CPC_KEY_J1_UP
  0x61,                   // CPC_KEY_J1_DOWN
  0x62,                   // CPC_KEY_J1_LEFT
  0x63,                   // CPC_KEY_J1_RIGHT
  0x64,                   // CPC_KEY_J1_FIRE1
  0x65,                   // CPC_KEY_J1_FIRE2
  KEY_INVALID,            // CPC_KEY_ES_NTILDE
  KEY_INVALID,            // CPC_KEY_ES_nTILDE
  KEY_INVALID,            // CPC_KEY_ES_PESETA
  KEY_INVALID,            // CPC_KEY_FR_eACUTE
  KEY_INVALID,            // CPC_KEY_FR_eGRAVE
  KEY_INVALID,            // CPC_KEY_FR_cCEDIL
  KEY_INVALID,            // CPC_KEY_FR_aGRAVE
  KEY_INVALID,            // CPC_KEY_FR_uGRAVE
};

//
// Copied from Core/System/Chars.h
//
static const tUChar palm_kbd[128] =
{
  KEY_INVALID,            // 0x0000 chrNull
  KEY_INVALID,            // 0x0001 chrStartOfHeading
  KEY_INVALID,            // 0x0002 chrStartOfText
  KEY_INVALID,            // 0x0003 chrEndOfText
  KEY_INVALID,            // 0x0004 chrEndOfTransmission
  KEY_INVALID,            // 0x0005 chrEnquiry
  KEY_INVALID,            // 0x0006 chrAcknowledge
  KEY_INVALID,            // 0x0007 chrBell
  CPC_KEY_DEL,            // 0x0008 chrBackspace
  CPC_KEY_TAB,            // 0x0009 chrHorizontalTabulation
  CPC_KEY_RETURN,         // 0x000A chrLineFeed
  KEY_INVALID,            // 0x000B chrVerticalTabulation
  KEY_INVALID,            // 0x000C chrFormFeed
  CPC_KEY_RETURN,         // 0x000D chrCarriageReturn
  KEY_INVALID,            // 0x000E chrShiftOut
  KEY_INVALID,            // 0x000F chrShiftIn
  KEY_INVALID,            // 0x0010 chrDataLinkEscape
  KEY_INVALID,            // 0x0011 chrDeviceControlOne
  KEY_INVALID,            // 0x0012 chrDeviceControlTwo
  KEY_INVALID,            // 0x0013 chrDeviceControlThree
  KEY_INVALID,            // 0x0014 chrDeviceControlFour
  KEY_INVALID,            // 0x0015 chrNegativeAcknowledge
  KEY_INVALID,            // 0x0016 chrSynchronousIdle
  KEY_INVALID,            // 0x0017 chrEndOfTransmissionBlock
  KEY_INVALID,            // 0x0018 chrCancel
  KEY_INVALID,            // 0x0019 chrEndOfMedium
  KEY_INVALID,            // 0x001A chrSubstitute
  CPC_KEY_ESC,            // 0x001B chrEscape
  KEY_INVALID,            // 0x001C chrFileSeparator
  KEY_INVALID,            // 0x001D chrGroupSeparator
  KEY_INVALID,            // 0x001E chrRecordSeparator
  KEY_INVALID,            // 0x001F chrUnitSeparator
  CPC_KEY_SPACE,          // 0x0020 chrSpace
  CPC_KEY_EXCLAMATN,      // 0x0021 chrExclamationMark
  CPC_KEY_DBLQUOTE,       // 0x0022 chrQuotationMark
  CPC_KEY_HASH,           // 0x0023 chrNumberSign
  CPC_KEY_DOLLAR,         // 0x0024 chrDollarSign
  CPC_KEY_PERCENT,        // 0x0025 chrPercentSign
  CPC_KEY_AMPERSAND,      // 0x0026 chrAmpersand
  CPC_KEY_QUOTE,          // 0x0027 chrApostrophe
  CPC_KEY_LEFTPAREN,      // 0x0028 chrLeftParenthesis
  CPC_KEY_RIGHTPAREN,     // 0x0029 chrRightParenthesis
  CPC_KEY_ASTERISK,       // 0x002A chrAsterisk
  CPC_KEY_PLUS,           // 0x002B chrPlusSign
  CPC_KEY_COMMA,          // 0x002C chrComma/
  CPC_KEY_MINUS,          // 0x002D chrHyphenMinus
  CPC_KEY_PERIOD,         // 0x002E chrFullStop/Period
  CPC_KEY_SLASH,          // 0x002F chrSolidus/Slash
  CPC_KEY_0,              // 0x0030 chrDigitZero
  CPC_KEY_1,              // 0x0031 chrDigitOne
  CPC_KEY_2,              // 0x0032 chrDigitTwo
  CPC_KEY_3,              // 0x0033 chrDigitThree
  CPC_KEY_4,              // 0x0034 chrDigitFour
  CPC_KEY_5,              // 0x0035 chrDigitFive
  CPC_KEY_6,              // 0x0036 chrDigitSix
  CPC_KEY_7,              // 0x0037 chrDigitSeven
  CPC_KEY_8,              // 0x0038 chrDigitEight
  CPC_KEY_9,              // 0x0039 chrDigitNine
  CPC_KEY_COLON,          // 0x003A chrColon
  CPC_KEY_SEMICOLON,      // 0x003B chrSemicolon
  CPC_KEY_LESS,           // 0x003C chrLessThanSign
  CPC_KEY_EQUAL,          // 0x003D chrEqualsSign
  CPC_KEY_GREATER,        // 0x003E chrGreaterThanSign
  CPC_KEY_QUESTION,       // 0x003F chrQuestionMark
  CPC_KEY_AT,             // 0x0040 chrCommercialAt
  CPC_KEY_A,              // 0x0041 chrCapital_A
  CPC_KEY_B,              // 0x0042 chrCapital_B
  CPC_KEY_C,              // 0x0043 chrCapital_C
  CPC_KEY_D,              // 0x0044 chrCapital_D
  CPC_KEY_E,              // 0x0045 chrCapital_E
  CPC_KEY_F,              // 0x0046 chrCapital_F
  CPC_KEY_G,              // 0x0047 chrCapital_G
  CPC_KEY_H,              // 0x0048 chrCapital_H
  CPC_KEY_I,              // 0x0049 chrCapital_I
  CPC_KEY_J,              // 0x004A chrCapital_J
  CPC_KEY_K,              // 0x004B chrCapital_K
  CPC_KEY_L,              // 0x004C chrCapital_L
  CPC_KEY_M,              // 0x004D chrCapital_M
  CPC_KEY_N,              // 0x004E chrCapital_N
  CPC_KEY_O,              // 0x004F chrCapital_O
  CPC_KEY_P,              // 0x0050 chrCapital_P
  CPC_KEY_Q,              // 0x0051 chrCapital_Q
  CPC_KEY_R,              // 0x0052 chrCapital_R
  CPC_KEY_S,              // 0x0053 chrCapital_S
  CPC_KEY_T,              // 0x0054 chrCapital_T
  CPC_KEY_U,              // 0x0055 chrCapital_U
  CPC_KEY_V,              // 0x0056 chrCapital_V
  CPC_KEY_W,              // 0x0057 chrCapital_W
  CPC_KEY_X,              // 0x0058 chrCapital_X
  CPC_KEY_Y,              // 0x0059 chrCapital_Y
  CPC_KEY_Z,              // 0x005A chrCapital_Z
  CPC_KEY_LBRACKET,       // 0x005B chrLeftSquareBracket
  CPC_KEY_BACKSLASH,      // 0x005C chrReverseSolidus
  CPC_KEY_RBRACKET,       // 0x005D chrRightSquareBracket
  KEY_INVALID,            // 0x005E chrCircumflexAccent
  CPC_KEY_UNDERSCORE,     // 0x005F chrLowLine
  KEY_INVALID,            // 0x0060 chrGraveAccent
  CPC_KEY_a,              // 0x0061 chrSmall_A
  CPC_KEY_b,              // 0x0062 chrSmall_B
  CPC_KEY_c,              // 0x0063 chrSmall_C
  CPC_KEY_d,              // 0x0064 chrSmall_D
  CPC_KEY_e,              // 0x0065 chrSmall_E
  CPC_KEY_f,              // 0x0066 chrSmall_F
  CPC_KEY_g,              // 0x0067 chrSmall_G
  CPC_KEY_h,              // 0x0068 chrSmall_H
  CPC_KEY_i,              // 0x0069 chrSmall_I
  CPC_KEY_j,              // 0x006A chrSmall_J
  CPC_KEY_k,              // 0x006B chrSmall_K
  CPC_KEY_l,              // 0x006C chrSmall_L
  CPC_KEY_m,              // 0x006D chrSmall_M
  CPC_KEY_n,              // 0x006E chrSmall_N
  CPC_KEY_o,              // 0x006F chrSmall_O
  CPC_KEY_p,              // 0x0070 chrSmall_P
  CPC_KEY_q,              // 0x0071 chrSmall_Q
  CPC_KEY_r,              // 0x0072 chrSmall_R
  CPC_KEY_s,              // 0x0073 chrSmall_S
  CPC_KEY_t,              // 0x0074 chrSmall_T
  CPC_KEY_u,              // 0x0075 chrSmall_U
  CPC_KEY_v,              // 0x0076 chrSmall_V
  CPC_KEY_w,              // 0x0077 chrSmall_W
  CPC_KEY_x,              // 0x0078 chrSmall_X
  CPC_KEY_y,              // 0x0079 chrSmall_Y
  CPC_KEY_z,              // 0x007A chrSmall_Z
  CPC_KEY_LCBRACE,        // 0x007B chrLeftCurlyBracket
  CPC_KEY_PIPE,           // 0x007C chrVerticalLine
  CPC_KEY_RCBRACE,        // 0x007D chrRightCurlyBracket
  KEY_INVALID,            // 0x007E chrTilde
  CPC_KEY_CLR,            // 0x007F chrDelete
};


static Boolean DetectPressedKey(tKey* keyP,
                                tUShort ScreenX,
                                tUShort ScreenY,
                                tUShort* keySaveP);
static Boolean ReleaseLastPressedKey(void);
static Boolean PenDownHandleKeyboardEvent(tKeyboard* keyboardP,
                                          tUShort ScreenX,
                                          tUShort ScreenY,
                                          tUShort y_base);

static void EnableTrueSpeed(void) SECTION_KEYBOARD;
static void DisableTrueSpeed(void) SECTION_KEYBOARD;
static void EnableDisplayEmuSpeed(void) SECTION_KEYBOARD;
static void DisableDisplayEmuSpeed(void) SECTION_KEYBOARD;
static void EnableFonctionKeys(void) SECTION_KEYBOARD;
static void DisableFonctionKeys(void) SECTION_KEYBOARD;
static void SHIFTKeyPressed(void) SECTION_KEYBOARD;
static void CTRLKeyPressed(void)SECTION_KEYBOARD;
static void SubPanelActivePressed(void) SECTION_KEYBOARD;
static void SubPanelActiveReleased(void) SECTION_KEYBOARD;
static UInt32 DetectScreenDirectionalInput(Coord,
                                           Coord,
                                           UInt8,
                                           UInt8,
                                           UInt8) SECTION_KEYBOARD;

static void DisplayCPCColouredKeyboard(void) SECTION_KEYBOARD;
static void DisplayCPCSpecialKeyboard(void) SECTION_KEYBOARD;

static void UpdatePanelPosition(Coord y_base,
                                UInt8 nbKeys,
                                tKey* keyP) SECTION_KEYBOARD;

static void InitKeysPanel(tKey* keyP,
                          UInt8 nbKeys,
                          const tKeySetting* settingsP,
                          tUChar DefaultHighlight) SECTION_KEYBOARD;

static Err StartKeyboard(tKeyboard* keyboardP);
static void SetLightPenCoordinates(Coord,
                                   Coord,
                                   Boolean) SECTION_KEYBOARD;
static void SetPhaserCoordinates(Coord,
                                 Coord,
                                 Boolean) SECTION_KEYBOARD;


static tUShort usCurrentPressedKey = PRESSED_KEY_INVALID;
static tUShort usSaveShiftCtrl = 0;
static UInt32 debounceTimeoutTicks = 0;
static tUShort usKeyState = 0;
static UInt32 oldScreenKey = 0;
static tKey* lastPressedKeyP = NULL;
static tUShort* lastPressedKeyStateP = NULL;


//
// Auto toggle
//
UInt8 AutoToggleActive = 0;
UInt8 AutoToggleDurationInTicks = 0;
UInt8 AutoToggleDurationIndex = 0;
static UInt32 u32AutoToggleKeyState = keyBitRockerLeft;
static UInt32 u32OldRockerStateWhileAutoToggle = keyBitRockerLeft;

static const UInt8 AutoToggleDurationA[] =
{ 0  /*End*/,
	30,
	16 /*Default*/,
	12,
	9,
	6,
	0  /*End*/ };


//
// Sound volume adjustment
//
UInt8 AdjustSoundVolumeActive = 0;


//
// Keys
//
#define FIRST_ICON_X            5
#define ICONS_Y                 10
#define ICONS_SUB_Y             1
#define BACKGROUND_Y            9
#define BACKGROUND_SUB_Y        0

#define ARROW_X                 287
#define ARROW_Y                 1
#define ARROW_WIDTH             29
#define ARROW_HEIGHT            38

#define KEY_ICON_WIDTH          32
#define KEY_ICON_HEIGHT         32

#define EMU_BACKGROUND_WIDTH    54
#define CPC_BACKGROUND_1_WIDTH  8
#define CPC_BACKGROUND_2_WIDTH  180
#define BACKGROUND_HEIGHT       34

#define ESC_KEY_WIDTH           32
#define COPY_KEY_WIDTH          32
#define SHIFT_KEY_WIDTH         40
#define CTRL_KEY_WIDTH          32
#define CAPSLOCK_KEY_WIDTH      40
#define TAB_KEY_WIDTH           32
#define DEL_KEY_WIDTH           32
#define ENTER_KEY_WIDTH         40

#define RESET_KEY_X             FIRST_ICON_X
#define PAUSE_KEY_X             RESET_KEY_X + KEY_ICON_WIDTH
#define TRUESPEED_KEY_X         PAUSE_KEY_X + KEY_ICON_WIDTH
#define SOUND_KEY_X             TRUESPEED_KEY_X + KEY_ICON_WIDTH
#define SOUND_VOLUME_KEY_X      SOUND_KEY_X + KEY_ICON_WIDTH
#define JOYSTICK_KEY_X          SOUND_VOLUME_KEY_X + KEY_ICON_WIDTH
#define AUTOTOGGLE_KEY_X        JOYSTICK_KEY_X + KEY_ICON_WIDTH
#define EMU_BACKGROUND_X        AUTOTOGGLE_KEY_X + KEY_ICON_WIDTH
#define SUBPANEL_KEY_X          EMU_BACKGROUND_X + EMU_BACKGROUND_WIDTH
//
#define ESC_KEY_X               FIRST_ICON_X
#define COPY_KEY_X              ESC_KEY_X + ESC_KEY_WIDTH
#define SHIFT_KEY_X             COPY_KEY_X + COPY_KEY_WIDTH
#define CTRL_KEY_X              SHIFT_KEY_X + SHIFT_KEY_WIDTH
#define CAPSLOCK_KEY_X          CTRL_KEY_X + CTRL_KEY_WIDTH
#define TAB_KEY_X               CAPSLOCK_KEY_X + CAPSLOCK_KEY_WIDTH
#define DEL_KEY_X               TAB_KEY_X + TAB_KEY_WIDTH
#define ENTER_KEY_X             DEL_KEY_X + DEL_KEY_WIDTH
//
#define CLR_KEY_X               FIRST_ICON_X
#define CPC_BACKGROUND_1_X      CLR_KEY_X + KEY_ICON_WIDTH
#define FX_KEY_X                CPC_BACKGROUND_1_X + CPC_BACKGROUND_1_WIDTH
#define EMUSPEED_KEY_X          FX_KEY_X + KEY_ICON_WIDTH
#define CPC_BACKGROUND_2_X      EMUSPEED_KEY_X + KEY_ICON_WIDTH


static const tKeySetting DriveKeysSettings[] =
{
	// tKeySetting
	// Left, Top, Width, Height,
	// resKeyUp, resKeyDown,
	// KeyMode, CPCKey, charCode_Normal, charCode_Shift,
	// OnPushedP, OnReleasedP, KeyHighlight
	
  // Drive Swap Key
  { 286, 4, 32, 32,
    DriveKey_Swap_Released, DriveKey_Swap_Pushed,
    KeyImpulse, CPC_KEY_NONE, 0, 0,
    CPCSwapDrive, NULL},
  // Disk Eject Key
  { 2, 17, 26, 20,
    DriveKey_Eject_Released, DriveKey_Eject_Pushed,
    KeyImpulse, CPC_KEY_NONE, 0, 0,
    CPCEjectBoth, NULL},
  // Drive A Key
  { 30, 1, 254, 18,
    NULL, NULL,
    KeyImpulse, CPC_KEY_NONE, 0, 0,
    ChangeDriveAContent, NULL},
  // Drive B Key
  { 30, 20, 254, 18,
    NULL, NULL,
    KeyImpulse, CPC_KEY_NONE, 0, 0,
    ChangeDriveBContent, NULL},
};


static const tKeySetting EmulatorKeysSettings[] =
{
	// tKeySetting
	// Left, Top, Width, Height,
	// resKeyUp, resKeyDown,
	// KeyMode, CPCKey, charCode_Normal, charCode_Shift,
	// OnPushedP, OnReleasedP
	
  // Reset Key
  { RESET_KEY_X, ICONS_Y, KEY_ICON_WIDTH, KEY_ICON_HEIGHT,
    EmuKey_Reset_Released, EmuKey_Reset_Pushed,
    KeyImpulse, CPC_KEY_NONE, 0, 0,
    NULL, CPCResetWithConfirmation },
  // Pause Key
  { PAUSE_KEY_X, ICONS_Y, KEY_ICON_WIDTH, KEY_ICON_HEIGHT,
    EmuKey_Pause_Released, EmuKey_Pause_Pushed,
    KeyToggle, CPC_KEY_NONE, 0, 0,
    EmulatorFreeze, EmulatorUnfreeze },
  // True Speed Key
  { TRUESPEED_KEY_X, ICONS_Y, KEY_ICON_WIDTH, KEY_ICON_HEIGHT,
    EmuKey_TrueSpeed_Released, EmuKey_TrueSpeed_Pushed,
    KeyToggle, CPC_KEY_NONE, 0, 0,
    EnableTrueSpeed, DisableTrueSpeed },
  // Sound Key
  { SOUND_KEY_X, ICONS_Y, KEY_ICON_WIDTH, KEY_ICON_HEIGHT,
    EmuKey_Sound_Released, EmuKey_Sound_Pushed,
    KeyToggle, CPC_KEY_NONE, 0, 0,
    EnableSound, DisableSound },
  // Sound Volume Key
  { SOUND_VOLUME_KEY_X, ICONS_Y, KEY_ICON_WIDTH, KEY_ICON_HEIGHT,
    EmuKey_Sound_Volume_Released, EmuKey_Sound_Volume_Pushed,
    KeyToggle, CPC_KEY_NONE, 0, 0,
    AdjustSoundVolumeKeyPressed, AdjustSoundVolumeKeyReleased },
  // Joystick Key
  { JOYSTICK_KEY_X, ICONS_Y, KEY_ICON_WIDTH, KEY_ICON_HEIGHT,
    EmuKey_Joystick_Released, EmuKey_Joystick_Pushed,
    KeyToggle, CPC_KEY_NONE, 0, 0,
    EnableJoystick, DisableJoystick },
  // AutoToggle Key
  { AUTOTOGGLE_KEY_X, ICONS_Y, KEY_ICON_WIDTH, KEY_ICON_HEIGHT,
    CPCKey_AutoToggle_Released, CPCKey_AutoToggle_Pushed,
    KeyToggle, CPC_KEY_NONE, 0, 0,
    AutoToggleKeyPressed, AutoToggleKeyReleased },
  // Background
  { EMU_BACKGROUND_X, BACKGROUND_Y, EMU_BACKGROUND_WIDTH, BACKGROUND_HEIGHT,
    EmuKey_Background, EmuKey_Background,
    KeyStatic, CPC_KEY_NONE, 0, 0,
    NULL, NULL },
  // SubPanel Key
  { SUBPANEL_KEY_X, ICONS_Y, KEY_ICON_WIDTH, KEY_ICON_HEIGHT,
    EmuKey_SubPanel_Released, EmuKey_SubPanel_Pushed,
    KeyToggle, CPC_KEY_NONE, 0, 0,
    SubPanelActivePressed, SubPanelActiveReleased },
};


static const tKeySetting CPCColouredKeysSettings[] =
{
	// tKeySetting
	// Left, Top, Width, Height,
	// resKeyUp, resKeyDown,
	// KeyMode, CPCKey, charCode_Normal, charCode_Shift,
	// OnPushedP, OnReleasedP
	
  // ESC Key
  { ESC_KEY_X, ICONS_SUB_Y, ESC_KEY_WIDTH, KEY_ICON_HEIGHT,
    CPCKey_ESC_Released, CPCKey_ESC_Pushed,
    KeyImpulse, CPC_KEY_ESC, 0, 0,
    NULL, NULL },
  // COPY Key
  { COPY_KEY_X, ICONS_SUB_Y, COPY_KEY_WIDTH, KEY_ICON_HEIGHT,
    CPCKey_COPY_Released, CPCKey_COPY_Pushed,
    KeyImpulse, CPC_KEY_COPY, 0, 0,
    NULL, NULL },
  // SHIFT Key
  { SHIFT_KEY_X, ICONS_SUB_Y, SHIFT_KEY_WIDTH, KEY_ICON_HEIGHT,
    CPCKey_SHIFT_Released, CPCKey_SHIFT_Pushed,
    KeyToggle, CPC_KEY_NONE, 0, 0,
    SHIFTKeyPressed, SHIFTKeyPressed },
  // CTRL Key
  { CTRL_KEY_X, ICONS_SUB_Y, CTRL_KEY_WIDTH, KEY_ICON_HEIGHT,
    CPCKey_CTRL_Released, CPCKey_CTRL_Pushed,
    KeyToggle, CPC_KEY_NONE, 0, 0,
    CTRLKeyPressed, CTRLKeyPressed },
  // CAPSLOCK Key
  { CAPSLOCK_KEY_X, ICONS_SUB_Y, CAPSLOCK_KEY_WIDTH, KEY_ICON_HEIGHT,
    CPCKey_CAPSLOCK_Released, CPCKey_CAPSLOCK_Pushed,
    KeyImpulse, CPC_KEY_CAPSLOCK, 0, 0,
    NULL, NULL },
  // TAB Key
  { TAB_KEY_X, ICONS_SUB_Y, TAB_KEY_WIDTH, KEY_ICON_HEIGHT,
    CPCKey_TAB_Released, CPCKey_TAB_Pushed,
    KeyImpulse, CPC_KEY_TAB, 0, 0,
    NULL, NULL },
  // DEL Key
  { DEL_KEY_X, ICONS_SUB_Y, DEL_KEY_WIDTH, KEY_ICON_HEIGHT,
    CPCKey_DEL_Released, CPCKey_DEL_Pushed,
    KeyImpulse, CPC_KEY_DEL, 0, 0,
    NULL, NULL },
  // ENTER Key
  { ENTER_KEY_X, ICONS_SUB_Y, ENTER_KEY_WIDTH, KEY_ICON_HEIGHT,
    CPCKey_ENTER_Released, CPCKey_ENTER_Pushed,
    KeyImpulse, CPC_KEY_ENTER, 0, 0,
    NULL, NULL },
  // Top Arrow
  { ARROW_X, ARROW_Y, ARROW_WIDTH, ARROW_HEIGHT,
    NULL, NULL,
    KeyImpulse, CPC_KEY_NONE, 0, 0,
    DisplayCPCSpecialKeyboard, NULL },
};


static const tKeySetting CPCSpecialKeysSettings[] =
{
	// tKeySetting
	// Left, Top, Width, Height,
	// resKeyUp, resKeyDown,
	// KeyMode, CPCKey, charCode_Normal, charCode_Shift,
	// OnPushedP, OnReleasedP
	
  // CLR Key
  { CLR_KEY_X, ICONS_SUB_Y, KEY_ICON_WIDTH, KEY_ICON_HEIGHT,
    CPCKey_CLR_Released, CPCKey_CLR_Pushed,
    KeyImpulse, CPC_KEY_CLR, 0, 0,
    NULL, NULL },
  // Background 1
  { CPC_BACKGROUND_1_X, BACKGROUND_SUB_Y, CPC_BACKGROUND_1_WIDTH, BACKGROUND_HEIGHT,
    CPCKey_Background_1, CPCKey_Background_1,
    KeyStatic, CPC_KEY_NONE, 0, 0,
    NULL, NULL },
  // Fx Key
  { FX_KEY_X, ICONS_SUB_Y, KEY_ICON_WIDTH, KEY_ICON_HEIGHT,
    EmuKey_Fx_Released, EmuKey_Fx_Pushed,
    KeyToggle, CPC_KEY_NONE, 0, 0,
    EnableFonctionKeys, DisableFonctionKeys },
  // Display Emulator Speed Key
  { EMUSPEED_KEY_X, ICONS_SUB_Y, KEY_ICON_WIDTH, KEY_ICON_HEIGHT,
    EmuKey_EmuSpeed_Released, EmuKey_EmuSpeed_Pushed,
    KeyToggle, CPC_KEY_NONE, 0, 0,
    EnableDisplayEmuSpeed, DisableDisplayEmuSpeed },
  // Background 2
  { CPC_BACKGROUND_2_X, BACKGROUND_SUB_Y, CPC_BACKGROUND_2_WIDTH, BACKGROUND_HEIGHT,
    CPCKey_Background_2, CPCKey_Background_2,
    KeyStatic, CPC_KEY_NONE, 0, 0,
    NULL, NULL },
  // Arrow
  { ARROW_X, ARROW_Y, ARROW_WIDTH, ARROW_HEIGHT,
    NULL, NULL,
    KeyImpulse, CPC_KEY_NONE, 0, 0,
    DisplayCPCColouredKeyboard, NULL },
};




tKey DriveKeys[NB_DRIVE_KEYS];
tKey EmulatorKeys[NB_EMULATORS_KEYS];
tKey CPCColouredKeys[NB_CPC_COLOURED_KEYS];
tKey CPCSpecialKeys[NB_CPC_SPECIAL_KEYS];

tKey* EmulatorKeysPanelP = CPCColouredKeys;
tULong NbEmulatorKeys = NB_CPC_COLOURED_KEYS;

tEmulatorKeysStatus EmulatorKeysStatus;

static tUShort cpcKeyUp;
static tUShort cpcKeyDown;
static tUShort cpcKeyRight;
static tUShort cpcKeyLeft;
static tUShort cpcKeyCenter;
static UInt32 centerKeyMask;
static UInt32 HardCPCKeyCodeMaskA;
static UInt32 HardCPCKeyCodeMaskB;
static UInt32 HardCPCKeyCodeMaskC;
static UInt32 HardCPCKeyCodeMaskD;

//
// CPC Keyboards
//
tKeyboard* ActiveKeyboardP = NULL;
tKeyboard* DIAKeyboardP = NULL;
tKeyboard* MiniKeyboardP = NULL;


typedef struct
{
  UInt32* KeycodeMaskP;
  UInt32  HardKeyIndex;
} tUpdateCPCKeycode;


tUShort GetKeySHIFTandCTRLState(tVoid)
/***********************************************************************
 *
 *  GetKeySHIFTandCTRLState
 *
 ***********************************************************************/
{
tUShort usKeyStatus = 0;

  // Save SHIFT state
  if (IS_KEY_PRESSED(CPC_KBD_SHIFT))
  {
    usKeyStatus |= MOD_CPC_SHIFT;
  }

  // Save CTRL state
  if (IS_KEY_PRESSED(CPC_KBD_CTRL))
  {
    usKeyStatus |= MOD_CPC_CTRL;
  }

  return usKeyStatus;
}
/*----------------------------------------------------------------------------*/


void SetKeySHIFTandCTRLState(tUShort usKeyState)
/***********************************************************************
 *
 *  SetKeySHIFTandCTRLState
 *
 ***********************************************************************/
{
  if (usKeyState & MOD_CPC_SHIFT) // CPC SHIFT key required?
  {
    PRESS_KEY(CPC_KBD_SHIFT); // key needs to be SHIFTed
  }
  else
  {
    RELEASE_KEY(CPC_KBD_SHIFT); // make sure key is unSHIFTed
  }

  if (usKeyState & MOD_CPC_CTRL) // CPC CONTROL key required?
  {
    PRESS_KEY(CPC_KBD_CTRL); // CONTROL key is held down
  }
  else
  {
    RELEASE_KEY(CPC_KBD_CTRL); // // make sure CONTROL key is released
  }
}
/*----------------------------------------------------------------------------*/


Boolean KeyDownHandleEvent(EventPtr Event)
/***********************************************************************
 *
 *  KeyDownHandleEvent
 *
 ***********************************************************************/
{
Boolean Handled = false;
WChar Char;
tUChar cpc_index;
tUShort cpc_key = KEY_INVALID;

  if (usCurrentPressedKey != PRESSED_KEY_INVALID)
  	return true;

  //
  // Copied from Core/System/Chars.h
  //
  Char = Event->data.keyDown.chr;

  if (Char <= 128)
  {
    switch(Char)
    {
      case chrLeftArrow:
      {
        cpc_key = cpc_kbd[CPC_KEY_CUR_LEFT];
      }
      break;

      case chrRightArrow:
      {
        cpc_key = cpc_kbd[CPC_KEY_CUR_RIGHT];
      }
      break;

      case chrUpArrow:
      {
        cpc_key = cpc_kbd[CPC_KEY_CUR_UP];
      }
      break;

      case chrDownArrow:
      {
        cpc_key = cpc_kbd[CPC_KEY_CUR_DOWN];
      }
      break;

      // All other keys
      default:
      {
        if ( (EmulatorKeysStatus.FxKeyStatus == KeyPressed) &&
             (Char >= chrDigitZero) && (Char <= chrDigitNine) )
        {
          // "Function" numerical keys simulation
          cpc_key = cpc_kbd[CPC_KEY_F0 + (Char - chrDigitZero)];
        }
        else
        {
          cpc_index = palm_kbd[(tUChar)Char];
          if (cpc_index != KEY_INVALID)
          {
            cpc_key = cpc_kbd[cpc_index];
          }
        }
      }
    }
  }
  // Copied from Core/System/CharLatin.h
  else if (Char == chrPoundSign)
  {
    cpc_key = cpc_kbd[CPC_KEY_POUND];
  }

  if (cpc_key != KEY_INVALID)
  {
    // Save pressed key
    usCurrentPressedKey = cpc_key & 0xFF;
    // Save SHIFT and CTRL state
    usCurrentPressedKey |= GetKeySHIFTandCTRLState();

    // key is being held down
    PRESS_KEY(cpc_key); 

    // Restore SHIFT and CTRL state
    SetKeySHIFTandCTRLState(cpc_key);

    debounceTimeoutTicks = TimGetTicks() + u32DebounceInTicks;

    Handled = true;
  }

  return Handled;
}
/*----------------------------------------------------------------------------*/


Boolean PenDownHandleEvent(EventPtr Event)
/***********************************************************************
 *
 *  PenDownHandleEvent
 *
 ***********************************************************************/
{
Boolean Handled = false;
tKey* keyP;
Coord ScreenY = Event->screenY * 2; // Double density display
Coord ScreenX = Event->screenX * 2; // Double density display
UInt8 keyLoop;
Coord y_base;

  //
  // Keyboard panel
  //
  if (prefP->Display320x480)
  {
    if (!LandscapeActive)
    {
      y_base = GetYBaseKeyBoardPanel();

      if ( (ScreenY >= y_base) &&
           (ScreenY < (y_base + Y_SIZE_KEYBOARD_PANEL)) )
      {
        PenDownHandleKeyboardEvent(DIAKeyboardP,
      	                           ScreenX,
      	                           ScreenY,
      	                           y_base);
      	                                
        return true;
      }
    }
    
  
    //
    // Landscape border
    //
    else
    {
      if ( (ScreenX < ONSCREEN_OFFSET_LANDSCAPE_X) ||
           (ScreenX >= (ONSCREEN_OFFSET_LANDSCAPE_X + CPC_SCR_WIDTH)) ||
           (ScreenY < ONSCREEN_OFFSET_LANDSCAPE_Y) ||
           (ScreenY >= (ONSCREEN_OFFSET_LANDSCAPE_Y + CPC_VISIBLE_SCR_HEIGHT)) )
      {
        UInt32 data;
          
        if (StatGetAttribute(statAttrBarVisible,
                             &data) == errNone)
        {
          if (data) // Bar is visible
          {
            StatHide();
          }
          else // Bar is hidden
          {
            StatShow();
          }
        }
        
        return true;
      }
    }
  }

  //
  // CPC Screen
  //
  if ( (ScreenY < (FullscreenActive ? SCREEN_PALM_HEIGHT_FS : SCREEN_PALM_HEIGHT)) ||
       LandscapeActive )
  {
    // OnScreen Rocker
    if (prefP->OnScreenRockerActive)
    {
      SetCursorAndJoystickState(DetectScreenDirectionalInput(ScreenX,
                                                             ScreenY,
                                                             FullscreenActive,
                                                             LandscapeActive,
                                                             prefP->Display320x480),
                                &oldScreenKey);
                                  
      return true;
    }
  
    // OnScreen LightPen
    else if (prefP->OnScreenLightPenActive)
    {
      SetLightPenCoordinates(ScreenX,
                             ScreenY,
                             FullscreenActive);
                               
      return true;
    }

    // OnScreen West Phaser
    else if (prefP->OnScreenWestPhaserActive) 
    {
#ifdef _DEBUG          	
      NativeCPC->lightgun_debug_flags = 0;
      NativeCPC->lightgun_debug_counter = 0;
#endif /* _DEBUG */     

      //NativeCPC->lightgun_beam_detect = 1;
      NativeCPC->lightgun_beam_key_line = (tULong)EndianSwap32(LINE_FROM_KEYCODE(WEST_PHASER_BEAM_KEY));
      NativeCPC->lightgun_beam_key_mask = (tULong)EndianSwap32(~(1 << ROW_FROM_KEYCODE(WEST_PHASER_BEAM_KEY)));
      NativeCPC->lightgun_sensitivity = (tULong)EndianSwap32(WEST_PHASER_SENSITIVITY);
      //NativeCPC->lightgun_one_shot = 1;

      PRESS_KEY(WEST_PHASER_TRIG_KEY);
      WestPhaserTriggerActiveDelay = TimGetTicks() + WEST_PHASER_TRIG_TICKS;
       
      SetPhaserCoordinates(ScreenX,
                           ScreenY,
                           FullscreenActive);
                               
      return true;
    }

    // OnScreen GunStick
    else if (prefP->OnScreenGunstickActive) 
    {
      NativeCPC->lightgun_beam_detect = 1;
      NativeCPC->lightgun_beam_key_line = (tULong)EndianSwap32(LINE_FROM_KEYCODE(GUNSTICK_BEAM_KEY));
      NativeCPC->lightgun_beam_key_mask = (tULong)EndianSwap32(~(1 << ROW_FROM_KEYCODE(GUNSTICK_BEAM_KEY)));
      NativeCPC->lightgun_sensitivity = (tULong)EndianSwap32(GUNSTICK_SENSITIVITY);

      PRESS_KEY(GUNSTICK_TRIG_KEY);
       
      SetPhaserCoordinates(ScreenX,
                           ScreenY,
                           FullscreenActive);
                               
      return true;
    }

    // OnScreen Magnum Gun
    else if (prefP->OnScreenMagnumGunActive)
    {
      NativeCPC->lightgun_random_crtc = 0;
      
      SetLightPenCoordinates(ScreenX,
                             ScreenY,
                             FullscreenActive);
                               
      return true;
    }
  }

  // Fullscreen active, do not manage others panels
  if (FullscreenActive)
    return false;
    
  // Mini Keyboard not activated or 320x480 display
  if ( prefP->Display320x480 || !MiniKeyboardActive )
  {
    //
    // Emulator Buttons panel
    //
    y_base = GetYBaseEmulatorPanel();
  
    if ( (ScreenY >= y_base) &&
         (ScreenY < (y_base+Y_SIZE_EMULATOR_PANEL)) )
    {
      keyP = EmulatorKeys;
      keyLoop = NB_EMULATORS_KEYS;
      while ( (Handled == false) && (keyLoop--) )
      {
        Handled = DetectPressedKey(keyP,
                                   ScreenX,
                                   ScreenY,
                                   NULL);
        keyP++;
      }
  
      return true;
    }
  
    //
    // Sub panel
    //
    y_base = GetYBaseSubPanel();
  
    if ( (ScreenY >= y_base) &&
         (ScreenY < (y_base + Y_SIZE_SUB_PANEL)) )
    {
      keyP = !SubPanelActive ? DriveKeys : EmulatorKeysPanelP;
      keyLoop = !SubPanelActive ? NB_DRIVE_KEYS : NbEmulatorKeys;
      while ( (Handled == false) && (keyLoop--) )
      {
        Handled = DetectPressedKey(keyP,
                                   ScreenX,
                                   ScreenY,
                                   NULL);
        keyP++;
      }
  
      return true;
    }
  }
  else
  {
    //
    // Mini Keyboard
    //
    y_base = GetYBaseEmulatorPanel();

    if ( (ScreenY >= y_base) &&
         (ScreenY < (y_base + Y_SIZE_MINI_KEYBOARD_PANEL)) )
    {
      PenDownHandleKeyboardEvent(MiniKeyboardP,
      	                         ScreenX,
      	                         ScreenY,
      	                         y_base);
      	                         
      return true;
    }
  }

  return false;
}
/*----------------------------------------------------------------------------*/


static Boolean PenDownHandleKeyboardEvent(tKeyboard* keyboardP,
                                          tUShort ScreenX,
                                          tUShort ScreenY,
                                          tUShort y_base)
/***********************************************************************
 *
 *  PenDownHandleKeyboardEvent
 *
 ***********************************************************************/
{
const tKeySection* sectionP = keyboardP->sectionsP;
tKey* SectionkeyP = keyboardP->keysP;
tKey* keyP;
UInt8 keyLoop;
UInt8 sectionLoop = keyboardP->headerP->NbTotalSections;
Boolean Handled = false;

  for ( ;
        (Handled == false) && (sectionLoop--);
        SectionkeyP += (sectionP++)->nb_keys )
  {
  	if (ScreenX < sectionP->Left) continue;
  	if (ScreenX >= sectionP->Right) continue;
  	
  	if (ScreenY < (sectionP->Top+y_base)) continue;
  	if (ScreenY >= (sectionP->Bottom+y_base)) continue;
  	
    keyLoop = sectionP->nb_keys;
    keyP = SectionkeyP;
      
    while ( (Handled == false) && (keyLoop--) )
    {
      Handled = DetectPressedKey(keyP,
                                 ScreenX,
                                 ScreenY,
                                 &usSaveShiftCtrl);
      keyP++;
    }
  }

  return Handled;
}
/*----------------------------------------------------------------------------*/


Boolean PenMoveHandleEvent(EventPtr Event)
/***********************************************************************
 *
 *  PenMoveHandleEvent
 *
 ***********************************************************************/
{
tUShort ScreenX = Event->screenX * 2; // Double density display
tUShort ScreenY = Event->screenY * 2; // Double density display

  do
  {
    // OnScreen Rocker
    if (prefP->OnScreenRockerActive)
      continue;
      
    // On screen light pen active
    if (prefP->OnScreenLightPenActive)
      continue;

    // On screen Magnum gun active
    if (prefP->OnScreenMagnumGunActive)
      continue;

    // On screen west phaser active
    if (prefP->OnScreenWestPhaserActive)
      continue;

    // On screen gun stick active
    if (prefP->OnScreenGunstickActive)
      continue;

    return true;
  }
  while (0);

  // Pen below CPC screen
  if (ScreenY >= (FullscreenActive ? SCREEN_PALM_HEIGHT_FS : SCREEN_PALM_HEIGHT))
    return true;
    
  //
  // CPC Screen
  //
  if (prefP->OnScreenRockerActive) // OnScreen Rocker
  {
    SetCursorAndJoystickState(DetectScreenDirectionalInput(ScreenX,
                                                           ScreenY,
                                                           FullscreenActive,
                                                           LandscapeActive,
                                                           prefP->Display320x480),
                              &oldScreenKey);
  }

  else if (prefP->OnScreenLightPenActive)
  {
    SetLightPenCoordinates(ScreenX,
                           ScreenY,
                           FullscreenActive);
  }

  // OnScreen Magnum Gun
  else if (prefP->OnScreenMagnumGunActive)
  {
    SetLightPenCoordinates(ScreenX,
                           ScreenY,
                           FullscreenActive);
  }
  // OnScreen West Phaser
  else if (prefP->OnScreenWestPhaserActive)
  {
    SetPhaserCoordinates(ScreenX,
                         ScreenY,
                         FullscreenActive);
  }

  return true;
}
/*----------------------------------------------------------------------------*/


Boolean PenUpHandleEvent(EventPtr Event)
/***********************************************************************
 *
 *  PenUpHandleEvent
 *
 ***********************************************************************/
{
Boolean Handled = false;

  Handled = ReleaseLastPressedKey();

  // Screen keys
  if (oldScreenKey != 0)
  {
    SetCursorAndJoystickState(0,
                              &oldScreenKey);
  }

  if (prefP->OnScreenWestPhaserActive)
  {
    NativeCPC->lightgun_beam_detect = 0;
    RELEASE_KEY(WEST_PHASER_TRIG_KEY);
  }
  else if (prefP->OnScreenMagnumGunActive)
  {
    NativeCPC->lightgun_random_crtc = 1;
  }
  else if (prefP->OnScreenGunstickActive)
  {
    NativeCPC->lightgun_beam_detect = 0;
    RELEASE_KEY(GUNSTICK_TRIG_KEY);
  }

  return Handled;
}
/*----------------------------------------------------------------------------*/


#define INVERT_KEY_DISPLAY(keyP) \
if (keyP->KeyHighlight) \
{ \
  RectangleType rectT; \
  \
  rectT.topLeft.x = keyP->Left; \
  rectT.topLeft.y = keyP->Top; \
  rectT.extent.x = keyP->Right - keyP->Left; \
  rectT.extent.y = keyP->Bottom - keyP->Top; \
  WinInvertRectangleFrame(roundFrame, &rectT); \
}

static Boolean DetectPressedKey(tKey* keyP,
                                tUShort ScreenX,
                                tUShort ScreenY,
                                tUShort* keySaveP)
/***********************************************************************
 *
 *  DetectPressedKey
 *
 ***********************************************************************/
{
const tKeySetting* settingP = keyP->settingP;
tUChar keyCode;
UInt16 oldCoordSys;

#ifndef __RELEASE__
  if (keyP == NULL)
    ErrDisplay("keyP == NULL");
#endif /* __RELEASE__ */

  if ( (ScreenY < keyP->Top) || (ScreenY >= (keyP->Bottom)) )
    return false;

  if ( (ScreenX < keyP->Left) || (ScreenX >= (keyP->Right)) )
    return false;

  if (settingP->KeyMode == KeyStatic)
    return false;

  if (lastPressedKeyP)
    return false;

  keyCode = (tUChar)cpc_kbd[keyP->CPCKey];
  
  // Pen down on that key
  if ( (settingP->KeyMode == KeyImpulse) ||
       ((settingP->KeyMode == KeyToggle) && (keyP->KeyStatus == KeyReleased)) )
  {
    keyP->KeyStatus = KeyPressed;
    
    if (settingP->KeyMode == KeyImpulse)
    {
      lastPressedKeyP = keyP;
      lastPressedKeyStateP = keySaveP;
    }

    // Execute associated routine
    if (keyP->OnPushedP != NULL)
    {
      keyP->OnPushedP();
    }

    // Simulate associated CPC Key press
    if (keyP->CPCKey != CPC_KEY_NONE)
    {
      if (keySaveP)
      {
        // Save pressed key, SHIFT and CTRL state
        *keySaveP = GetKeySHIFTandCTRLState();
      }

      // Update hardware matrix
      PRESS_KEY(keyCode);
    }
  }
  else if ((settingP->KeyMode == KeyToggle) && (keyP->KeyStatus == KeyPressed))
  {
    keyP->KeyStatus = KeyReleased;

    // Execute associated routine
    if (keyP->OnReleasedP != NULL)
    {
      keyP->OnReleasedP();
    }

    // Simulate associated CPC Key release
    if (settingP->CPCKey != CPC_KEY_NONE)
    {
      // Update hardware keyboard matrix
      RELEASE_KEY(keyCode);
    }
  }
  else
  {
  	return true;
  }

  // Update key display
  oldCoordSys = WinSetCoordinateSystem(kCoordinatesNative);
    
  if ( settingP->resKeyUp && settingP->resKeyDown )
  {
    DisplayKey(keyP);
  }
  else
  {
    // Invert key display
    INVERT_KEY_DISPLAY(keyP)
  }
    
  WinSetCoordinateSystem(oldCoordSys);

  return true;
}
/*----------------------------------------------------------------------------*/



static Boolean ReleaseLastPressedKey(void)
/***********************************************************************
 *
 *  ReleaseLastPressedKey
 *
 ***********************************************************************/
{
const tKeySetting* settingP;
UInt16 oldCoordSys;

  if (lastPressedKeyP == NULL)
    return false;

  settingP = lastPressedKeyP->settingP;
    
  lastPressedKeyP->KeyStatus = KeyReleased;

  // Execute associated routine
  if (lastPressedKeyP->OnReleasedP != NULL)
  {
    lastPressedKeyP->OnReleasedP();
  }

  // Simulate associated CPC Key release
  if (lastPressedKeyP->CPCKey != CPC_KEY_NONE)
  {
    RELEASE_KEY(cpc_kbd[settingP->CPCKey]);

    if (lastPressedKeyStateP)
    {
      // Restore SHIFT and CTRL state
      SetKeySHIFTandCTRLState(*lastPressedKeyStateP);
    }
  }

  // Update key display
  oldCoordSys = WinSetCoordinateSystem(kCoordinatesNative);
  
  if ( settingP->resKeyUp && settingP->resKeyDown )
  {
    DisplayKey(lastPressedKeyP);
  }
  else
  {
    // Invert key display
    INVERT_KEY_DISPLAY(lastPressedKeyP)
  }

  WinSetCoordinateSystem(oldCoordSys);

  lastPressedKeyP = NULL;
  lastPressedKeyStateP = NULL;

  return true;
}
/*----------------------------------------------------------------------------*/


void SelectCPCColouredKeyboard(void)
/***********************************************************************
 *
 *  SelectCPCColouredKeyboard
 *
 ***********************************************************************/
{
  EmulatorKeysPanelP = CPCColouredKeys;
  NbEmulatorKeys = sizeof(CPCColouredKeys) / sizeof(CPCColouredKeys[0]);
}
/*----------------------------------------------------------------------------*/


void SelectCPCSpecialKeyboard(void)
/***********************************************************************
 *
 *  SelectCPCSpecialKeyboard
 *
 ***********************************************************************/
{
  EmulatorKeysPanelP = CPCSpecialKeys;
  NbEmulatorKeys = sizeof(CPCSpecialKeys) / sizeof(CPCSpecialKeys[0]);
}
/*----------------------------------------------------------------------------*/


static void DisplayCPCColouredKeyboard(void)
/***********************************************************************
 *
 *  DisplayCPCColouredKeyboard
 *
 ***********************************************************************/
{
  SelectCPCColouredKeyboard();

  FrmUpdateForm(MainForm,
                SubPanelRedrawUpdateCode);
}
/*----------------------------------------------------------------------------*/


static void DisplayCPCSpecialKeyboard(void)
/***********************************************************************
 *
 *  DisplayCPCSpecialKeyboard
 *
 ***********************************************************************/
{
  SelectCPCSpecialKeyboard();
  
  FrmUpdateForm(MainForm,
                SubPanelRedrawUpdateCode);
}
/*----------------------------------------------------------------------------*/


void EnableJoystick(void)
/***********************************************************************
 *
 *  EnableJoystick
 *
 ***********************************************************************/
{
UInt32 OldKeyState = ~0;

  // Release potential pressed keys
  SetCursorAndJoystickState(0,
                            &OldKeyState);

  EmulatorKeysStatus.JoystickKeyStatus = KeyPressed;
  
  // Joystick
  cpcKeyUp = cpc_kbd[CPC_KEY_J0_UP];
  cpcKeyDown = cpc_kbd[CPC_KEY_J0_DOWN];
  cpcKeyRight = cpc_kbd[CPC_KEY_J0_RIGHT];
  cpcKeyLeft = cpc_kbd[CPC_KEY_J0_LEFT];
  cpcKeyCenter = cpc_kbd[CPC_KEY_J0_FIRE1];
  
  ShowMessage(JOYSTICK_ON_MSG);
}
/*----------------------------------------------------------------------------*/
void DisableJoystick(void)
/***********************************************************************
 *
 *  DisableJoystick
 *
 ***********************************************************************/
{
UInt32 OldKeyState = ~0;

const tUChar CenterKeys[] =
{
  CPC_KEY_ESC,
  CPC_KEY_RETURN,
  CPC_KEY_COPY,
  CPC_KEY_CONTROL,
  CPC_KEY_RSHIFT,
  CPC_KEY_SPACE,
  CPC_KEY_TAB,
  CPC_KEY_CAPSLOCK
};

  // Release potential pressed keys
  SetCursorAndJoystickState(0,
                            &OldKeyState);

  EmulatorKeysStatus.JoystickKeyStatus = KeyReleased;
  
  // Cursor
  cpcKeyUp = cpc_kbd[CPC_KEY_CUR_UP];
  cpcKeyDown = cpc_kbd[CPC_KEY_CUR_DOWN];
  cpcKeyRight = cpc_kbd[CPC_KEY_CUR_RIGHT];
  cpcKeyLeft = cpc_kbd[CPC_KEY_CUR_LEFT];
  cpcKeyCenter = cpc_kbd[CenterKeys[prefP->RockerCenterKeyIndex]];
  
  ShowMessage(JOYSTICK_OFF_MSG);
}
/*----------------------------------------------------------------------------*/


static void EnableTrueSpeed(void)
/***********************************************************************
 *
 *  EnableTrueSpeed
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (prefP == NULL)
    ErrDisplay("prefP == NULL");
#endif /* __RELEASE__ */

  prefP->CPCTrueSpeed = 1;
  prefP->PreferencesChanged = 1;
  ShowMessage(CPC_TRUE_SPEED_MSG);
}
/*----------------------------------------------------------------------------*/
static void DisableTrueSpeed(void)
/***********************************************************************
 *
 *  DisableTrueSpeed
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (prefP == NULL)
    ErrDisplay("prefP == NULL");
#endif /* __RELEASE__ */

  prefP->CPCTrueSpeed = 0;
  prefP->PreferencesChanged = 1;
  ShowMessage(FULL_SPEED_MSG);
}
/*----------------------------------------------------------------------------*/


static void EnableDisplayEmuSpeed(void)
/***********************************************************************
 *
 *  EnableDisplayEmuSpeed
 *
 ***********************************************************************/
{
  DisplayEmuSpeed = 1;
}
/*----------------------------------------------------------------------------*/
static void DisableDisplayEmuSpeed(void)
/***********************************************************************
 *
 *  DisableDisplayEmuSpeed
 *
 ***********************************************************************/
{
  DisplayEmuSpeed = 0;
}
/*----------------------------------------------------------------------------*/


void EnableSound(void)
/***********************************************************************
 *
 *  EnableSound
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (prefP == NULL)
    ErrDisplay("prefP == NULL");
#endif /* __RELEASE__ */

  EmulatorKeysStatus.SoundKeyStatus = KeyPressed;
  prefP->SoundEnabled = 1;
  prefP->PreferencesChanged = 1;

	// Should be done after prefP update
  SoundPlay(NativeCPC);
  
  ShowMessage(AUDIO_MSG);
}
/*----------------------------------------------------------------------------*/
void DisableSound(void)
/***********************************************************************
 *
 *  DisableSound
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (prefP == NULL)
    ErrDisplay("prefP == NULL");
#endif /* __RELEASE__ */

  SoundPause(NativeCPC);

  EmulatorKeysStatus.SoundKeyStatus = KeyReleased;
  prefP->SoundEnabled = 0;
  prefP->PreferencesChanged = 1;
  
  ShowMessage(MUTE_MSG);
}
/*----------------------------------------------------------------------------*/


static void EnableFonctionKeys(void)
/***********************************************************************
 *
 *  EnableFonctionKeys
 *
 ***********************************************************************/
{
  EmulatorKeysStatus.FxKeyStatus = KeyPressed;
  ShowMessage(FX_KEYS_MSG);
}
/*----------------------------------------------------------------------------*/
static void DisableFonctionKeys(void)
/***********************************************************************
 *
 *  DisableFonctionKeys
 *
 ***********************************************************************/
{
  EmulatorKeysStatus.FxKeyStatus = KeyReleased;
  ShowMessage(NUM_KEYS_MSG);
}
/*----------------------------------------------------------------------------*/


static void SHIFTKeyPressed(void)
/***********************************************************************
 *
 *  SHIFTKeyPressed
 *
 ***********************************************************************/
{
  if ( (CPCColouredKeys[KEYINDEX_SHIFT].KeyStatus == KeyPressed) ||
       (ActiveKeyboardP->keyRShiftP->KeyStatus == KeyPressed) ||
       (ActiveKeyboardP->keyLShiftP->KeyStatus == KeyPressed) )
  {
    // SHIFT is pressed
    PRESS_KEY(CPC_KBD_SHIFT);
  }
  else
  {
    // SHIFT is released
    RELEASE_KEY(CPC_KBD_SHIFT);
  }
  
	// Update form display
  FrmUpdateForm(MainForm,
                keyboardRedrawUpdateCode);
}
/*----------------------------------------------------------------------------*/


static void CTRLKeyPressed(void)
/***********************************************************************
 *
 *  CTRLKeyPressed
 *
 ***********************************************************************/
{
  if ( (CPCColouredKeys[KEYINDEX_CTRL].KeyStatus == KeyPressed) ||
       (ActiveKeyboardP->keyCTRLP->KeyStatus == KeyPressed) )
  {
    // CTRL is pressed
    PRESS_KEY(CPC_KBD_CTRL);
  }
  else
  {
    // CTRL is released
    RELEASE_KEY(CPC_KBD_CTRL);
  }

	// Update form display
  FrmUpdateForm(MainForm,
                keyboardRedrawUpdateCode);
}
/*----------------------------------------------------------------------------*/


void AutoToggleKeyPressed(void)
/***********************************************************************
 *
 *  AutoToggleKeyPressed
 *
 ***********************************************************************/
{
  AutoToggleActive = 1;
  AutoToggleDurationIndex = AUTOTOGGLE_DURATION_INDEX_DEFAULT;
  AutoToggleDurationInTicks = AutoToggleDurationA[AutoToggleDurationIndex];
}
/*----------------------------------------------------------------------------*/
void AutoToggleKeyReleased(void)
/***********************************************************************
 *
 *  AutoToggleKeyReleased
 *
 ***********************************************************************/
{
  AutoToggleActive = 0;
}
/*----------------------------------------------------------------------------*/


void AdjustSoundVolumeKeyPressed(void)
/***********************************************************************
 *
 *  AdjustSoundVolumeKeyPressed
 *
 ***********************************************************************/
{
  AdjustSoundVolumeActive = 1;
}
/*----------------------------------------------------------------------------*/
void AdjustSoundVolumeKeyReleased(void)
/***********************************************************************
 *
 *  AdjustSoundVolumeKeyReleased
 *
 ***********************************************************************/
{
  AdjustSoundVolumeActive = 0;
}
/*----------------------------------------------------------------------------*/


static void SubPanelActivePressed(void)
/***********************************************************************
 *
 *  SubPanelActivePressed
 *
 ***********************************************************************/
{
  SubPanelActive = 1;
  
  FrmUpdateForm(MainForm,
                emulatorRedrawUpdateCode | SubPanelRedrawUpdateCode);
}
/*----------------------------------------------------------------------------*/
static void SubPanelActiveReleased(void)
/***********************************************************************
 *
 *  SubPanelActiveReleased
 *
 ***********************************************************************/
{
  SubPanelActive = 0;
  
  FrmUpdateForm(MainForm,
                emulatorRedrawUpdateCode | SubPanelRedrawUpdateCode);
}
/*----------------------------------------------------------------------------*/



void SetCursorAndJoystickState(UInt32 keyState,
                               UInt32* oldkeystateP)
/***********************************************************************
 *
 *  SetCursorAndJoystickState
 *
 ***********************************************************************/
{
UInt32 keyDiff;

  //
  // Auto toggle Left-Right
  //
  if (AutoToggleActive)
  {
    keyDiff = keyState ^ u32OldRockerStateWhileAutoToggle;

    // Auto toggle duration update
    if (keyDiff)
    {
      u32OldRockerStateWhileAutoToggle = keyState;

      // Rocker RIGHT pushed = duration divided by 2
      if (keyDiff & keyBitRockerRight)
      {
        if (keyState & keyBitRockerRight)
        {
          if (AutoToggleDurationA[AutoToggleDurationIndex+1])
          {
            AutoToggleDurationInTicks = AutoToggleDurationA[++AutoToggleDurationIndex];
          }
        }
      }
      // Rocker LEFT pushed = duration multiply by 2
      else if (keyDiff & keyBitRockerLeft)
      {
        if (keyState & keyBitRockerLeft)
        {
          if (AutoToggleDurationA[AutoToggleDurationIndex-1])
          {
            AutoToggleDurationInTicks = AutoToggleDurationA[--AutoToggleDurationIndex];
          }
        }
      }
    }

    // Add only left and right simulated rocker status
    keyState &= ~(keyBitRockerRight | keyBitRockerLeft);
    keyState |= u32AutoToggleKeyState;
  }


  //
  // Check if rocker status changed
  //
  keyDiff = keyState ^ *oldkeystateP;
  if (!keyDiff)
    return;


  // Save key states
  *oldkeystateP = keyState;

  //
  // Adjust sound volume
  //
  if (AdjustSoundVolumeActive)
  {
  	if (!AutoToggleActive)
  	{
      if (keyDiff & (keyBitRockerRight))
      {
      	// Rocker RIGHT pushed = Volume raised
        if (keyState & (keyBitRockerRight) )
        {
          SoundIncreaseVolume(1);
          return;
        }
      }
      else if (keyDiff & (keyBitPageUp | keyBitRockerUp))
      {
        // Rocker UP pushed = Volums fast raised
        if (keyState & (keyBitPageUp | keyBitRockerUp))
        {
          SoundIncreaseVolume(10);
          return;
        }
      }
      else if (keyDiff & (keyBitRockerLeft))
      {
        // Rocker LEFT pushed = Volums lowered
        if (keyState & (keyBitRockerLeft))
        {
          SoundDecreaseVolume(1);
          return;
        }
      }
      else if (keyDiff & (keyBitPageDown | keyBitRockerDown))
      {
        // Rocker DOWN pushed = Volums lowered
        if (keyState & (keyBitPageDown | keyBitRockerDown))
        {
          SoundDecreaseVolume(10);
          return;
        }
      }
  	}
  }

  // Rocker UP...
  if (keyDiff & (keyBitPageUp | keyBitRockerUp))
  {
    if (keyState & (keyBitPageUp | keyBitRockerUp))
    {
      PRESS_KEY(cpcKeyUp);
    }
    else
    {
      RELEASE_KEY(cpcKeyUp);
    }
  }
  
  // Rocker DOWN.
  if (keyDiff & (keyBitPageDown | keyBitRockerDown))
  {
    if (keyState & (keyBitPageDown | keyBitRockerDown))
    {
      PRESS_KEY(cpcKeyDown);
    }
    else
    {
      RELEASE_KEY(cpcKeyDown);
    }
  }

  // Rocker RIGHT...
  if (keyDiff & keyBitRockerRight)
  {
    if (keyState & keyBitRockerRight)
    {
      PRESS_KEY(cpcKeyRight);
    }
    else
    {
      RELEASE_KEY(cpcKeyRight);
    }
  }
  
  // Rocker LEFT.
  if (keyDiff & keyBitRockerLeft)
  {
    if (keyState & keyBitRockerLeft)
    {
      PRESS_KEY(cpcKeyLeft);
    }
    else
    {
      RELEASE_KEY(cpcKeyLeft);
    }
  }

  // Rocker Center
  if (keyDiff & centerKeyMask)
  {
    if (keyState & centerKeyMask)
    {
      PRESS_KEY(cpcKeyCenter);
    }
    else
    {
      RELEASE_KEY(cpcKeyCenter);
    }
  }

  //
  // Hard CPC keys
  //
  if (keyDiff & HardCPCKeyCodeMaskA)
  {
  	tUChar keyCode = NativeCPC->HardKeyCPCKeyCodeA >> 3;
  	keyCode <<= 4;
  	keyCode |= NativeCPC->HardKeyCPCKeyCodeA & 7;
  	
    if (keyState & HardCPCKeyCodeMaskA)
    {
      PRESS_KEY(keyCode);
    }
    else
    {
      RELEASE_KEY(keyCode);
    }
  }
  
  if (keyDiff & HardCPCKeyCodeMaskB)
  {
  	tUChar keyCode = NativeCPC->HardKeyCPCKeyCodeB >> 3;
  	keyCode <<= 4;
  	keyCode |= NativeCPC->HardKeyCPCKeyCodeB & 7;
  	
    if (keyState & HardCPCKeyCodeMaskB)
    {
      PRESS_KEY(keyCode);
    }
    else
    {
      RELEASE_KEY(keyCode);
    }
  }

  if (keyDiff & HardCPCKeyCodeMaskC)
  {
  	tUChar keyCode = NativeCPC->HardKeyCPCKeyCodeC >> 3;
  	keyCode <<= 4;
  	keyCode |= NativeCPC->HardKeyCPCKeyCodeC & 7;
  	
    if (keyState & HardCPCKeyCodeMaskC)
    {
      PRESS_KEY(keyCode);
    }
    else
    {
      RELEASE_KEY(keyCode);
    }
  }

  if (keyDiff & HardCPCKeyCodeMaskD)
  {
  	tUChar keyCode = NativeCPC->HardKeyCPCKeyCodeD >> 3;
  	keyCode <<= 4;
  	keyCode |= NativeCPC->HardKeyCPCKeyCodeD & 7;
  	
    if (keyState & HardCPCKeyCodeMaskD)
    {
      PRESS_KEY(keyCode);
    }
    else
    {
      RELEASE_KEY(keyCode);
    }
  }
}
/*----------------------------------------------------------------------------*/


void ReleasePressedKey(void)
/***********************************************************************
 *
 *  ReleasePressedKey
 *
 ***********************************************************************/
{
UInt16 ScreenX;
UInt16 ScreenY;
Boolean PenDown;

  if (usCurrentPressedKey == 0xffff)
    return;

  if (TimGetTicks() < debounceTimeoutTicks)
    return;

  EvtGetPen(&ScreenX,
            &ScreenY,
            &PenDown);

  if (PenDown == false)
  {
    RELEASE_KEY(usCurrentPressedKey); // key has been released

    // Restore SHIFT and CTRL state
    SetKeySHIFTandCTRLState(usCurrentPressedKey);

    usCurrentPressedKey = 0xffff;
  }
}
/*----------------------------------------------------------------------------*/


void InitKeys(void)
/***********************************************************************
 *
 *  InitKeys
 *
 ***********************************************************************/
{
  // Drive panel
  InitKeysPanel(DriveKeys,
                NB_DRIVE_KEYS,
                DriveKeysSettings,
                0);

  // Emulator panel
  InitKeysPanel(EmulatorKeys,
                NB_EMULATORS_KEYS,
                EmulatorKeysSettings,
                0);
  
  // Sub panel
  InitKeysPanel(CPCColouredKeys,
                NB_CPC_COLOURED_KEYS,
                CPCColouredKeysSettings,
                0);
  InitKeysPanel(CPCSpecialKeys,
                NB_CPC_SPECIAL_KEYS,
                CPCSpecialKeysSettings,
                0);

  EmulatorKeysStatus.SoundKeyStatus = KeyReleased;
  EmulatorKeysStatus.FxKeyStatus = KeyReleased;
  EmulatorKeysStatus.JoystickKeyStatus = KeyReleased;
  EmulatorKeysStatus.AutoToggleKeyStatus = KeyReleased;
}
/*----------------------------------------------------------------------------*/


static void InitKeysPanel(tKey* keyP,
                          UInt8 nbKeys,
                          const tKeySetting* settingsP,
                          tUChar DefaultHighlight)
/***********************************************************************
 *
 *  InitKeysPanel
 *
 ***********************************************************************/
{
UInt8 loop;

  for (loop = 0; loop < nbKeys; loop++)
  {
  	UInt8 Keycode;
  	
    keyP[loop].settingP = &settingsP[loop];
    keyP[loop].KeyStatus = KeyReleased;
    keyP[loop].CPCKey = settingsP[loop].CPCKey;
    keyP[loop].OnPushedP = settingsP[loop].OnPushedP;
    keyP[loop].OnReleasedP = settingsP[loop].OnReleasedP;
    keyP[loop].KeyHighlight = DefaultHighlight;
    
    // CPC Keycode
    Keycode = (tUChar)cpc_kbd[keyP[loop].CPCKey];
    keyP[loop].CPCKeycode = ((Keycode >> 4) << 3) | (Keycode & 0xf);
  }
}
/*----------------------------------------------------------------------------*/


Err StartDIAKeyboard(void)
/***********************************************************************
 *
 *  StartDIAKeyboard
 *
 ***********************************************************************/
{
Err Result;

  DIAKeyboardP = (tKeyboard*)MemPtrNew(sizeof(tKeyboard));
  if (DIAKeyboardP == NULL)
  {
#ifndef __RELEASE__
    ErrDisplay("DIAKeyboardP == NULL");
#endif /*__RELEASE__*/

    return memErrNotEnoughSpace;
  }
  
  // Keyboard settings Table
  if (prefP->CPCModel == CPC_MODEL_6128)
  {
    DIAKeyboardP->headerH = DmGetResource(TABLE_ID_TYPE,
                                          TABLE_ID_KEYBOARD_6128);
  }
  else /* CPC_MODEL_464 + CPC_MODEL_664 */
  {
    DIAKeyboardP->headerH = DmGetResource(TABLE_ID_TYPE,
                                          TABLE_ID_KEYBOARD_464);
  }

  Result = StartKeyboard(DIAKeyboardP);
  
  return Result;
}
/*----------------------------------------------------------------------------*/


Err StartMiniKeyboard(void)
/***********************************************************************
 *
 *  StartMiniKeyboard
 *
 ***********************************************************************/
{
Err Result;

  MiniKeyboardP = (tKeyboard*)MemPtrNew(sizeof(tKeyboard));
  if (MiniKeyboardP == NULL)
  {
#ifndef __RELEASE__
    ErrDisplay("MiniKeyboardP == NULL");
#endif /*__RELEASE__*/

    return memErrNotEnoughSpace;
  }
  
  // Keyboard settings Table
  MiniKeyboardP->headerH = DmGetResource(TABLE_ID_TYPE,
                                         TABLE_ID_MINI_KEYBOARD);
  
  Result = StartKeyboard(MiniKeyboardP);
  
  return Result;
}
/*----------------------------------------------------------------------------*/


static Err StartKeyboard(tKeyboard* keyboardP)
/***********************************************************************
 *
 *  StartMiniKeyboard
 *
 ***********************************************************************/
{
UInt8 loop;

  keyboardP->headerP = (tKeyboardHeader*)MemHandleLock(keyboardP->headerH);

#ifndef __RELEASE__
  if (keyboardP->headerP == NULL)
    ErrDisplay("keyboardP->headerP == NULL");
#endif /* __RELEASE__ */

  keyboardP->sectionsP = (tKeySection*)(((tUChar*)keyboardP->headerP) + keyboardP->headerP->HeaderSize);

  keyboardP->keysP = (tKey*)MemPtrNew(keyboardP->headerP->NbTotalKeys * sizeof(tKey));
  if (keyboardP->keysP == NULL)
  {
#ifndef __RELEASE__
    ErrDisplay("keyboardP->keysP == NULL");
#endif /*__RELEASE__*/

    return memErrNotEnoughSpace;
  }

  InitKeysPanel(keyboardP->keysP, 
                keyboardP->headerP->NbTotalKeys, 
                (tKeySetting*)(((tUChar*)keyboardP->sectionsP) + keyboardP->headerP->SectionsSize),
                1);

  // Special keys settings
  for (loop = 0; loop < keyboardP->headerP->NbTotalKeys; loop++)
  {
    switch (keyboardP->keysP[loop].settingP->CPCKey)
    {
      case CPC_KEY_RSHIFT:
      {
        keyboardP->keyRShiftP = &keyboardP->keysP[loop];
        keyboardP->keyRShiftP->CPCKey = CPC_KEY_NONE;
        keyboardP->keyRShiftP->OnPushedP = keyboardP->keyRShiftP->OnReleasedP = SHIFTKeyPressed;
        keyboardP->keyRShiftP->KeyHighlight = 0;
      }
      break;
      
      case CPC_KEY_LSHIFT:
      {
        keyboardP->keyLShiftP = &keyboardP->keysP[loop];
        keyboardP->keyLShiftP->CPCKey = CPC_KEY_NONE;
        keyboardP->keyLShiftP->OnPushedP = keyboardP->keyLShiftP->OnReleasedP = SHIFTKeyPressed;
        keyboardP->keyLShiftP->KeyHighlight = 0;
      }
      break;
      
      case CPC_KEY_CONTROL:
      {
        keyboardP->keyCTRLP = &keyboardP->keysP[loop];
        keyboardP->keyCTRLP->CPCKey = CPC_KEY_NONE;
        keyboardP->keyCTRLP->OnPushedP = keyboardP->keyCTRLP->OnReleasedP = CTRLKeyPressed;
        keyboardP->keyCTRLP->KeyHighlight = 0;
      }
      break;
    }
  }
  
  return errNone;
}
/*----------------------------------------------------------------------------*/


void StopKeyboard(tKeyboard** keyboardPP)
/***********************************************************************
 *
 *  StopKeyboard
 *
 ***********************************************************************/
{
  if (*keyboardPP != NULL)
  {
    if ((*keyboardPP)->headerH != NULL)
    {
      MemHandleUnlock((*keyboardPP)->headerH);
      DmReleaseResource((*keyboardPP)->headerH);
    }

    if ((*keyboardPP)->keysP != NULL)
    {
      MemPtrFree((*keyboardPP)->keysP);
    }
    
    MemPtrFree(*keyboardPP);
    *keyboardPP = NULL;
  }
}
/*----------------------------------------------------------------------------*/


static void UpdatePanelPosition(Coord y_base,
                                UInt8 nbKeys,
                                tKey* keyP)
/***********************************************************************
 *
 *  UpdateKeysPosition
 *
 ***********************************************************************/
{
UInt8 loop;

#ifndef __RELEASE__
  if (keyP == NULL)
    ErrDisplay("keyP == NULL");
#endif /* NOT __RELEASE__ */

  for (loop=nbKeys; loop; loop--, keyP++)
  {
    keyP->Left = keyP->settingP->Left;
    keyP->Top = keyP->settingP->Top + y_base;
    keyP->Right = keyP->Left + keyP->settingP->Width;
    keyP->Bottom = keyP->Top + keyP->settingP->Height;
  }
}



void UpdateKeysPosition(void)
/***********************************************************************
 *
 *  UpdateKeysPosition
 *
 ***********************************************************************/
{
Coord y_base;

  // DIA Keyboard panel
  UpdatePanelPosition(GetYBaseKeyBoardPanel(),
                      DIAKeyboardP->headerP->NbTotalKeys,
                      DIAKeyboardP->keysP);

  // Mini Keyboard panel
  UpdatePanelPosition(GetYBaseEmulatorPanel(),
                      MiniKeyboardP->headerP->NbTotalKeys,
                      MiniKeyboardP->keysP);

  // Fullscreen active, do not manage others panels
  if (FullscreenActive)
    return;

  // Emulator panel
  UpdatePanelPosition(GetYBaseEmulatorPanel(),
                      NB_EMULATORS_KEYS,
                      EmulatorKeys);

  // Sub panel
  y_base = GetYBaseSubPanel();
  UpdatePanelPosition(y_base,
                      NB_DRIVE_KEYS,
                      DriveKeys);
  UpdatePanelPosition(y_base,
                      NB_CPC_COLOURED_KEYS,
                      CPCColouredKeys);
  UpdatePanelPosition(y_base,
                      NB_CPC_SPECIAL_KEYS,
                      CPCSpecialKeys);
}
/*----------------------------------------------------------------------------*/


void ToggleAutoToggle(void)
/***********************************************************************
 *
 *  ToggleAutoToggle
 *
 ***********************************************************************/
{
  u32AutoToggleKeyState = (u32AutoToggleKeyState & keyBitRockerLeft) ? keyBitRockerRight : keyBitRockerLeft;
}
/*----------------------------------------------------------------------------*/


tBool IsShiftPressed(void)
/***********************************************************************
 *
 *  IsShiftPressed
 *
 ***********************************************************************/
{
  if (CPCColouredKeys[KEYINDEX_SHIFT].KeyStatus == KeyPressed)
    return cTrue;
    
  if (ActiveKeyboardP->keyRShiftP->KeyStatus == KeyPressed)
    return cTrue;
    
  if (ActiveKeyboardP->keyLShiftP->KeyStatus == KeyPressed)
    return cTrue;
    
  return cFalse;
}
/*----------------------------------------------------------------------------*/


void RestoreEmulatorKeysStatus(tEmulatorKeysStatus* keyStatusP)
/***********************************************************************
 *
 *  RestoreEmulatorKeysStatus
 *
 ***********************************************************************/
{
  // Pause always inactive to allow screen display !!
  EmulatorKeys[KEYINDEX_PAUSE].KeyStatus = KeyReleased;
  EmulatorUnfreeze();

  // Sound
  EmulatorKeys[KEYINDEX_SOUND].KeyStatus = keyStatusP->SoundKeyStatus;
  if (keyStatusP->SoundKeyStatus == KeyPressed)
  {
    EnableSound();
  }
  else
  {
    DisableSound();
  }

  // Fx
  EmulatorKeys[KEYINDEX_FX].KeyStatus = keyStatusP->FxKeyStatus;
  if (keyStatusP->FxKeyStatus == KeyPressed)
  {
    EnableFonctionKeys();
  }
  else
  {
    DisableFonctionKeys();
  }

  // Joystick
  EmulatorKeys[KEYINDEX_JOYSTICK].KeyStatus = keyStatusP->JoystickKeyStatus;
  if (keyStatusP->JoystickKeyStatus == KeyPressed)
  {
    EnableJoystick();
  }
  else
  {
    DisableJoystick();
  }

  // Auto Toggle
  EmulatorKeys[KEYINDEX_AUTOTOGGLE].KeyStatus = keyStatusP->AutoToggleKeyStatus;
  if (keyStatusP->AutoToggleKeyStatus == KeyPressed)
  {
    AutoToggleKeyPressed();
  }
  else
  {
    AutoToggleKeyReleased();
  }

  HideMessage();

  // Update Display
  if (EmulatorKeysPanelP == EmulatorKeys)
  {
    DisplayEmulatorCommandPanel();
  }
}
/*----------------------------------------------------------------------------*/


static UInt32 DetectScreenDirectionalInput(Coord screenX,
                                           Coord screenY,
                                           UInt8 FullScreen,
                                           UInt8 Landscape,
                                           UInt8 Display320x480)
/***********************************************************************
 *
 *  DetectScreenDirectionalInput
 *
 ***********************************************************************/
{
  //
  // Landscape and Fullscreen Screen
  //
  if (Landscape && Display320x480)
  {
    // Left
    if (screenX < (ONSCREEN_OFFSET_LANDSCAPE_X + OFFSCREEN_OFFSET_X_FS + (SCREEN_CPC_WIDTH_FS / 3)))
    {
      // Left & Up
      if (screenY < (ONSCREEN_OFFSET_LANDSCAPE_Y + (SCREEN_CPC_HEIGHT_LANDSCAPE / 3)))
      {
        return (keyBitRockerLeft | keyBitRockerUp);
      }
      // Left & Down
      else if (screenY >= (ONSCREEN_OFFSET_LANDSCAPE_Y + (SCREEN_CPC_HEIGHT_LANDSCAPE * 2 / 3)))
      {
        return (keyBitRockerLeft | keyBitRockerDown);
      }
      // Center
      else
      {
        return (keyBitRockerLeft);
      }
    }
    // Right
    else if (screenX >= (ONSCREEN_OFFSET_LANDSCAPE_X + OFFSCREEN_OFFSET_X_FS + (SCREEN_CPC_WIDTH_FS * 2 / 3)))
    {
      // Right & Up
      if (screenY < (ONSCREEN_OFFSET_LANDSCAPE_Y + (SCREEN_CPC_HEIGHT_LANDSCAPE / 3)))
      {
        return (keyBitRockerRight | keyBitRockerUp);
      }
      // Right & Down
      else if (screenY >= (ONSCREEN_OFFSET_LANDSCAPE_Y + (SCREEN_CPC_HEIGHT_LANDSCAPE * 2 / 3)))
      {
        return (keyBitRockerRight | keyBitRockerDown);
      }
      // Center
      else
      {
        return (keyBitRockerRight);
      }
    }
    // Center
    else
    {
      // Center & Up
      if (screenY < (ONSCREEN_OFFSET_LANDSCAPE_Y + (SCREEN_CPC_HEIGHT_LANDSCAPE / 3)))
      {
        return (keyBitRockerUp);
      }
      // Center & Down
      else if (screenY >= (ONSCREEN_OFFSET_LANDSCAPE_Y + (SCREEN_CPC_HEIGHT_LANDSCAPE * 2 / 3)))
      {
        return (keyBitRockerDown);
      }
      // Center
      else
      {
        return (centerKeyMask);
      }
    }
  }
  //
  // Normal Screen
  //
  else if (!FullScreen)
  {
    // Left
    if (screenX < (SCREEN_CPC_WIDTH / 3))
    {
      // Left & Up
      if (screenY < (ONSCREEN_OFFSET_Y + (SCREEN_CPC_HEIGHT / 3)))
      {
        return (keyBitRockerLeft | keyBitRockerUp);
      }
      // Left & Down
      else if (screenY >= (ONSCREEN_OFFSET_Y + (SCREEN_CPC_HEIGHT * 2 / 3)))
      {
        return (keyBitRockerLeft | keyBitRockerDown);
      }
      // Center
      else
      {
        return (keyBitRockerLeft);
      }
    }
    // Right
    else if (screenX >= (SCREEN_CPC_WIDTH * 2 / 3))
    {
      // Right & Up
      if (screenY < (ONSCREEN_OFFSET_Y + (SCREEN_CPC_HEIGHT / 3)))
      {
        return (keyBitRockerRight | keyBitRockerUp);
      }
      // Right & Down
      else if (screenY >= (ONSCREEN_OFFSET_Y + (SCREEN_CPC_HEIGHT * 2 / 3)))
      {
        return (keyBitRockerRight | keyBitRockerDown);
      }
      // Center
      else
      {
        return (keyBitRockerRight);
      }
    }
    // Center
    else
    {
      // Center & Up
      if (screenY < (ONSCREEN_OFFSET_Y + (SCREEN_CPC_HEIGHT / 3)))
      {
        return (keyBitRockerUp);
      }
      // Center & Down
      else if (screenY >= (ONSCREEN_OFFSET_Y + (SCREEN_CPC_HEIGHT * 2 / 3)))
      {
        return (keyBitRockerDown);
      }
      // Center
      else
      {
        return (centerKeyMask);
      }
    }
  }
  //
  // Fullscreen
  //
  else
  {
    // Left
    if (screenX < (SCREEN_CPC_WIDTH / 3))
    {
      // Left & Up
      if (screenY < (ONSCREEN_OFFSET_Y_FS + (SCREEN_CPC_HEIGHT_FS / 3)))
      {
        return (keyBitRockerLeft | keyBitRockerUp);
      }
      // Left & Down
      else if (screenY >= (ONSCREEN_OFFSET_Y_FS + (SCREEN_CPC_HEIGHT_FS * 2 / 3)))
      {
        return (keyBitRockerLeft | keyBitRockerDown);
      }
      // Center
      else
      {
        return (keyBitRockerLeft);
      }
    }
    // Right
    else if (screenX >= (SCREEN_CPC_WIDTH * 2 / 3))
    {
      // Right & Up
      if (screenY < (ONSCREEN_OFFSET_Y_FS + (SCREEN_CPC_HEIGHT_FS / 3)))
      {
        return (keyBitRockerRight | keyBitRockerUp);
      }
      // Right & Down
      else if (screenY >= (ONSCREEN_OFFSET_Y_FS + (SCREEN_CPC_HEIGHT_FS * 2 / 3)))
      {
        return (keyBitRockerRight | keyBitRockerDown);
      }
      // Center
      else
      {
        return (keyBitRockerRight);
      }
    }
    // Center
    else
    {
      // Center & Up
      if (screenY < (ONSCREEN_OFFSET_Y_FS + (SCREEN_CPC_HEIGHT_FS / 3)))
      {
        return (keyBitRockerUp);
      }
      // Center & Down
      else if (screenY >= (ONSCREEN_OFFSET_Y_FS + (SCREEN_CPC_HEIGHT_FS * 2 / 3)))
      {
        return (keyBitRockerDown);
      }
      // Center
      else
      {
        return (centerKeyMask);
      }
    }
  }
}
/*----------------------------------------------------------------------------*/


void SwapKeyboardState(tUChar Display320x480)
/***********************************************************************
 *
 *  SwapKeyboardState
 *
 ***********************************************************************/
{
tKeyboard* oldKeyboardP = ActiveKeyboardP;
tUChar KeyWasPressed = 0;

  ActiveKeyboardP = Display320x480 ? DIAKeyboardP : MiniKeyboardP;
	
	if (oldKeyboardP->keyRShiftP->KeyStatus == KeyPressed)
	{
		oldKeyboardP->keyRShiftP->KeyStatus = KeyReleased;
		ActiveKeyboardP->keyRShiftP->KeyStatus = KeyPressed;
		KeyWasPressed=1;
	}
	
	if (oldKeyboardP->keyLShiftP->KeyStatus == KeyPressed)
	{
		oldKeyboardP->keyLShiftP->KeyStatus = KeyReleased;
		ActiveKeyboardP->keyLShiftP->KeyStatus = KeyPressed;
		KeyWasPressed=1;
	}
	
	if (oldKeyboardP->keyCTRLP->KeyStatus == KeyPressed)
	{
		oldKeyboardP->keyCTRLP->KeyStatus = KeyReleased;
		ActiveKeyboardP->keyCTRLP->KeyStatus = KeyPressed;
		KeyWasPressed=1;
	}
	
	if (KeyWasPressed)
	{
		// Display mini keyboard
    MiniKeyboardActive = 1;
	}
}
/*----------------------------------------------------------------------------*/


void UpdateRockerCenterKeyMask(void)
/***********************************************************************
 *
 *  UpdateRockerCenterKeyMask
 *
 ***********************************************************************/
{
  centerKeyMask = keyBitRockerCenter;

	if (prefP->HardKey1Index == HARDKEY_ROCKER_CENTER_INDEX)
	{
		centerKeyMask |= keyBitHard1;
	}
	if (prefP->HardKey2Index == HARDKEY_ROCKER_CENTER_INDEX)
	{
		centerKeyMask |= keyBitHard2;
	}
	if (prefP->HardKey3Index == HARDKEY_ROCKER_CENTER_INDEX)
	{
		centerKeyMask |= keyBitHard3;
	}
	if (prefP->HardKey4Index == HARDKEY_ROCKER_CENTER_INDEX)
	{
		centerKeyMask |= keyBitHard4;
	}
}
/*----------------------------------------------------------------------------*/


void UpdateHardCPCKeyCodeMask(void)
/***********************************************************************
 *
 *  UpdateHardCPCKeyCodeMask
 *
 ***********************************************************************/
{
UInt8 Loop;
tUpdateCPCKeycode UpdateCPCKeycodes[] =
{
  { &HardCPCKeyCodeMaskA, HARDKEY_CPCKEYCODE_A_INDEX },
  { &HardCPCKeyCodeMaskB, HARDKEY_CPCKEYCODE_B_INDEX },
  { &HardCPCKeyCodeMaskC, HARDKEY_CPCKEYCODE_C_INDEX },
  { &HardCPCKeyCodeMaskD, HARDKEY_CPCKEYCODE_D_INDEX }
};

  for (Loop=0; Loop < NUMBER_OF_ITEMS(UpdateCPCKeycodes); Loop++)
  {
    UInt32* maskP = UpdateCPCKeycodes[Loop].KeycodeMaskP;
    UInt32 index = UpdateCPCKeycodes[Loop].HardKeyIndex;
    
    *maskP = 0;

    //
    // CPC keycode
    //
	  if (prefP->HardKey1Index == index)
	  {
		  *maskP |= keyBitHard1;
	  }
	  if (prefP->HardKey2Index == index)
	  {
		  *maskP |= keyBitHard2;
	  }
	  if (prefP->HardKey3Index == index)
	  {
		  *maskP |= keyBitHard3;
	  }
	  if (prefP->HardKey4Index == index)
	  {
		  *maskP |= keyBitHard4;
	  }
  }
}
/*----------------------------------------------------------------------------*/


#if defined(_DEBUG) || defined(_SHOW_KEYS)

tUShort GetCurrentPressedKeys(void)
/***********************************************************************
 *
 *  GetCurrentPressedKeys
 *
 ***********************************************************************/
{
	return usCurrentPressedKey;
}

#endif /* _DEBUG || _SHOW_KEYS */


void KeyboardSetAsciiKeyDown(char key)
/***********************************************************************
 *
 *  KeyboardSetAsciiKeyDown
 *
 ***********************************************************************/
{
tUChar cpc_index;
tUShort cpc_key = KEY_INVALID;

  cpc_index = palm_kbd[(tUChar)key];
  if (cpc_index != KEY_INVALID)
  {
    cpc_key = cpc_kbd[cpc_index];
  }

  if (cpc_key != KEY_INVALID)
  {
    PRESS_KEY(cpc_key);

    // Restore SHIFT and CTRL state
    SetKeySHIFTandCTRLState(cpc_key);
  }
}
/*----------------------------------------------------------------------------*/


void KeyboardSetAsciiKeyUp(char key)
/***********************************************************************
 *
 *  KeyboardSetAsciiKeyUp
 *
 ***********************************************************************/
{
tUChar cpc_index;
tUShort cpc_key = KEY_INVALID;

  cpc_index = palm_kbd[(tUChar)key];
  if (cpc_index != KEY_INVALID)
  {
    cpc_key = cpc_kbd[cpc_index];
  }

  if (cpc_key != KEY_INVALID)
  {
    RELEASE_KEY(cpc_key);

    if (cpc_key & MOD_CPC_SHIFT) // CPC SHIFT key required?
    {
      RELEASE_KEY(CPC_KBD_SHIFT); // make sure key is unSHIFTed
    }

    if (cpc_key & MOD_CPC_CTRL) // CPC CONTROL key required?
    {
      RELEASE_KEY(CPC_KBD_CTRL); // make sure CONTROL key is released
    }
  }
}
/*----------------------------------------------------------------------------*/


static void SetLightPenCoordinates(Coord screenX,
                                   Coord screenY,
                                   Boolean FullScreen)
/***********************************************************************
 *
 *  SetLightPenCoordinates
 *
 ***********************************************************************/
{
tCRTC* CRTC;
tShort offscreenX;
tShort offscreenY;
tUShort addr_offset;
tUChar crtc_R0;
tUChar crtc_R1;
tUChar crtc_R2;
tUChar crtc_R4;
tUChar crtc_R7;
tUChar crtc_hsw;
tUChar crtc_vsw;
tUShort originX;
tUShort originY;
tChar posY;

  // Compute offscreen coordinates
  offscreenX = screenX + prefP->ScreenDisplayOffsetX;
  if (!FullScreen)
  {
    offscreenY = screenY + prefP->ScreenDisplayOffsetY - ONSCREEN_OFFSET_Y;
  }
  else
  {
    offscreenY = screenY - ONSCREEN_OFFSET_Y_FS;
  }
  
  // Get CRTC Structure address
  CRTC = (tCRTC*)EndianSwap32(NativeCPC->CRTC);
#ifndef __RELEASE__
  if (CRTC == NULL)
    ErrDisplay("CRTC == NULL");
#endif /* __RELEASE__ */

  // Get registers
  crtc_R0 = (tUChar)EndianSwap32(CRTC->registers[0]); // Horizontal total character number - 1
  crtc_R1 = (tUChar)EndianSwap32(CRTC->registers[1]); // Horizontal displayed
  crtc_R2 = (tUChar)EndianSwap32(CRTC->registers[2]); // Horizontal Sync Pulse position
  crtc_R4 = (tUChar)EndianSwap32(CRTC->registers[4]); // Vertical total line character number - 1
  crtc_R7 = (tUChar)EndianSwap32(CRTC->registers[7]); // Vertical Sync position
  crtc_hsw = (tUChar)EndianSwap32(CRTC->hsw);         // Horizontal sync width
  crtc_vsw = (tUChar)EndianSwap32(CRTC->vsw);         // Vertical sync width
  
  // Compute display origins
  originX = (crtc_R0+1) - (crtc_R2 + crtc_hsw) - 4 /* ?? */;
  originY = (crtc_R4+1) - (crtc_R7 + 3 /* ?? */ + (crtc_vsw / 8));
  
  // Compute LightPen address
  posY = (offscreenY / 8) - (originY - 1);
  //FC!!posY = (offscreenY / 8) - originY; // Better for Magnum Light Phaser ??
  addr_offset = ((tUChar)(posY >= 0 ? posY : (crtc_R4+1) - posY) * crtc_R1) +
                (offscreenX / 8) - originX;
            
  // Set LightPen offset 
  CRTC->lpen_offset = EndianSwap32(addr_offset);
}
/*----------------------------------------------------------------------------*/


static void SetPhaserCoordinates(Coord screenX,
                                 Coord screenY,
                                 Boolean FullScreen)
/***********************************************************************
 *
 *  SetPhaserCoordinates
 *
 ***********************************************************************/
{
tCRTC* CRTC;
tShort offscreenX;
tShort offscreenY;
tUShort start_address;
tUShort address;
tUChar crtc_R0;
tUChar crtc_R2;
tUChar crtc_R4;
tUChar crtc_R7;
tUChar crtc_hsw;
tUChar crtc_vsw;
tUShort originX;
tUShort originY;

  // Retrieve offscreen coordinates
  offscreenX = screenX + prefP->ScreenDisplayOffsetX;
  if (!FullScreen)
  {
    offscreenY = screenY + prefP->ScreenDisplayOffsetY;
  }
  else
  {
    offscreenY = screenY - ONSCREEN_OFFSET_Y_FS;
  }
  
  // Get CRTC Structure address
  CRTC = (tCRTC*)EndianSwap32(NativeCPC->CRTC);
#ifndef __RELEASE__
  if (CRTC == NULL)
    ErrDisplay("CRTC == NULL");
#endif /* __RELEASE__ */

  // Get registers
  crtc_R0 =  (tUChar)EndianSwap32(CRTC->registers[0]); // Horizontal total character number - 1
  crtc_R2 =  (tUChar)EndianSwap32(CRTC->registers[2]); // Horizontal Sync Pulse position
  crtc_R4 =  (tUChar)EndianSwap32(CRTC->registers[4]); // Vertical total line character number - 1
  crtc_R7 =  (tUChar)EndianSwap32(CRTC->registers[7]); // Vertical Sync position
  crtc_hsw = (tUChar)EndianSwap32(CRTC->hsw);         // Horizontal sync width
  crtc_vsw = (tUChar)EndianSwap32(CRTC->vsw);         // Vertical sync width
  
  // Compute display origins
  originX = (crtc_R0+1) - (crtc_R2 + crtc_hsw);
  originY = (crtc_R4+1) - (crtc_R7 + 3 + (crtc_vsw / 8)); // 3 ??
  
  // Set Phaser coordinates
  NativeCPC->lightgun_x_pos = EndianSwap32((tULong)((offscreenX / 8) - originX));
  NativeCPC->lightgun_y_pos = EndianSwap32((tULong)((offscreenY / 8) - originY));

  /*{
  	char string1[50];
  	char string2[50];
  	StrPrintF(string1, "OffX=%d, OffY=%d", offscreenX, offscreenY);
  	StrPrintF(string2, "OrgX=%d, OrgY=%d", originX, originY);
    FrmCustomAlert(CustomTraceAlert,
                   "SetPhaserCoordinates",
                   string1,
                   string2);
  }*/
}
/*----------------------------------------------------------------------------*/
