/*
    CaPriCe for Palm OS - Amstrad CPC 464/664/6128 emulator for Palm devices
    Copyright (C) 2009  Frdric Coste

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <PalmOS.h>
#include <VFSMgr.h>
#include <ByteOrderUtils.h>

#include "Sections.h"
#include "CaPriCe.h"
#include "Resources.h"
#include "Display.h"
#include "Files.h"
#include "Forms.h"
#include "CPC.h"
#include "Trace.h"
#include "Sound.h"
#include "Prefs.h"
#include "Keyboard.h"
#include "Routines.h"


//===================
// PATCH begin
#ifdef _PATCH_ENABLE

#endif /* _PATCH_ENABLE */
// PATCH end
//===================


const char DrivesList[][8] =
{
  "Drive A",
  "Drive B"
};
#define MAX_DRIVES							NUMBER_OF_ITEMS(DrivesList)

const char DevicesList[][9] =
{
  "CPC 464 ",
  "CPC 664 ",
  "CPC 6128"
};
#define MAX_DEVICES							NUMBER_OF_ITEMS(DevicesList)

const char ScreensList[][11] =
{
  "Color     ",
  "Monochrome"
};
#define MAX_SCREENS							NUMBER_OF_ITEMS(ScreensList)

const char BrandsList[][10] =
{
  "Amstrad  ",
  "Orion    ",
  "Schneider",
  "Awa      ",
  "Solavox  ",
  "Saisho   ",
  "Triumph  ",
  "Isp      "
};
#define MAX_BRANDS							NUMBER_OF_ITEMS(BrandsList)

const char RockerCenterKeysList[][14] =
{
  "CPC Escape   ",
  "CPC Return   ",
  "CPC Copy     ",
  "CPC Control  ",
  "CPC Shift    ",
  "CPC Space    ",
  "CPC Tab      ",
  "CPC CapsLock "
};
#define MAX_ROCKERCENTERKEYS		NUMBER_OF_ITEMS(RockerCenterKeysList)

const char HardKeysList[][14] =
{
  "Not used     ",
  "Rocker Center",
  "Menu         ",
  "Mini-keyboard",
  "Fullscreen   ",
  "Decathlon    ",
  "CPC keycode A",
  "CPC keycode B",
  "CPC keycode C",
  "CPC keycode D"
};
#define MAX_HARDKEYS						NUMBER_OF_ITEMS(HardKeysList)

const char OnScreenInputList[][12] =
{
  "None       ",
  "Palm Rocker",
  "Light Pen  ",
  "Magnum Gun ",
  "GunStick   ",
  "West Phaser"
};
#define MAX_ONSCREENINPUTS			NUMBER_OF_ITEMS(OnScreenInputList)



const char* ExtensionList[] =
{
  ".dsk",
  ".cpc"
};
#define EXTENSION_DRIVEFORM     0
#define EXTENSION_LOADFORM      1


//
// Custom Data structures
//

typedef struct
{
  char** FilenameListP;
  tDIAState DIAState;
  MemHandle DrivesListHandle;
  Int16 FileSelected;
} tFilesFormCustomData;

typedef struct
{
  tDIAState DIAState;
  tPreferences* prefP;
  MemHandle CenterKeyListHandle;
  MemHandle HardKeyListHandle;
} tHardKeysFormCustomData;

typedef struct
{
  tDIAState DIAState;
  tPreferences* prefP;
  MemHandle BrandsListHandle;
  MemHandle DevicesListHandle;
  MemHandle ScreensListHandle;
} tDeviceSettingsFormCustomData;

typedef struct
{
  tDIAState DIAState;
  tNativeCPC* NativeCPC;
  tPreferences* prefP;
  MemHandle OnScreenInputListHandle;
} tEmulatorSettingsFormCustomData;


tDriveEnum DriveSelected;

#define SEARCH_STRING_SIZE      4
static char SearchFilter[SEARCH_STRING_SIZE] = "";



static Int32 GetFieldInteger(FormPtr frmP,
                             UInt16 ResourceID) SECTION_GUI;
static void SetFieldInteger(FormPtr frmP,
                            UInt16 ResourceID,
                            UInt32 sizeMax,
                            Int32 Integer) SECTION_GUI;
static Int16 GetListIndex(FormPtr frmP,
                          UInt16 ResourceID) SECTION_GUI;
static void InitListAndSetIndex(FormPtr frmP,
                                UInt16 ResourceListID,
                                UInt16 ResourceTriggerID,
                                char** listContentPP,
                                Int16 nbItems,
                                Int16 index) SECTION_GUI;

static void FilesFormInit(FormPtr frmP,
                          tFilesFormCustomData* CustomDataP) SECTION_GUI;
static void FilesFormClose(FormPtr frmP,
                           tFilesFormCustomData* CustomDataP) SECTION_GUI;
static Boolean FilesFormLoad(FormPtr frmP,
                             tFilesFormCustomData* CustomDataP) SECTION_GUI;
static Boolean FilesFormSelect(Int16 Selection,
                               tFilesFormCustomData* CustomDataP) SECTION_GUI;

static void FilesFormResize(FormPtr frmP,
                            RectanglePtr beforeP,
                            RectanglePtr afterP) SECTION_GUI;
static void MoveFormObject(FormPtr frmP,
                           UInt16 objectId,
                           Int16 heightDelta) SECTION_GUI;
static void MoveFormGSI(FormPtr frmP,
                        Int16 heightDelta) SECTION_GUI;
static void DisplayFilenameList(FormPtr frmP,
                                char* pathP,
                                tFilesFormCustomData* CustomDataP,
                                UInt16 listID) SECTION_GUI;
static void FreeFilenameList(FormPtr frmP,
                             tFilesFormCustomData* CustomDataP,
                             UInt16 listID) SECTION_GUI;
static void OnDrawFilenameEntryList(Int16 itemNum,
                                    RectangleType* bounds,
                                    char** itemText) SECTION_GUI;
                                    
static void AboutFormInit(FormPtr frmP) SECTION_GUI;

static void DeviceSettingsFormInit(FormPtr frmP,
                                   tDeviceSettingsFormCustomData* CustomDataP) SECTION_GUI;
static void DeviceSettingsFormApply(FormPtr frmP,
                                    tDeviceSettingsFormCustomData* CustomDataP) SECTION_GUI;
static void DeviceSettingsFormClose(FormPtr frmP,
                                    tDeviceSettingsFormCustomData* CustomDataP) SECTION_GUI;

static void EmulatorSettingsFormInit(FormPtr frmP,
                                     tEmulatorSettingsFormCustomData* CustomDataP) SECTION_GUI;
static void EmulatorSettingsFormApply(FormPtr frmP,
                                      tEmulatorSettingsFormCustomData* CustomDataP) SECTION_GUI;
static void EmulatorSettingsFormClose(FormPtr frmP,
                                      tEmulatorSettingsFormCustomData* CustomDataP) SECTION_GUI;

static void ScreenOffsetFormInit(FormPtr frmP) SECTION_GUI;
static void ScreenOffsetFormApply(FormPtr frmP) SECTION_GUI;

static void HardKeysFormInit(FormPtr frmP,
                             tHardKeysFormCustomData* CustomDataP) SECTION_GUI;
static void HardKeysFormApply(FormPtr frmP,
                              tHardKeysFormCustomData* CustomDataP) SECTION_GUI;
static void HardKeysFormClose(FormPtr frmP,
                              tHardKeysFormCustomData* CustomDataP) SECTION_GUI;

static Boolean MainFormMenuHandler(UInt16 itemID);

static void* GetFormCustomData(FormPtr frmP,
                               UInt16 GadgetCustomDataID) SECTION_GUI;



/***********************************************************************
 *
 *  Internal Functions
 *
 ***********************************************************************/

static void* GetFormObjectFromID(FormPtr frmP,
                                 UInt16 ObjectID)
/***********************************************************************
 *
 *  GetFieldInteger
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

  return FrmGetObjectPtr(frmP,
                         FrmGetObjectIndex(frmP,
                                           ObjectID));
}
/*----------------------------------------------------------------------------*/


static void* GetFormCustomData(FormPtr frmP,
                               UInt16 GadgetCustomDataID)
/***********************************************************************
 *
 *  GetFormCustomData
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

  // Get string from field
  return FrmGetGadgetData(frmP,
                          FrmGetObjectIndex(frmP,
                                            GadgetCustomDataID));
}
/*----------------------------------------------------------------------------*/


static void UpdateFormBounds(FormPtr frmP)
/***********************************************************************
 *
 *  UpdateFormBounds
 *
 ***********************************************************************/
{
Coord displayExtX;
Coord displayExtY;
WinHandle frmWinH;
RectangleType formBounds;

#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

  if (collapsibleIA)
  {
    WinGetDisplayExtent(&displayExtX,
                        &displayExtY);
    
    frmWinH = FrmGetWindowHandle(frmP);
#ifndef __RELEASE__
    if (frmWinH == NULL)
      ErrDisplay("frmWinH == NULL");
#endif /* __RELEASE__ */

    WinGetBounds(frmWinH,
                 &formBounds);
    
    formBounds.extent.x = displayExtX - (formBounds.topLeft.x * 2);
    formBounds.extent.y = displayExtY - (formBounds.topLeft.y * 2);
    
    WinSetBounds(frmWinH,
                 &formBounds);
  }
}
/*----------------------------------------------------------------------------*/


static Int32 GetFieldInteger(FormPtr frmP,
                             UInt16 ResourceID)
/***********************************************************************
 *
 *  GetFieldInteger
 *
 ***********************************************************************/
{
char* textP;

  // Get string from field
  textP = FldGetTextPtr((FieldPtr)GetFormObjectFromID(frmP,
                                                      ResourceID));

#ifndef __RELEASE__
    if (textP == NULL)
      ErrDisplay("textP == NULL");
#endif /* __RELEASE__ */

  return StrAToI(textP);
}
/*----------------------------------------------------------------------------*/


static void SetFieldInteger(FormPtr frmP,
                            UInt16 ResourceID,
                            UInt32 sizeMax,
                            Int32 Integer)
/***********************************************************************
 *
 *  SetFieldInteger
 *
 ***********************************************************************/
{
FieldPtr fldP;
MemHandle memH, oldMemH;
MemPtr memP;

  memH = MemHandleNew(sizeMax);

  memP = MemHandleLock(memH);
  StrIToA(memP, Integer);
  MemHandleUnlock(memH);

  fldP = GetFormObjectFromID(frmP,
                             ResourceID);
                                           
  oldMemH = FldGetTextHandle(fldP);
  FldSetTextHandle(fldP, memH);
  
  if (oldMemH != NULL)
  {
    MemHandleFree(oldMemH);
  }
}
/*----------------------------------------------------------------------------*/


static void SetFieldString(FormPtr frmP,
                           UInt16 ResourceID,
                           UInt32 sizeMax,
                           char* String)
/***********************************************************************
 *
 *  SetFieldString
 *
 ***********************************************************************/
{
FieldPtr fldP;
MemHandle memH, oldMemH;
MemPtr memP;

  memH = MemHandleNew(sizeMax);

  memP = MemHandleLock(memH);
  StrCopy(memP, String);
  MemHandleUnlock(memH);

  fldP = GetFormObjectFromID(frmP,
                             ResourceID);
                                           
  oldMemH = FldGetTextHandle(fldP);
  FldSetTextHandle(fldP, memH);
  
  if (oldMemH != NULL)
  {
    MemHandleFree(oldMemH);
  }
}
/*----------------------------------------------------------------------------*/


static Int16 GetControlValue(FormPtr frmP,
                             UInt16 ResourceID)
/***********************************************************************
 *
 *  GetControlValue
 *
 ***********************************************************************/
{
ControlPtr ControlP;

  ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                             ResourceID);
#ifndef __RELEASE__
  if (ControlP == NULL)
    ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

  return (CtlGetValue(ControlP));
}
/*----------------------------------------------------------------------------*/




static Int16 GetListIndex(FormPtr frmP,
                          UInt16 ResourceID)
/***********************************************************************
 *
 *  GetListIndex
 *
 ***********************************************************************/
{
ListPtr ListP;

  ListP = (ListPtr)GetFormObjectFromID(frmP,
                                       ResourceID);
#ifndef __RELEASE__
  if (ListP == NULL)
    ErrDisplay("ListP == NULL");
#endif /* __RELEASE__ */

  return (LstGetSelection(ListP));
}
/*----------------------------------------------------------------------------*/


static void InitListAndSetIndex(FormPtr frmP,
                                UInt16 ResourceListID,
                                UInt16 ResourceTriggerID,
                                char** listContentPP,
                                Int16 nbItems,
                                Int16 index)
/***********************************************************************
 *
 *  InitListAndSetIndex
 *
 ***********************************************************************/
{
ListPtr ListP;

  ListP = (ListPtr)GetFormObjectFromID(frmP,
                                       ResourceListID);
#ifndef __RELEASE__
  if (ListP == NULL)
    ErrDisplay("ListP == NULL");
#endif /* __RELEASE__ */

  LstSetListChoices(ListP,
                    listContentPP,
                    nbItems);
                    
  // Select key
  LstSetSelection(ListP,
                  index);
                  
  // Update trigger
  CtlSetLabel(GetFormObjectFromID(frmP,
                                  ResourceTriggerID),
              LstGetSelectionText(ListP,
                                  index));
}
/*----------------------------------------------------------------------------*/



/***********************************************************************
 *
 *  External Functions
 *
 ***********************************************************************/

void SetDIAPolicy(FormPtr frmP,
                  UInt16 DIATriggerState,
                  UInt16 ExtentDisplayLibRef,
                  tUChar Display320x480)
/***********************************************************************
 *
 *  SetDIAPolicy
 *
 ***********************************************************************/
{
UInt16 RequestedState;

#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

  if (!collapsibleIA)
    return;

  WinSetConstraintsSize(FrmGetWindowHandle(frmP),
                        160,
                        240,
                        240,
                        160,
                        240,
                        240);

  FrmSetDIAPolicyAttr(frmP,
                      frmDIAPolicyCustom);
                      
  PINSetInputTriggerState(DIATriggerState);
    

  if (!Display320x480)
  {
    // Display Status Bar
    StatShow();
  }
    
#ifdef __RELEASE__
  if (!ExtentDisplayLibRef)
  {
    // Landscape mode disabled
    SysSetOrientation(sysOrientationPortrait);
    SysSetOrientationTriggerState(sysOrientationTriggerDisabled);
  }
  else
#endif /* __RELEASE__ */
  {
    // Landscape mode enabled
    SysSetOrientationTriggerState(sysOrientationTriggerEnabled);
  }

  // DIA Display    
  RequestedState = Display320x480 ? pinInputAreaClosed : pinInputAreaOpen;
  PINSetInputAreaState(RequestedState); // Invoke winDisplayChangedEvent message posting
}
/*----------------------------------------------------------------------------*/


void SaveDIAState(tDIAState* DIAStateP)
/***********************************************************************
 *
 *  SaveDIAState
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (DIAStateP == NULL)
    ErrDisplay("DIAStateP == NULL");
#endif /* __RELEASE__ */

  if (!collapsibleIA)
    return;

  DIAStateP->InputAreaState = PINGetInputAreaState();
  DIAStateP->InputTriggerState = PINGetInputTriggerState();
  
  DIAStateP->Orientation = SysGetOrientation();
  DIAStateP->OrientationTriggerState = SysGetOrientationTriggerState();
  
  StatGetAttribute(statAttrBarVisible,
                   &DIAStateP->StatAttributes);
}
/*----------------------------------------------------------------------------*/


void RestoreDIAState(const tDIAState* DIAStateP)
/***********************************************************************
 *
 *  RestoreDIAState
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (DIAStateP == NULL)
    ErrDisplay("DIAStateP == NULL");
#endif /* __RELEASE__ */

  if (!collapsibleIA)
    return;
    
  if (DIAStateP->StatAttributes) // Bar is visible
  {
    StatShow();
  }
  else // Bar is hidden
  {
    StatHide();
  }

  PINSetInputTriggerState(DIAStateP->InputTriggerState);
  PINSetInputAreaState(DIAStateP->InputAreaState);

  SysSetOrientationTriggerState(DIAStateP->OrientationTriggerState);
  SysSetOrientation(DIAStateP->Orientation);
}
/*----------------------------------------------------------------------------*/


Boolean MainFormHandleEvent(EventPtr eventP)
/***********************************************************************
 *
 *  MainFormHandleEvent
 *
 ***********************************************************************/
#undef SAVE_SESSION_AT_START
//#define SAVE_SESSION_AT_START
{
Boolean handled = false;
FormPtr frmP;
UInt32 Ticks;
Err Result;
UInt16 oldCoordSys;

  if (CPCStarted)
  {
    if (CPCHandleEvent(eventP) == true)
      return true;
  }

  switch (eventP->eType)
  {
    //
    // menuOpenEvent
    //
    case menuOpenEvent:
#if defined(_DEBUG)
      MenuShowItem(9990);
      MenuShowItem(9999);
#endif /* _DEBUG */
#if defined(_DEBUG)
      MenuShowItem(9997);
      MenuShowItem(9996);
#endif /* _DEBUG */
#if defined(_TRACE)
      MenuShowItem(9990);
      MenuShowItem(9998);
#endif /* _TRACE */
      break;

    //
    // menuEvent
    //
    case menuEvent:
      handled = MainFormMenuHandler(eventP->data.menu.itemID);
      break;

    //
    // frmOpenEvent
    //
    case frmOpenEvent:
    {
      frmP = FrmGetActiveForm();

#ifndef __RELEASE__
      if (frmP == NULL)
        ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

      SetDIAPolicy(frmP,
                   pinInputTriggerEnabled,
                   DexLibRefNum,
                   prefP->Display320x480);

      UpdateFormBounds(frmP);
      FrmDrawForm(frmP);

      // First form opening : CPC not started !!
      if (!CPCStarted)
      {
        if (WinGetDrawWindow() == FrmGetWindowHandle(frmP))
        {
          oldCoordSys = WinSetCoordinateSystem(kCoordinatesNative);

          // Always start in portrait mode
          if (collapsibleIA)
          {
            SysSetOrientation(sysOrientationPortrait);
          }

          //
          // Afficher le panneau de controle pour la premire fois
          //
          DisplayEmulatorCommandPanel();
          DisplaySubPanel(SubPanelActive);

          if (prefP->Display320x480)
          {
            DisplayDIAKeyboard(prefP->CPCModel);
          }

          DisplayStartupScreen();

          // Begin display duration
          Ticks = TimGetTicks() + (SysTicksPerSecond() * SHOWSPLASHTIME_S) + 1;
        
          // Wait for splash screen minimum duration.
          while (TimGetTicks() < Ticks);

          if (CPCFirstStart() == errNone)
          {
            if (CPCColdReset(DEFAULT_CAPRICE_PATH,
                             LASTSESSION_FILENAME,
                             CmdToGoParamsP ? NoLoad : LoadWithConfirmation) == errNone)
            {
              SystemHalt = 0;
              FrmUpdateForm(MainForm,
                            emulatorRedrawUpdateCode | SubPanelRedrawUpdateCode); // Update last session buttons state
              UpdateKeysPosition();
            }
          }
        
#ifdef SAVE_SESSION_AT_START
          SaveCPC(DEFAULT_CAPRICE_PATH,
                  "Start.cpc",
                  contextP);
#endif /* SAVE_SESSION_AT_START */

          // Delete Last session file
          DeleteFile(DEFAULT_CAPRICE_PATH, LASTSESSION_FILENAME);

          WinSetCoordinateSystem(oldCoordSys);
        }
      }
      else
      {
        FrmUpdateForm(MainForm,
                      frmRedrawUpdateCode);

        if (EmulatorKeys[KEYINDEX_PAUSE].KeyStatus == KeyReleased)
        {
          EmulatorUnfreeze();
        }
      }
    }
    break;

    //
    // frmCloseEvent
    //
    case frmCloseEvent:
    {
      EmulatorFreeze();
    }
    break;

    //
    // frmUpdateEvent
    //
    case frmUpdateEvent:
    {
    	// Freeze screen
    	WinScreenLock(winLockCopy);

      frmP = FrmGetActiveForm();
#ifndef __RELEASE__
      if (frmP == NULL)
        ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */
    	
      if (eventP->data.frmUpdate.updateCode & frmRedrawUpdateCode)
      {
        FrmDrawForm(frmP);
      }
      
      oldCoordSys = WinSetCoordinateSystem(kCoordinatesNative);

      if (WinGetDrawWindow() == FrmGetWindowHandle(frmP))
      {
			  // Screen display
        if (eventP->data.frmUpdate.updateCode & (frmRedrawUpdateCode | screenRedrawUpdateCode))
        {
          if (FullscreenActive || LandscapeActive)
          {
            ClearFullscreen();
          }

          PrepareLandscapeScreenCopy(NativeCPC,
                                     DexLibRefNum);

          DisplayCPCScreen(prefP->ScreenDisplayOffsetX,
                           prefP->ScreenDisplayOffsetY);
        }

        if (!LandscapeActive)
        {
			    // Sub screen display
          if (!FullscreenActive)
          {
      	    if ( (!MiniKeyboardActive) || (prefP->Display320x480) )
      	    {
      		    // Command panel display
        	    if (eventP->data.frmUpdate.updateCode & (frmRedrawUpdateCode | emulatorRedrawUpdateCode))
        	    {
          	    DisplayEmulatorCommandPanel();
        	    }

					    // Sub panel display
        	    if (eventP->data.frmUpdate.updateCode & (frmRedrawUpdateCode | SubPanelRedrawUpdateCode))
        	    {
          	    DisplaySubPanel(SubPanelActive);
        	    }
      	    }
      	    else
      	    {
      		    // Mini Keyboard display
        	    if (eventP->data.frmUpdate.updateCode & (frmRedrawUpdateCode | keyboardRedrawUpdateCode))
        	    {
        		    DisplayMiniKeyboard();
        	    }
      	    }
          }
      
          // DIA Keyboard display
          if ( (prefP->Display320x480) &&
               (eventP->data.frmUpdate.updateCode & (frmRedrawUpdateCode | keyboardRedrawUpdateCode)) )
          {
            DisplayDIAKeyboard(prefP->CPCModel);
          }
        }
      }
  		
  		UpdateFDCLedOffScreenDisplay();

      WinSetCoordinateSystem(oldCoordSys);
      
      // Unfreeze screen
      WinScreenUnlock();

      handled = true;
    }
    break;

    //
    // winDisplayChangedEvent
    //
    case winDisplayChangedEvent:
    {
      LandscapeActive = SysGetOrientation() == sysOrientationPortrait ? 0 : 1;
      if (LandscapeActive)
      {
      	FullscreenActive = 1; // Fullscreen automatically activated while in landscape mode.
      }

      UpdateFormBounds(FrmGetActiveForm());
      prefP->Display320x480 = PINGetInputAreaState() == pinInputAreaClosed ? 1 : 0;
      UpdateKeysPosition();
      SwapKeyboardState(prefP->Display320x480);

      FrmUpdateForm(MainForm,
                    frmRedrawUpdateCode);
    }
    break;

    //
    // winEnterEvent
    //
    case winEnterEvent:
    {
      // For menu enter/exit, event is thrown to the main form event handler
      if (eventP->data.winEnter.enterWindow == FrmGetWindowHandle(FrmGetFormPtr(MainForm)))
      {
        if ( (CPCStarted) && (EmulatorKeys[KEYINDEX_PAUSE].KeyStatus == KeyReleased) )
        {
          // Dmarrer l'mulation
          EmulatorUnfreeze();
        }
      }
    }
    break;

    //
    // winExitEvent
    //
    case winExitEvent:
    {
      // Arrter l'mulation
      EmulatorFreeze();
    }
    break;
  }

  return handled;
}
/*----------------------------------------------------------------------------*/


static Boolean MainFormMenuHandler(UInt16 itemID)
/***********************************************************************
 *
 *  MainFormMenuHandler
 *
 ***********************************************************************/
{
Boolean handled = false;
tDrive* driveP;
UInt16 result;
FormPtr frmP;


  switch (itemID)
  {
  	//
  	// File
  	//
    case FileLoadSessionMenu:
    {
      FrmGotoForm(LoadForm);
      handled = true;
    }
    break;

    case FileSaveSessionMenu:
    {
    	SaveSession(NativeCPC,
    	            contextP,
    	            sessionFilenameP);
      handled = true;
    }
    break;

    case FileLoadEjectDiskMenu:
    {
      // Go to Drive form
      DriveSelected = noDrive;
      FrmGotoForm(DriveForm);
      handled = true;
    }
    break;

    case FileCreateDiskMenu:
    {
      result = FrmAlert(FormatDriveAlert);
      if (result)
      {
        driveP = result == 1 ? NativeCPC->DriveA : NativeCPC->DriveB;

        SaveAndEjectDiskImage(prefP->FilesPathname,
                              driveP,
                              NativeCPC);

        if (CreateDisk(driveP,
                       DEFAULT_DISK_FORMAT,
                       NativeCPC) == errNone)
        {
          SetDriveFilename(driveP, NEW_DISK_FILENAME);

          if (SaveDiskImage(prefP->FilesPathname,
                            driveP,
                            NativeCPC) != errNone)
          {
            FrmCustomAlert(BadCreateDiskAlert,
                           NEW_DISK_FILENAME,
                           "",
                           "");
            EjectDisk(driveP,
                      NativeCPC);
          }
        }
        else
        {
          EjectDisk(driveP,
                    NativeCPC);
        }

        FrmUpdateForm(MainForm,
                      SubPanelRedrawUpdateCode);
      }
      handled = true;
    }
    break;

    case FileSwapDiskMenu:
    {
      SwapDrive(NativeCPC);
      handled = true;
    }
    break;

    case FileScreenshotMenu:
    {
    	ScreenshotRequested = 1;
      EmulatorUnfreeze(); // Start emulator until video frame is complete.
      handled = true;
    }
    break;

    case FileExitMenu:
    {
      // Terminate application
      EventType newEvent;
      newEvent.eType = appStopEvent;
      EvtAddEventToQueue(&newEvent);
    }
    break;

		//
		// Play
		//
    case PlayPauseMenu:
    {
      if (!SystemHalt)
      {
        EmulatorFreeze();
      }
      else
      {
        EmulatorUnfreeze();
      }
      // Update Pause key display
      EmulatorKeys[KEYINDEX_PAUSE].KeyStatus = SystemHalt ? KeyPressed : KeyReleased;
      FrmUpdateForm(MainForm,
                    emulatorRedrawUpdateCode);
      handled = true;
    }
    break;

    case PlayResetMenu:
    {
      CPCResetWithConfirmation();
      handled = true;
    }
    break;

		case PlayToggleJoystickMenu:
		{
			if (EmulatorKeys[KEYINDEX_JOYSTICK].KeyStatus == KeyReleased)
			{
				EmulatorKeys[KEYINDEX_JOYSTICK].KeyStatus = KeyPressed;
				EnableJoystick();
			}
			else
			{
				EmulatorKeys[KEYINDEX_JOYSTICK].KeyStatus = KeyReleased;
				DisableJoystick();
			}
      FrmUpdateForm(MainForm,
                    emulatorRedrawUpdateCode);
      handled = true;
		}
		break;

		case PlayToggleSoundMenu:
		{
			if (!prefP->SoundEnabled)
			{
				EnableSound();
			}
			else
			{
				DisableSound();
			}
			EmulatorKeys[KEYINDEX_SOUND].KeyStatus = prefP->SoundEnabled ? KeyPressed : KeyReleased;
      FrmUpdateForm(MainForm,
                    emulatorRedrawUpdateCode);
      handled = true;
		}
		break;

		case PlayToggleDecathlonMenu:
		{
			if (EmulatorKeys[KEYINDEX_AUTOTOGGLE].KeyStatus == KeyReleased)
			{
				EmulatorKeys[KEYINDEX_AUTOTOGGLE].KeyStatus = KeyPressed;
				AutoToggleKeyPressed();
			}
			else
			{
				EmulatorKeys[KEYINDEX_AUTOTOGGLE].KeyStatus = KeyReleased;
				AutoToggleKeyReleased();
			}
      FrmUpdateForm(MainForm,
                    emulatorRedrawUpdateCode);
      handled = true;
		}
		break;

    case PlayAutostartMenu:
    {
      // Get command to execute
      DiskAutostart(NativeCPC,
        		        NativeCPC->DriveA,
        		        AutoStartCommand);

      handled = true;
    }
    break;

		//
		// Display
		//
    case DisplayMiniKeyboardMenu:
    {
      MiniKeyboardActive ^= 1;  // Invert Mini Keyboard state

  		ActiveKeyboardP = MiniKeyboardActive ? MiniKeyboardP : DIAKeyboardP;

      // Update form
      FrmUpdateForm(MainForm, 
                    emulatorRedrawUpdateCode | SubPanelRedrawUpdateCode | keyboardRedrawUpdateCode);

      handled = true;
    }
    break;

    case DisplayCPCDrivesMenu:
    {
      SubPanelActive = 0;
      MiniKeyboardActive = 0;
      ActiveKeyboardP = DIAKeyboardP;
      EmulatorKeys[KEYINDEX_SUBPANEL].KeyStatus = KeyReleased;
      FrmUpdateForm(MainForm,
                    SubPanelRedrawUpdateCode | emulatorRedrawUpdateCode);
      handled = true;
    }
    break;

    case DisplayCPCColouredKeyboardMenu:
    case DisplayCPCSpecialKeyboardMenu:
    {
      if (itemID == DisplayCPCColouredKeyboardMenu)
      {
        SelectCPCColouredKeyboard();
      }
      else
      {
        SelectCPCSpecialKeyboard();
      }
      SubPanelActive = 1;
      MiniKeyboardActive = 0;
      ActiveKeyboardP = DIAKeyboardP;
      EmulatorKeys[KEYINDEX_SUBPANEL].KeyStatus = KeyPressed;
      FrmUpdateForm(MainForm,
                    SubPanelRedrawUpdateCode | emulatorRedrawUpdateCode);
      handled = true;
    }
    break;

    case DisplayFullscreenMenu:
    {
      if (!LandscapeActive)
      {
        FullscreenActive ^= 1;  // Invert fullscreen state

        // Update form
        FrmUpdateForm(MainForm,
                      frmRedrawUpdateCode);
        UpdateKeysPosition();
      }

      handled = true;
    }
    break;

    case DisplayOffsetMenu:
    {
      tDIAState oldDIAStateP;
    	
      // Create Save Form
      frmP = FrmInitForm(ScreenOffsetForm);
#ifndef __RELEASE__
      if (frmP == NULL)
        ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */
      
      // Set event handler
      FrmSetEventHandler(frmP,
                         ScreenOffsetFormHandleEvent);

      // Ouvrir la boite de dialogue
      SaveDIAState(&oldDIAStateP);
      SetDIAPolicy(frmP,
                   pinInputTriggerDisabled,
                   DexLibRefNum,
                   0 /*pinInputAreaOpen*/);
      ScreenOffsetFormInit(frmP);
      FrmDoDialog(frmP);

      // Free resources
      FrmDeleteForm(frmP);
      frmP = FrmGetActiveForm();
      FrmSetEventHandler(frmP,
                         MainFormHandleEvent);
      RestoreDIAState(&oldDIAStateP);

      handled = true;
    }
    break;

    case DisplayOnScreenRockerHelpMenu:
    {
      OnScreenRockerHelpDisplay ^= 1;
      handled = true;
    }
    break;

    case DisplayKeycodesMenu:
    {
      SetShowCPCKeycodes(GetShowCPCKeycodes() ^ 1);
      MiniKeyboardActive = 1;  // Display Mini Keyboard state
  		ActiveKeyboardP = MiniKeyboardP;

      // Update form
      FrmUpdateForm(MainForm, 
                    emulatorRedrawUpdateCode | SubPanelRedrawUpdateCode | keyboardRedrawUpdateCode);

      handled = true;
    }
    break;

		//
		// Settings
		//
    case SettingsDeviceMenu:
    {
      FrmGotoForm(DeviceSettingsForm);
      handled = true;
    }
    break;

    case SettingsEmulatorMenu:
    {
      FrmGotoForm(EmulatorSettingsForm);
      handled = true;
    }
    break;

    case SettingsHardKeysMenu:
    {
      FrmGotoForm(HardKeysForm);
      handled = true;
    }
    break;

    case SettingsNightModeMenu:
    {
    	UInt32 Loop;
    	
    	do
    	{
    	  if (prefP->NightModeActive == 0)
    	  {
          RGBColorType* newPaletteP = (RGBColorType*)MemPtrNew(sizeof(RGBColorType) * 256);
          if (newPaletteP == NULL)
          {
            FrmAlert(NotEnoughMemoryAlert);
            continue;
          }
          for (Loop=0; Loop < 256; Loop++)
          {
            UInt32 Value = (originalPaletteP[Loop].r + originalPaletteP[Loop].g + originalPaletteP[Loop].b) / 3;

    	      newPaletteP[Loop].b = 0;
  	        newPaletteP[Loop].g = 0;
  	        newPaletteP[Loop].r = (UInt8)Value;
          }
          WinPalette(winPaletteSet, 0, 256, newPaletteP);
          MemPtrFree(newPaletteP);
    	  }
    	  else
    	  {
          WinPalette(winPaletteSet, 0, 256, originalPaletteP);
    	  }
    	
    	  prefP->NightModeActive ^= -1; // Invert state
    	  
    	  if (NativeCPC)
    	  {
    	    CPCSetColor();
    	  }
    	}
    	while (0);
    	
      handled = true;
    }
    break;

    case SettingsAboutMenu:
    {
      FrmGotoForm(AboutForm);
      handled = true;
    }
    break;

#ifdef _TRACE
    case 9998:
    {
      NativeCPC->TraceInstruction ^= 1;
      NativeCPC->TraceDisplay = NativeCPC->TraceInstruction;

      handled = true;
    }
    break;
#endif /* _TRACE */

#ifdef _DEBUG
    case 9996:
    {
      tZ80* Z80 = (tZ80*)EndianSwap32((tULong)NativeCPC->Z80);
      Z80->Regs.breakpoint = EndianSwap32(TRACE_BREAKPOINT);

      handled = true;
    }
    break;
#endif /* _DEBUG */
  }

  return handled;
}
/*----------------------------------------------------------------------------*/


Boolean FilesFormHandleEvent(EventPtr eventP)
/***********************************************************************
 *
 *  FilesFormHandleEvent
 *
 ***********************************************************************/
{
Boolean handled = false;
FormPtr frmP;
tFilesFormCustomData* CustomDataP;
ListPtr ListP;
char* textP;
Int16 NbIndex;
Int16 Index;
UInt16 size;
MemHandle memH;
MemPtr memP;
UInt16 result;

  frmP = FrmGetActiveForm();
#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

  CustomDataP = (tFilesFormCustomData*)GetFormCustomData(frmP,
                                                         FilesFormCustomData);

  switch (eventP->eType)
  {
    case ctlSelectEvent:
    {
      switch (eventP->data.ctlSelect.controlID)
      {
        //
        // CANCEL button
        //
        case FilesFormCancelButton:
        {
          FrmGotoForm(MainForm);
          handled = true;
        }
        break;

        //
        // EJECT button
        //
        case FilesFormEjectButton:
        {
          SaveAndEjectDiskImage(prefP->FilesPathname,
                                DriveSelected == DriveA ? NativeCPC->DriveA : NativeCPC->DriveB,
                                NativeCPC);
          FrmGotoForm(MainForm);
          handled = true;
        }
        break;

        //
        // LOAD button
        //
        case FilesFormLoadButton:
        {
#ifndef __RELEASE__
          if (CustomDataP == NULL)
            ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

          if (FilesFormLoad(frmP,
                            CustomDataP) == true)
          {
            // Save search filter string
            StrNCopy(SearchFilter,
                     CustomDataP->FilenameListP[CustomDataP->FileSelected],
                     SEARCH_STRING_SIZE-1);

            FrmGotoForm(MainForm);
          }
          
          handled = true;
        }
        break;

        //
        // Delete button
        //
        case FilesFormDeleteButton:
        {
#ifndef __RELEASE__
          if (CustomDataP == NULL)
            ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

          result = FrmCustomAlert(ConfirmDeleteDiskFileAlert,
                                  CustomDataP->FilenameListP[CustomDataP->FileSelected],
                                  "",
                                  "");
          if (result)
          {
            textP = FldGetTextPtr((FieldPtr)GetFormObjectFromID(frmP,
                                                                FilesFormPathField));
#ifndef __RELEASE__
          if (textP == NULL)
            ErrDisplay("textP == NULL");
#endif /* __RELEASE__ */

            if (DeleteFile(textP,
                           CustomDataP->FilenameListP[CustomDataP->FileSelected]) != errNone)
            {
            	FrmAlert(BadDeleteDiskFileAlert);
            }

            // Refresh content
            WinScreenLock(winLockCopy);
            FrmHideObject(frmP,
                          FrmGetObjectIndex(frmP,
                                            FilesFormDeleteButton));
            FrmHideObject(frmP,
                          FrmGetObjectIndex(frmP,
                                            FilesFormLoadButton));
            FilesFormClose(frmP,
                           CustomDataP);
            FilesFormInit(frmP,
                          CustomDataP);
                          
            FrmUpdateForm(FrmGetActiveFormID(),
                          frmRedrawUpdateCode);
            WinScreenUnlock();
          }
          handled = true;
        }
        break;

        //
        // SEARCH button
        //
        case FilesFormSearchButton:
        {
#ifndef __RELEASE__
          if (CustomDataP == NULL)
            ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

          // Get text to search
          textP = FldGetTextPtr((FieldPtr)GetFormObjectFromID(frmP,
                                                              FilesFormSearchField));
#ifndef __RELEASE__
          if (textP == NULL)
            ErrDisplay("textP == NULL");
#endif /* __RELEASE__ */

          // Look into filename table to find name begining by the search string
          ListP = (ListPtr)GetFormObjectFromID(frmP,
                                               FilesFormFilesList);
#ifndef __RELEASE__
          if (ListP == NULL)
            ErrDisplay("ListP == NULL");
#endif /* __RELEASE__ */

          NbIndex = LstGetNumberOfItems(ListP);
          size = StrLen(textP);
          for (Index=0; Index<NbIndex; Index++)
          {
            if (!StrNCaselessCompare(textP, 
                                     CustomDataP->FilenameListP[Index],
                                     size))
            {
              // Set the first item found at the top of the visible list
              LstSetTopItem(ListP, Index);
              LstDrawList(ListP);
              break;
            }
          }
          handled = true;
        }
        break;
      }
    }
    break;


    case popSelectEvent:
    {
      Int16 DriveEmpty;

#ifndef __RELEASE__
      if (CustomDataP == NULL)
        ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

      // Drive A
      if (eventP->data.popSelect.selection == 0)
      {
        DriveSelected = DriveA;
        DriveEmpty = IsDriveFilenameExist(NativeCPC->DriveA) == cFalse ? 0 : 1;
      }
      // Drive B
      else
      {
        DriveSelected = DriveB;
        DriveEmpty = IsDriveFilenameExist(NativeCPC->DriveB) == cFalse ? 0 : 1;
      }

      if (DriveEmpty != 0)
      {
        FrmShowObject(frmP,
                      FrmGetObjectIndex(frmP,
                                        FilesFormEjectButton));
      }
      else
      {
        FrmHideObject(frmP,
                      FrmGetObjectIndex(frmP,
                                        FilesFormEjectButton));
      }

      if (CustomDataP->FileSelected != noListSelection)
      {
        FrmShowObject(frmP,
                      FrmGetObjectIndex(frmP,
                                        FilesFormLoadButton));
        FrmShowObject(frmP,
                      FrmGetObjectIndex(frmP,
                                        FilesFormDeleteButton));
      }

      // Leave handled to false to allow trigger display update
      //handled = false;
    }
    break;

    case lstSelectEvent:
    {
      switch (eventP->data.lstSelect.listID)
      {
        case FilesFormFilesList:
        {
#ifndef __RELEASE__
          if (CustomDataP == NULL)
            ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

          FilesFormSelect(eventP->data.lstSelect.selection,
                          CustomDataP);
        }
        break;
      }
    }
    break;

    case frmOpenEvent:
    {
      CustomDataP = (tFilesFormCustomData*)MemPtrNew(sizeof(tFilesFormCustomData));
      if (CustomDataP == NULL)
      {
        FrmAlert(NotEnoughMemoryAlert);
        FrmGotoForm(MainForm);
        break;
      }
      
      FrmSetGadgetData(frmP,
                       FrmGetObjectIndex(frmP,
                                         FilesFormCustomData),
                       CustomDataP);
        
      SaveDIAState(&CustomDataP->DIAState);
      CustomDataP->FilenameListP = NULL;

      SetDIAPolicy(frmP,
                   pinInputTriggerEnabled,
                   DexLibRefNum,
                   0 /*pinInputAreaOpen*/);
      FilesFormInit(frmP,
                    CustomDataP);
      FrmDrawForm(frmP);
    }
    break;


    case frmCloseEvent:
    {
      frmP = FrmGetFormPtr(eventP->data.frmClose.formID);
#ifndef __RELEASE__
      if (frmP == NULL)
        ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

      FilesFormClose(frmP,
                     CustomDataP);
      
      if (CustomDataP)
      {
        RestoreDIAState(&CustomDataP->DIAState);
        
        FrmSetGadgetData(frmP,
                         FrmGetObjectIndex(frmP,
                                           FilesFormCustomData),
                         NULL);
      	
        MemPtrFree(CustomDataP);
      }
    }
    break;


    case winDisplayChangedEvent:
    {
      RectangleType FormBoundsBefore;
      RectangleType FormBoundsAfter;
      
    	// Freeze screen
    	WinScreenLock(winLockCopy);

      FrmGetFormBounds(frmP,
                       &FormBoundsBefore);
      UpdateFormBounds(frmP);
      FrmGetFormBounds(frmP,
                       &FormBoundsAfter);
                       
      if (FormBoundsBefore.extent.y != FormBoundsAfter.extent.y)
      {
        if (Navigation5WaysActive)
        {    	
          // Remove navigation focus to avoid bad drawing while resizing
          FrmNavRemoveFocusRing(frmP);
        }
        
        FilesFormResize(frmP,
                        &FormBoundsBefore,
                        &FormBoundsAfter);
        
        FrmUpdateForm(FrmGetActiveFormID(),
                      frmRedrawUpdateCode);
      }
                      
      // Unfreeze screen
      WinScreenUnlock();
    }
    break;

    case volumeMountedEvent:
    case volumeUnmountedEvent:
    {
#ifndef __RELEASE__
      if (CustomDataP == NULL)
        ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

      FreeFilenameList(frmP,
                       CustomDataP,
                       FilesFormFilesList);
      DisplayFilenameList(frmP,
                          prefP->FilesPathname,
                          CustomDataP,
                          FilesFormFilesList);
                          
      FrmUpdateForm(FrmGetActiveFormID(),
                    frmRedrawUpdateCode);
    }
    break;
  }

  return handled;
}
/*----------------------------------------------------------------------------*/

static void FilesFormInit(FormPtr frmP,
                          tFilesFormCustomData* CustomDataP)
/***********************************************************************
 *
 *  FilesFormInit
 *
 ***********************************************************************/
{
ListPtr ListP;
Int16 DriveEmpty;
UInt16 frmID = FrmGetActiveFormID();
char* pathP;

#ifndef __RELEASE__
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

  CustomDataP->FileSelected = noListSelection;
  
  //
  // Add Drives into Drive List
  //
  if (frmID == DriveForm)
  {
    CustomDataP->DrivesListHandle = SysFormPointerArrayToStrings((char*)DrivesList,
                                                                 MAX_DRIVES);
    ListP = (ListPtr)GetFormObjectFromID(frmP,
                                         FilesFormDrivesList);
#ifndef __RELEASE__
    if (ListP == NULL)
      ErrDisplay("ListP == NULL");
#endif /* __RELEASE__ */

    LstSetListChoices(ListP,
                      (char**)MemHandleLock(CustomDataP->DrivesListHandle),
                      MAX_DRIVES);
    
    if (DriveSelected == DriveA)
    {
      LstSetSelection(ListP, 0);
      CtlSetLabel(GetFormObjectFromID(frmP,
                                      FilesFormDriveTrigger),
                  LstGetSelectionText(ListP, 0));
      DriveEmpty = IsDriveFilenameExist(NativeCPC->DriveA) == cFalse ? 0 : 1;
    }
    else if (DriveSelected == DriveB)
    {
      LstSetSelection(ListP, 1);
      CtlSetLabel(GetFormObjectFromID(frmP,
                                      FilesFormDriveTrigger),
                  LstGetSelectionText(ListP, 1));
      DriveEmpty = IsDriveFilenameExist(NativeCPC->DriveB) == cFalse ? 0 : 1;
    }
    
    if ((DriveSelected != noDrive) && (DriveEmpty != 0) )
    {
      FrmShowObject(frmP,
                    FrmGetObjectIndex(frmP,
                                      FilesFormEjectButton));
    }
  }

  //
  // Show disk images path
  //
  if (frmID == DriveForm)
  {
    pathP = prefP->FilesPathname;
    if (IsDirectoryExist(pathP) == cFalse)
    {
    	StrCopy(prefP->FilesPathname, DEFAULT_DISK_PATH);
    }
  }
  else
  {
    pathP = DEFAULT_CAPRICE_PATH;
  }
  
  FldSetTextPtr((FieldPtr)GetFormObjectFromID(frmP,
                                              FilesFormPathField),
                pathP);


  //
  // Display folder content
  //
  ListP = GetFormObjectFromID(frmP,
                              FilesFormFilesList);
  DisplayFilenameList(frmP,
                      pathP,
                      CustomDataP,
                      FilesFormFilesList);
  LstSetDrawFunction(ListP,
                     OnDrawFilenameEntryList);


  //
  // Display search filter string
  //
  SetFieldString(frmP,
                 FilesFormSearchField,
                 SEARCH_STRING_SIZE,
                 SearchFilter);
}
/*----------------------------------------------------------------------------*/
static void FilesFormClose(FormPtr frmP,
                           tFilesFormCustomData* CustomDataP)
/***********************************************************************
 *
 *  FilesFormClose
 *
 ***********************************************************************/
{
UInt16 frmID = FrmGetActiveFormID();

#ifndef __RELEASE__
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

  FldSetTextPtr((FieldPtr)GetFormObjectFromID(frmP,
                                              FilesFormPathField),
                NULL);
  
  FreeFilenameList(frmP,
                   CustomDataP,
                   FilesFormFilesList);

  if (frmID == DriveForm)
  {
#ifndef __RELEASE__
    if (CustomDataP->DrivesListHandle == NULL)
      ErrDisplay("CustomDataP->DrivesListHandle == NULL");
#endif /* __RELEASE__ */

    MemHandleUnlock(CustomDataP->DrivesListHandle);
    MemHandleFree(CustomDataP->DrivesListHandle);
  }
}
/*----------------------------------------------------------------------------*/
static Boolean FilesFormLoad(FormPtr frmP,
                             tFilesFormCustomData* CustomDataP)
/***********************************************************************
 *
 *  FilesFormLoad
 *
 ***********************************************************************/
{
Boolean handled = false;

#ifndef __RELEASE__
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

  // Changer le titre du bouton
  CtlSetLabel(GetFormObjectFromID(frmP,
                                  FilesFormLoadButton),
              "Loading...");
        
  switch (FrmGetActiveFormID())
  {
    case DriveForm:
    {
      // Vrifier la taille du nom du fichier
      if (StrLen(CustomDataP->FilenameListP[CustomDataP->FileSelected]) >= SIZETAB_FILENAME)
      {
        FrmAlert(FilenameTooLongAlert);
      }
      else
      {
        // Sauvegarder le disque actuel
        tDrive* driveP = DriveSelected == DriveA ? NativeCPC->DriveA : NativeCPC->DriveB;
          
        // Charger le nouveau disque
        SetDriveFilename(driveP,
                         CustomDataP->FilenameListP[CustomDataP->FileSelected]);

        if (LoadDiskImage(prefP->FilesPathname,
                          driveP,
                          NativeCPC) != errNone)
        {
          // Problmes pendant le chargement
          EjectDisk(driveP,
                    NativeCPC);
        }
        else
        {
        	// Autostart enable and load on Drive A
        	if ( prefP->AutoStartEnable &&
        	     (DriveSelected == DriveA) )
        	{
#ifdef _TRACE
            tAutoStartReturnCode ReturnCode = 
#endif /* _TRACE */        		
        		// Get command to execute
        		DiskAutostart(NativeCPC,
        		              driveP,
        		              AutoStartCommand);
        		              
#ifdef _TRACE
            if (ReturnCode != AutoStartOK)
            {
            	TRACE_SHOW_INT("DiskAutostart\nReturnCode=", ReturnCode);
            }
#endif /* _TRACE */        		
        		              
        	}
        }
        
        handled = true;
      }
    }
    break;

    case LoadForm:
    {
      CPCColdReset(DEFAULT_CAPRICE_PATH,
                   CustomDataP->FilenameListP[CustomDataP->FileSelected],
                   AutomaticLoad);

      handled = true;
    }
    break;
  }
  
  return handled;
}
/*----------------------------------------------------------------------------*/
static Boolean FilesFormSelect(Int16 Selection,
                               tFilesFormCustomData* CustomDataP)
/***********************************************************************
 *
 *  FilesFormSelect
 *
 ***********************************************************************/
{
Boolean handled = false;
FormPtr frmP;
char* textP;

#ifndef __RELEASE__
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

  frmP = FrmGetActiveForm();
#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

  // Directory selection ??
  textP = CustomDataP->FilenameListP[Selection];
  if (textP[0] == '/')
  {
  	// Hide controls
    FrmHideObject(frmP,
                  FrmGetObjectIndex(frmP,
                                    FilesFormLoadButton));
    FrmHideObject(frmP,
                  FrmGetObjectIndex(frmP,
                                    FilesFormDeleteButton));
  	
    switch (FrmGetActiveFormID())
    {
      case DriveForm:
      {
        CustomDataP->FileSelected = noListSelection;

        // Go to parent
        if (!StrCompareAscii(textP, PARENT_PATH))
        {
          textP = &prefP->FilesPathname[StrLen(prefP->FilesPathname)-2];
          while (*textP != '/') textP--;
          *(textP+1) = 0;
        }
        else // Go to sub directory
        {
          StrCat(prefP->FilesPathname, textP+1);
          StrCat(prefP->FilesPathname, "/");
        }

        prefP->PreferencesChanged = 1; // Save new Images path

        // Refresh content
        FilesFormClose(frmP,
                       CustomDataP);
        FilesFormInit(frmP,
                      CustomDataP);

        FrmUpdateForm(FrmGetActiveFormID(),
                      frmRedrawUpdateCode);
      }
      break;

      case LoadForm:
      {
        CustomDataP->FileSelected = noListSelection;
      }
      break;
    }
  }
  // File selection
  else
  {
    CustomDataP->FileSelected = Selection;

    FrmShowObject(frmP,
                  FrmGetObjectIndex(frmP,
                                    FilesFormDeleteButton));
                                        
    switch (FrmGetActiveFormID())
    {
      case DriveForm:
      {
        if (DriveSelected != noDrive)
        {
          FrmShowObject(frmP,
                        FrmGetObjectIndex(frmP,
                                          FilesFormLoadButton));
        }
        else
        {
          FrmSetFocus(frmP,
                      FrmGetObjectIndex(frmP,
                                        FilesFormDriveTrigger));
          FrmUpdateForm(FrmGetActiveFormID(),
                        frmRedrawUpdateCode);
        }
      }
      break;
      
      case LoadForm:
      {
        FrmShowObject(frmP,
                      FrmGetObjectIndex(frmP,
                                        FilesFormLoadButton));
      }
      break;
    }
  }
  
  return handled;
}
/*----------------------------------------------------------------------------*/
static void FilesFormResize(FormPtr frmP,
                            RectanglePtr beforeP,
                            RectanglePtr afterP)
/***********************************************************************
 *
 *  FilesFormResize
 *
 ***********************************************************************/
{
RectangleType objectBounds;
UInt16 frmID = FrmGetFormId(frmP);
UInt16 objIndex;
Int16 heightDelta;
Int16 nbItems;
ListPtr listP;
Int16 topItem;
Int16 nbVisibleItems;
Int16 focusedItem;
Boolean IsNavSupported;

  // Determine the amount of change
  heightDelta = (afterP->extent.y - afterP->topLeft.y) - 
                (beforeP->extent.y - beforeP->topLeft.y);
                
  //
  // Resize objects
  //           
  // Files list
  objIndex = FrmGetObjectIndex(frmP,
                               FilesFormFilesList);
#ifndef __RELEASE__
  if (objIndex == NULL)
    ErrDisplay("objIndex == NULL");
#endif /* __RELEASE__ */
  listP = FrmGetObjectPtr(frmP,
                          objIndex);
#ifndef __RELEASE__
  if (listP == NULL)
    ErrDisplay("listP == NULL");
#endif /* __RELEASE__ */
  FrmGetObjectBounds(frmP,
                     objIndex,
                     &objectBounds);
  nbVisibleItems = heightDelta / FntCharHeight();
  objectBounds.extent.y += nbVisibleItems * FntCharHeight();
  FrmSetObjectBounds(frmP,
                     objIndex,
                     &objectBounds);
  //
  nbVisibleItems = objectBounds.extent.y / FntCharHeight();
  topItem = LstGetTopItem(listP);
  nbItems = LstGetNumberOfItems(listP);
  if (nbItems > nbVisibleItems)
  {
    if ((nbItems - topItem) < nbVisibleItems)
    {
      LstSetTopItem(listP, nbItems - nbVisibleItems);
    }
  }
  else
  {
  	if (nbItems)
  	{
      LstSetTopItem(listP, 0);
  	}
  }
  

  //
  // Move objects
  //  
  // Graffiti Shift Location
  MoveFormGSI(frmP,
              heightDelta);
  // "Search" button
  MoveFormObject(frmP,
                 FilesFormSearchButton,
                 heightDelta);
  // "Search" field
  MoveFormObject(frmP,
                 FilesFormSearchField,
                 heightDelta);
  // "Cancel" button
  MoveFormObject(frmP,
                 FilesFormCancelButton,
                 heightDelta);
  
  if ( (frmID == DriveForm) || (frmID == LoadForm) )
  {
    // "LOAD" button
    MoveFormObject(frmP,
                   FilesFormLoadButton,
                   heightDelta);
  }

  if (frmID == DriveForm)
  {
    // "Drive" trigger
    MoveFormObject(frmP,
                   FilesFormDriveTrigger,
                   heightDelta);
    // "Drive" list
    MoveFormObject(frmP,
                   FilesFormDrivesList,
                   heightDelta);
    // "Eject" button
    MoveFormObject(frmP,
                   FilesFormEjectButton,
                   heightDelta);
  }

  // "Delete" button
  MoveFormObject(frmP,
                 FilesFormDeleteButton,
                 heightDelta);
}
/*----------------------------------------------------------------------------*/
static void MoveFormObject(FormPtr frmP,
                           UInt16 objectId,
                           Int16 heightDelta)
/***********************************************************************
 *
 *  MoveFormObject
 *
 ***********************************************************************/
{
Coord x, y;

  FrmGetObjectPosition(frmP,
                       FrmGetObjectIndex(frmP,
                                         objectId),
                       &x,
                       &y);
  y += heightDelta;
  FrmSetObjectPosition(frmP,
                       FrmGetObjectIndex(frmP,
                                         objectId),
                       x,
                       y);
}
/*----------------------------------------------------------------------------*/
static void MoveFormGSI(FormPtr frmP,
                        Int16 heightDelta)
/***********************************************************************
 *
 *  MoveFormGSI
 *
 ***********************************************************************/
{
UInt16 numObjectsOnForm;
UInt16 index;
Coord x, y;

  // The GSI has no objectID, but it is typically towards the end of the index list
  numObjectsOnForm = FrmGetNumberOfObjects(frmP);
  for (index = numObjectsOnForm - 1; index >= 0; index--)
  {
    if (frmGraffitiStateObj == FrmGetObjectType(frmP,
                                                index))
    {
      FrmGetObjectPosition(frmP,
                           index,
                           &x,
                           &y);
      y += heightDelta;
      FrmSetObjectPosition(frmP,
                           index,
                           x,
                           y);
      break;
    }
  }
}
/*----------------------------------------------------------------------------*/


static void DisplayFilenameList(FormPtr frmP,
                                char* pathP,
                                tFilesFormCustomData* CustomDataP,
                                UInt16 listID)
/***********************************************************************
 *
 *  DisplayFilenameList
 *
 ***********************************************************************/
{
ListPtr ListP;
UInt16 NbFiles;
UInt16 frmID = FrmGetActiveFormID();
const char* extensionP = 0;
tCPCFolderContentOptions Options;

#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

  // Select extensionP
  if (frmID == DriveForm)
  {
    extensionP = ExtensionList[EXTENSION_DRIVEFORM];
    Options = Show_Folder;
  }
  else if (frmID == LoadForm)
  {
    extensionP = ExtensionList[EXTENSION_LOADFORM];
    Options = No_Option;
  }

  NbFiles = RetreiveFolderContent(pathP,
                                  &CustomDataP->FilenameListP,
                                  extensionP,
                                  Options);
  
  ListP = GetFormObjectFromID(frmP,
                              listID);
#ifndef __RELEASE__
  if (ListP == NULL)
    ErrDisplay("ListP == NULL");
#endif /* __RELEASE__ */

  LstSetListChoices(ListP,
                    CustomDataP->FilenameListP,
                    NbFiles);
  LstSetSelection(ListP,
                  noListSelection);
}
/*----------------------------------------------------------------------------*/


static void FreeFilenameList(FormPtr frmP, 
                             tFilesFormCustomData* CustomDataP,
                             UInt16 listID)
/***********************************************************************
 *
 *  FreeFilenameList
 *
 ***********************************************************************/
{
ListPtr listP;
UInt16 Loop;
Int16 itemCount;

#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

  listP = (ListPtr)GetFormObjectFromID(frmP,
                                       listID);
#ifndef __RELEASE__
  if (listP == NULL)
    ErrDisplay("ListP == NULL");
#endif /* __RELEASE__ */

  itemCount = LstGetNumberOfItems(listP);

  for (Loop=0; Loop < itemCount; Loop++)
  {
#ifndef __RELEASE__
    if (CustomDataP->FilenameListP[Loop] == NULL)
      ErrDisplay("CustomDataP->FilenameListP[Loop] == NULL");
#endif /* __RELEASE__ */
    MemPtrFree(CustomDataP->FilenameListP[Loop]);
  }

  MemPtrFree(CustomDataP->FilenameListP);
  CustomDataP->FilenameListP = NULL;
}
/*----------------------------------------------------------------------------*/


static void OnDrawFilenameEntryList(Int16 itemNum,
                                    RectangleType* bounds,
                                    char** itemText)
/***********************************************************************
 *
 *  OnDrawFilenameEntryList
 *
 ***********************************************************************/
{
IndexedColorType oldColor;
IndexedColorType SelectColor;
IndexedColorType ItemColor;

  //
  // Determine if display of selected item !!
  //
  // Retreive foreground color
  ItemColor = WinSetForeColor(0);
  WinSetForeColor(ItemColor);
  // Retreive system selected foreground color.
  SelectColor = UIColorGetTableEntryIndex(UIObjectSelectedForeground);

  // Not diplaying selected item
  if (ItemColor != SelectColor)
  {
    // Change Text color
    oldColor = WinSetTextColor(colorIndexP[*itemText[itemNum] == '/' ? COLORINDEX_RED : COLORINDEX_DARK_BLUE]);
  }

  WinDrawTruncChars(itemText[itemNum],
                    StrLen(itemText[itemNum]),
                    bounds->topLeft.x,
                    bounds->topLeft.y,
                    bounds->extent.x);

  // Not displaying selected item
  if (ItemColor != SelectColor)
  {
  // Restore old text color
    WinSetTextColor(oldColor);
  }
}
/*----------------------------------------------------------------------------*/



Boolean AboutFormHandleEvent(EventPtr eventP)
/***********************************************************************
 *
 *  AboutFormHandleEvent
 *
 ***********************************************************************/
{
Boolean handled = false;
FormPtr frmP;

  switch (eventP->eType)
  {
    case ctlSelectEvent:
    {
      switch (eventP->data.ctlSelect.controlID)
      {
        case AboutFormOKButton:
        {
          FrmGotoForm(MainForm);
          handled = true;
        }
        break;
      }
    }
    break;

    case frmOpenEvent:
    {
      frmP = FrmGetActiveForm();
#ifndef __RELEASE__
      if (frmP == NULL)
        ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

      SetDIAPolicy(frmP,
                   pinInputTriggerDisabled,
                   DexLibRefNum,
                   0 /*pinInputAreaOpen*/);
      AboutFormInit(frmP);
      FrmDrawForm(frmP);
      //handled = false; //remember to leave handled as false
    }
    break;
  }

  return handled;
}
/*----------------------------------------------------------------------------*/

static void AboutFormInit(FormPtr frmP)
/***********************************************************************
 *
 *  AboutFormInit
 *
 ***********************************************************************/
{
MemHandle resourceH;
MemPtr resourceP;
char stringA[30];

  StrCopy(stringA,
          "Version ");

  // Get Version string resource and add it to string
  resourceH = DmGetResource(verRsc,
                            1000);
  resourceP = MemHandleLock(resourceH);
  StrCat(stringA, resourceP);
  MemHandleUnlock(resourceH);
  DmReleaseResource(resourceH);

  // Update version label
  FrmCopyLabel(frmP,
               AboutFormVersionLabel,
               stringA);
}
/*----------------------------------------------------------------------------*/



Boolean DeviceSettingsFormHandleEvent(EventPtr eventP)
/***********************************************************************
 *
 *  DeviceSettingsFormHandleEvent
 *
 ***********************************************************************/
{
Boolean handled = false;
FormPtr frmP;
tDeviceSettingsFormCustomData* CustomDataP;

  frmP = FrmGetActiveForm();
#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

  CustomDataP = (tDeviceSettingsFormCustomData*)GetFormCustomData(frmP,
                                                                  DeviceSettingsFormCustomData);

  switch (eventP->eType)
  {
    case ctlSelectEvent:
    {
      switch (eventP->data.ctlSelect.controlID)
      {
        case DeviceSettingsFormApplyButton:
        {
#ifndef __RELEASE__
          if (CustomDataP == NULL)
            ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

          DeviceSettingsFormApply(frmP,
                                  CustomDataP);
        }
        // Then do all as for Cancel

        case DeviceSettingsFormCancelButton:
        {
          DeviceSettingsFormClose(frmP,
                                  CustomDataP);

          if (CustomDataP)
          {
            RestoreDIAState(&CustomDataP->DIAState);
            
            FrmSetGadgetData(frmP,
                             FrmGetObjectIndex(frmP,
                                               DeviceSettingsFormCustomData),
                             NULL);
      	
            MemPtrFree(CustomDataP);
          }

          FrmGotoForm(MainForm);

          handled = true;
        }
        break;
      }
    }
    break;

    case frmOpenEvent:
    {
      CustomDataP = (tDeviceSettingsFormCustomData*)MemPtrNew(sizeof(tDeviceSettingsFormCustomData));
      if (CustomDataP == NULL)
      {
        FrmAlert(NotEnoughMemoryAlert);
        FrmGotoForm(MainForm);
        //handled = false; //remember to leave handled as false
        break;
      }
      
      FrmSetGadgetData(frmP,
                       FrmGetObjectIndex(frmP,
                                         DeviceSettingsFormCustomData),
                       CustomDataP);
      
      // Initialize custom data
      SaveDIAState(&CustomDataP->DIAState);
      CustomDataP->prefP = prefP;

      SetDIAPolicy(frmP,
                   pinInputTriggerDisabled,
                   DexLibRefNum,
                   0 /*pinInputAreaOpen*/);
                
      DeviceSettingsFormInit(frmP,
                             CustomDataP);
      
      FrmDrawForm(frmP);
      //handled = false; //remember to leave handled as false
    }
    break;
  }

  return handled;
}
/*----------------------------------------------------------------------------*/

static void DeviceSettingsFormInit(FormPtr frmP,
                                   tDeviceSettingsFormCustomData* CustomDataP)
/***********************************************************************
 *
 *  DeviceSettingsFormInit
 *
 ***********************************************************************/
{
ListPtr ListP;
ControlPtr ControlP;
Int16 Index;

#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

  //
  // Brand
  //
  CustomDataP->BrandsListHandle = SysFormPointerArrayToStrings((char*)BrandsList,
                                                               MAX_BRANDS);
#ifndef __RELEASE__
  if (CustomDataP->BrandsListHandle == NULL)
    ErrDisplay("CustomDataP->BrandsListHandle == NULL");
#endif /* __RELEASE__ */

  ListP = (ListPtr)GetFormObjectFromID(frmP,
                                       DeviceSettingsFormBrandList);
#ifndef __RELEASE__
  if (ListP == NULL)
    ErrDisplay("ListP == NULL");
#endif /* __RELEASE__ */
  LstSetListChoices(ListP,
                    (char**)MemHandleLock(CustomDataP->BrandsListHandle),
                    MAX_BRANDS);
  // Select actual Brand
  switch (CustomDataP->prefP->CPCJumpers & CPC_BRAND_MASK)
  {
    case CPC_JUMPER_ORION:     Index = 1; break;
    case CPC_JUMPER_SCHNEIDER: Index = 2; break;
    case CPC_JUMPER_AWA:       Index = 3; break;
    case CPC_JUMPER_SOLAVOX:   Index = 4; break;
    case CPC_JUMPER_SAISHO:    Index = 5; break;
    case CPC_JUMPER_TRIUMPH:   Index = 6; break;
    case CPC_JUMPER_ISP:       Index = 7; break;
    default:
      Index = 0;
  }
  LstSetSelection(ListP, Index);
  CtlSetLabel(GetFormObjectFromID(frmP,
                                  DeviceSettingsFormBrandTrigger),
              LstGetSelectionText(ListP, Index));

  //
  // Device
  //
  CustomDataP->DevicesListHandle = SysFormPointerArrayToStrings((char*)DevicesList,
                                                                MAX_DEVICES);
#ifndef __RELEASE__
  if (CustomDataP->DevicesListHandle == NULL)
    ErrDisplay("CustomDataP->DevicesListHandle == NULL");
#endif /* __RELEASE__ */

  ListP = (ListPtr)GetFormObjectFromID(frmP,
                                       DeviceSettingsFormDeviceList);
#ifndef __RELEASE__
  if (ListP == NULL)
    ErrDisplay("ListP == NULL");
#endif /* __RELEASE__ */

  LstSetListChoices(ListP,
                    (char**)MemHandleLock(CustomDataP->DevicesListHandle),
                    MAX_DEVICES);
  // Select actual Device
  LstSetSelection(ListP,
                  CustomDataP->prefP->CPCModel);
  CtlSetLabel(GetFormObjectFromID(frmP,
                                  DeviceSettingsFormDeviceTrigger),
              LstGetSelectionText(ListP,
                                  CustomDataP->prefP->CPCModel));

  //
  // Screen
  //
  CustomDataP->ScreensListHandle = SysFormPointerArrayToStrings((char*)ScreensList,
                                                                MAX_SCREENS);
#ifndef __RELEASE__
  if (CustomDataP->ScreensListHandle == NULL)
    ErrDisplay("CustomDataP->ScreensListHandle == NULL");
#endif /* __RELEASE__ */

  ListP = (ListPtr)GetFormObjectFromID(frmP,
                                       DeviceSettingsFormScreenList);
#ifndef __RELEASE__
  if (ListP == NULL)
    ErrDisplay("ListP == NULL");
#endif /* __RELEASE__ */

  LstSetListChoices(ListP,
                    (char**)MemHandleLock(CustomDataP->ScreensListHandle),
                    MAX_SCREENS);
                    
  // Select actual Screen Type
  LstSetSelection(ListP,
                  CustomDataP->prefP->ScreenType);
  CtlSetLabel(GetFormObjectFromID(frmP,
                                  DeviceSettingsFormScreenTrigger),
              LstGetSelectionText(ListP,
                                  CustomDataP->prefP->ScreenType));

  //
  // Screen intensity
  //
  // Intensity slider
  ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                             DeviceSettingsFormIntensitySlider);
#ifndef __RELEASE__
  if (ControlP == NULL)
    ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

  CtlSetValue(ControlP,
              CustomDataP->prefP->ScreenIntensity);

  //
  // Use PARADOS
  //
  ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                             DeviceSettingsFormUseParadosCheck);
#ifndef __RELEASE__
  if (ControlP == NULL)
    ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

  CtlSetValue(ControlP,
              CustomDataP->prefP->UseParados);

  //
  // Use 64k Memory Extension
  //
  ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                             DeviceSettingsFormUse64kMemExtCheck);
#ifndef __RELEASE__
  if (ControlP == NULL)
    ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

  CtlSetValue(ControlP,
              CustomDataP->prefP->Use64kMemoryExtension);

  //
  // Antialiased mode 2
  //
  ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                             DeviceSettingsFormMode2Antialiased);
#ifndef __RELEASE__
  if (ControlP == NULL)
    ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

  CtlSetValue(ControlP,
              CustomDataP->prefP->Mode2AntiAliasing);
}
/*----------------------------------------------------------------------------*/

static void DeviceSettingsFormApply(FormPtr frmP,
                                    tDeviceSettingsFormCustomData* CustomDataP)
/***********************************************************************
 *
 *  DeviceSettingsFormApply
 *
 ***********************************************************************/
{
ListPtr ListP;
ControlPtr ControlP;
Int16 Index;

#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

  // Changer le titre du bouton
  CtlSetLabel(GetFormObjectFromID(frmP, 
                                  DeviceSettingsFormApplyButton),
              "Applying...");

  // Get Brand Selection
  ListP = (ListPtr)GetFormObjectFromID(frmP,
                                       DeviceSettingsFormBrandList);
#ifndef __RELEASE__
  if (ListP == NULL)
    ErrDisplay("ListP == NULL");
#endif /* __RELEASE__ */

  Index = LstGetSelection(ListP);
  CustomDataP->prefP->CPCJumpers = CPC_JUMPER_50HZ;
  switch (Index)
  {
    case 1: CustomDataP->prefP->CPCJumpers |= CPC_JUMPER_ORION; break;
    case 2: CustomDataP->prefP->CPCJumpers |= CPC_JUMPER_SCHNEIDER; break;
    case 3: CustomDataP->prefP->CPCJumpers |= CPC_JUMPER_AWA; break;
    case 4: CustomDataP->prefP->CPCJumpers |= CPC_JUMPER_SOLAVOX; break;
    case 5: CustomDataP->prefP->CPCJumpers |= CPC_JUMPER_SAISHO; break;
    case 6: CustomDataP->prefP->CPCJumpers |= CPC_JUMPER_TRIUMPH; break;
    case 7: CustomDataP->prefP->CPCJumpers |= CPC_JUMPER_ISP; break;
    default:
      CustomDataP->prefP->CPCJumpers |= CPC_JUMPER_AMSTRAD;
  }

  // Get Device selection
  ListP = (ListPtr)GetFormObjectFromID(frmP,
                                       DeviceSettingsFormDeviceList);
#ifndef __RELEASE__
  if (ListP == NULL)
    ErrDisplay("ListP == NULL");
#endif /* __RELEASE__ */

  CustomDataP->prefP->CPCModel = LstGetSelection(ListP);

  // Get Screen Type selection
  ListP = (ListPtr)GetFormObjectFromID(frmP,
                                       DeviceSettingsFormScreenList);
#ifndef __RELEASE__
  if (ListP == NULL)
    ErrDisplay("ListP == NULL");
#endif /* __RELEASE__ */

  CustomDataP->prefP->ScreenType = LstGetSelection(ListP);


  // Get Screen Intensity
  CustomDataP->prefP->ScreenIntensity = (tUChar)GetControlValue(frmP,
                                                                DeviceSettingsFormIntensitySlider);

  // Get Use PARADOS
  CustomDataP->prefP->UseParados = (tUChar)GetControlValue(frmP,
                                                           DeviceSettingsFormUseParadosCheck);

  // Get Use 64k Memory Extension
  CustomDataP->prefP->Use64kMemoryExtension = (tUChar)GetControlValue(frmP,
                                                                      DeviceSettingsFormUse64kMemExtCheck);

  // Get Antialiased mode 2
  CustomDataP->prefP->Mode2AntiAliasing = (tUChar)GetControlValue(frmP,
                                                                  DeviceSettingsFormMode2Antialiased);

  // Ask for Preferences save
  CustomDataP->prefP->PreferencesChanged = 1;

  // Perform Emulator reset
  CPCColdReset(NULL,
               NULL,
               NoLoad);
}
/*----------------------------------------------------------------------------*/

static void DeviceSettingsFormClose(FormPtr frmP,
                                    tDeviceSettingsFormCustomData* CustomDataP)
/***********************************************************************
 *
 *  DeviceSettingsFormClose
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */


#ifndef __RELEASE__
  if (CustomDataP->BrandsListHandle == NULL)
    ErrDisplay("CustomDataP->BrandsListHandle == NULL");
#endif /* __RELEASE__ */

  MemHandleUnlock(CustomDataP->BrandsListHandle);
  MemHandleFree(CustomDataP->BrandsListHandle);

#ifndef __RELEASE__
  if (CustomDataP->DevicesListHandle == NULL)
    ErrDisplay("CustomDataP->DevicesListHandle == NULL");
#endif /* __RELEASE__ */

  MemHandleUnlock(CustomDataP->DevicesListHandle);
  MemHandleFree(CustomDataP->DevicesListHandle);

#ifndef __RELEASE__
  if (CustomDataP->ScreensListHandle == NULL)
    ErrDisplay("CustomDataP->ScreensListHandle == NULL");
#endif /* __RELEASE__ */

  MemHandleUnlock(CustomDataP->ScreensListHandle);
  MemHandleFree(CustomDataP->ScreensListHandle);
}
/*----------------------------------------------------------------------------*/



Boolean EmulatorSettingsFormHandleEvent(EventPtr eventP)
/***********************************************************************
 *
 *  EmulatorSettingsFormHandleEvent
 *
 ***********************************************************************/
{
Boolean handled = false;
FormPtr frmP;
tEmulatorSettingsFormCustomData* CustomDataP;

  frmP = FrmGetActiveForm();
#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

  CustomDataP = (tEmulatorSettingsFormCustomData*)GetFormCustomData(frmP,
                                                                    EmulatorSettingsFormCustomData);

  switch (eventP->eType)
  {
    case ctlSelectEvent:
    {
      switch (eventP->data.ctlSelect.controlID)
      {
        case EmulatorSettingsFormApplyButton:
        {
#ifndef __RELEASE__
          if (CustomDataP == NULL)
            ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

          EmulatorSettingsFormApply(frmP,
                                    CustomDataP);
          
          // Apply new rendering parameters  
          SetRenderingParameters(prefP->Rendering);
          SetLightGunParameters(prefP,
                                NativeCPC);
          
        }
        // Then do all as for Cancel

        case EmulatorSettingsFormCancelButton:
        {
#ifndef __RELEASE__
          if (CustomDataP == NULL)
            ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

          EmulatorSettingsFormClose(frmP,
                                    CustomDataP);

          if (CustomDataP)
          {
            RestoreDIAState(&CustomDataP->DIAState);
            
            FrmSetGadgetData(frmP,
                             FrmGetObjectIndex(frmP,
                                               EmulatorSettingsFormCustomData),
                             NULL);
      	
            MemPtrFree(CustomDataP);
          }
          
          FrmGotoForm(MainForm);
          
          handled = true;
        }
        break;

       }
    }
    break;

    case frmOpenEvent:
    {
      CustomDataP = (tEmulatorSettingsFormCustomData*)MemPtrNew(sizeof(tEmulatorSettingsFormCustomData));
      if (CustomDataP == NULL)
      {
        FrmAlert(NotEnoughMemoryAlert);
        FrmGotoForm(MainForm);
        //handled = false; //remember to leave handled as false
        break;
      }
      
      FrmSetGadgetData(frmP,
                       FrmGetObjectIndex(frmP,
                                         EmulatorSettingsFormCustomData),
                       CustomDataP);
      
      // Initialize custom data
      SaveDIAState(&CustomDataP->DIAState);
      CustomDataP->prefP = prefP;
      CustomDataP->NativeCPC = NativeCPC;

      SetDIAPolicy(frmP,
                   pinInputTriggerDisabled,
                   DexLibRefNum,
                   0 /*pinInputAreaOpen*/);
                   
      EmulatorSettingsFormInit(frmP,
                               CustomDataP);

      FrmDrawForm(frmP);
      //handled = false; //remember to leave handled as false
    }
    break;
  }

  return handled;
}
/*----------------------------------------------------------------------------*/

static void EmulatorSettingsFormInit(FormPtr frmP,
                                     tEmulatorSettingsFormCustomData* CustomDataP)
/***********************************************************************
 *
 *  EmulatorSettingsFormInit
 *
 ***********************************************************************/
{
ControlPtr ControlP;
ListPtr ListP;
UInt16 Index;

#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

  //
  // Audio
  //
  // "Games" system volume checkbox
  ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                             EmulatorSettingsFormSystemVolumeCheck);
#ifndef __RELEASE__
  if (ControlP == NULL)
    ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

  CtlSetValue(ControlP,
              prefP->SoundSystemVolume);

  // Volume slider
  ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                             EmulatorSettingsFormVolumeSlider);
#ifndef __RELEASE__
  if (ControlP == NULL)
    ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

  CtlSetValue(ControlP,
              CustomDataP->prefP->SoundVolume);

  //
  // Context saving
  //
  // Automatic saving at exit
  ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                             EmulatorSettingsFormAutoSaveCheck);
#ifndef __RELEASE__
  if (ControlP == NULL)
    ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

  CtlSetValue(ControlP,
              CustomDataP->prefP->AutomaticContextSave);

  //
  // AutoStart
  //
  ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                             EmulatorSettingsFormAutoStartCheck);
#ifndef __RELEASE__
  if (ControlP == NULL)
    ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

  CtlSetValue(ControlP,
              CustomDataP->prefP->AutoStartEnable);
              
  //
  // OnScreen Input
  //
  CustomDataP->OnScreenInputListHandle = SysFormPointerArrayToStrings((char*)OnScreenInputList,
                                                                      MAX_ONSCREENINPUTS);
#ifndef __RELEASE__
  if (CustomDataP->OnScreenInputListHandle == NULL)
    ErrDisplay("CustomDataP->OnScreenInputListHandle == NULL");
#endif /* __RELEASE__ */

  ListP = (ListPtr)GetFormObjectFromID(frmP,
                                       EmulatorSettingsFormOnScreenInputList);
#ifndef __RELEASE__
  if (ListP == NULL)
    ErrDisplay("ListP == NULL");
#endif /* __RELEASE__ */
  LstSetListChoices(ListP,
                    (char**)MemHandleLock(CustomDataP->OnScreenInputListHandle),
                    MAX_ONSCREENINPUTS);
  // Select actual input
  if (CustomDataP->prefP->OnScreenRockerActive)
  {
    Index = 1;
  }
  else if (CustomDataP->prefP->OnScreenLightPenActive)
  {
    Index = 2;
  }
  else if (CustomDataP->prefP->OnScreenMagnumGunActive)
  {
    Index = 3;
  }
  else if (CustomDataP->prefP->OnScreenGunstickActive)
  {
    Index = 4;
  }
  else if (CustomDataP->prefP->OnScreenWestPhaserActive)
  {
    Index = 5;
  }
  else
  {
    Index = 0;
  }
  LstSetSelection(ListP, Index);
  CtlSetLabel(GetFormObjectFromID(frmP,
                                  EmulatorSettingsFormOnScreenInputTrigger),
              LstGetSelectionText(ListP, Index));
             

  //
  // Rendering
  //
  // Volume slider
  ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                             EmulatorSettingsFormRenderingSlider);
#ifndef __RELEASE__
  if (ControlP == NULL)
    ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

  CtlSetValue(ControlP,
              CustomDataP->prefP->Rendering);
}
/*----------------------------------------------------------------------------*/

static void EmulatorSettingsFormApply(FormPtr frmP,
                                      tEmulatorSettingsFormCustomData* CustomDataP)
/***********************************************************************
 *
 *  EmulatorSettingsFormApply
 *
 ***********************************************************************/
{
Int16 Index;

#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

  //
  // Audio
  //
  // "Games" system volume checkbox
  CustomDataP->prefP->SoundSystemVolume = (tUChar)GetControlValue(frmP,
                                                                  EmulatorSettingsFormSystemVolumeCheck);
  // Volume slider
  CustomDataP->prefP->SoundVolume = (tUChar)GetControlValue(frmP,
                                                            EmulatorSettingsFormVolumeSlider);

  //
  // Session saving at exit
  //
  CustomDataP->prefP->AutomaticContextSave = (tUChar)GetControlValue(frmP,
                                                                     EmulatorSettingsFormAutoSaveCheck);

  //
  // AutoStart
  //
  CustomDataP->prefP->AutoStartEnable = (tUChar)GetControlValue(frmP,
                                                                EmulatorSettingsFormAutoStartCheck);

  //
  // On Screen Input
  //
  Index = GetListIndex(frmP,
                       EmulatorSettingsFormOnScreenInputList);

  CustomDataP->prefP->OnScreenRockerActive     = Index == 1 ? 1 : 0;
  CustomDataP->prefP->OnScreenLightPenActive   = Index == 2 ? 1 : 0;
  CustomDataP->prefP->OnScreenMagnumGunActive  = Index == 3 ? 1 : 0;
  CustomDataP->prefP->OnScreenGunstickActive   = Index == 4 ? 1 : 0;
  CustomDataP->prefP->OnScreenWestPhaserActive = Index == 5 ? 1 : 0;

  //
  // Rendering
  //
  CustomDataP->prefP->Rendering = (tUChar)GetControlValue(frmP,
                                                          EmulatorSettingsFormRenderingSlider);

  // Ask for Preferences save
  CustomDataP->prefP->PreferencesChanged = 1;
}
/*----------------------------------------------------------------------------*/

static void EmulatorSettingsFormClose(FormPtr frmP,
                                      tEmulatorSettingsFormCustomData* CustomDataP)
/***********************************************************************
 *
 *  EmulatorSettingsFormClose
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

#ifndef __RELEASE__
  if (CustomDataP->OnScreenInputListHandle == NULL)
    ErrDisplay("CustomDataP->OnScreenInputListHandle == NULL");
#endif /* __RELEASE__ */

  MemHandleUnlock(CustomDataP->OnScreenInputListHandle);
  MemHandleFree(CustomDataP->OnScreenInputListHandle);
}
/*----------------------------------------------------------------------------*/



Boolean ScreenOffsetFormHandleEvent(EventPtr eventP)
/***********************************************************************
 *
 *  ScreenOffsetFormHandleEvent
 *
 ***********************************************************************/
{
Boolean handled = false;
FormPtr frmP;
ControlPtr ControlP;

  frmP = FrmGetActiveForm();

#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

  switch (eventP->eType)
  {
    case ctlSelectEvent:
    {
      switch (eventP->data.ctlSelect.controlID)
      {
        case ScreenOffsetFormCancelButton:
        {
          //handled = false; // Exit modal form
        }
        break;

        case ScreenOffsetFormApplyButton:
        {
          ScreenOffsetFormApply(frmP);
          //handled = false; // Exit modal form
        }
        break;

        case ScreenOffsetFormResetXButton:
        {
          // Offset X Slider
          ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                                     ScreenOffsetFormOffsetXSlider);
#ifndef __RELEASE__
          if (ControlP == NULL)
            ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

          CtlSetValue(ControlP,
                      OFFSCREEN_OFFSET_X);
          
          handled = true;   // Stay into modal form
        }
        break;

        case ScreenOffsetFormResetYButton:
        {
          // Offset Y Slider
          ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                                     ScreenOffsetFormOffsetYSlider);
#ifndef __RELEASE__
          if (ControlP == NULL)
            ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

          CtlSetValue(ControlP,
                      OFFSCREEN_OFFSET_Y);

          handled = true;   // Stay into modal form
        }
        break;
      }
    }
    break;
  }

  return handled;
}
/*----------------------------------------------------------------------------*/

static void ScreenOffsetFormInit(FormPtr frmP)
/***********************************************************************
 *
 *  ScreenOffsetFormInit
 *
 ***********************************************************************/
{
ControlPtr ControlP;
UInt16 minValue;
UInt16 maxValue;
UInt16 value;

  // Offset X Slider
  ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                             ScreenOffsetFormOffsetXSlider);
#ifndef __RELEASE__
  if (ControlP == NULL)
    ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

  minValue = OFFSCREEN_OFFSET_X - ((CPC_SCR_WIDTH - SCREEN_CPC_WIDTH) / 2);
  maxValue = OFFSCREEN_OFFSET_X + ((CPC_SCR_WIDTH - SCREEN_CPC_WIDTH) / 2);
  value = prefP->ScreenDisplayOffsetX;
  CtlSetSliderValues(ControlP,
                     &minValue,
                     &maxValue,
                     NULL,
                     &value);

  // Offset Y Slider
  ControlP = (ControlPtr)GetFormObjectFromID(frmP,
                                             ScreenOffsetFormOffsetYSlider);
#ifndef __RELEASE__
  if (ControlP == NULL)
    ErrDisplay("ControlP == NULL");
#endif /* __RELEASE__ */

  minValue = OFFSCREEN_OFFSET_Y - ((SCREEN_CPC_HEIGHT_FS - SCREEN_PALM_HEIGHT) / 2);
  maxValue = OFFSCREEN_OFFSET_Y + ((SCREEN_CPC_HEIGHT_FS - SCREEN_PALM_HEIGHT) / 2);
  value = prefP->ScreenDisplayOffsetY;
  CtlSetSliderValues(ControlP,
                     &minValue,
                     &maxValue,
                     NULL,
                     &value);
}
/*----------------------------------------------------------------------------*/

static void ScreenOffsetFormApply(FormPtr frmP)
/***********************************************************************
 *
 *  ScreenOffsetFormApply
 *
 ***********************************************************************/
{
  // Offset X Slider
  prefP->ScreenDisplayOffsetX = (tUChar)GetControlValue(frmP,
                                                        ScreenOffsetFormOffsetXSlider);

  // Offset Y Slider
  prefP->ScreenDisplayOffsetY = (tUChar)GetControlValue(frmP,
                                                        ScreenOffsetFormOffsetYSlider);
}
/*----------------------------------------------------------------------------*/


Boolean HardKeysFormHandleEvent(EventPtr eventP)
/***********************************************************************
 *
 *  HardKeysFormHandleEvent
 *
 ***********************************************************************/
{
Boolean handled = false;
FormPtr frmP;
tHardKeysFormCustomData* CustomDataP;

  frmP = FrmGetActiveForm();
#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

  CustomDataP = (tHardKeysFormCustomData*)GetFormCustomData(frmP,
                                                            HardKeysFormCustomData);

  switch (eventP->eType)
  {
    case ctlSelectEvent:
    {
      switch (eventP->data.ctlSelect.controlID)
      {
        case EmulatorSettingsFormApplyButton:
        {
#ifndef __RELEASE__
          if (CustomDataP == NULL)
            ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

          HardKeysFormApply(frmP,
                            CustomDataP);
            
          // Take into account new keys parameters
          UpdateRockerCenterKeyMask();
          UpdateHardCPCKeyCodeMask();
          RestoreEmulatorKeysStatus(&EmulatorKeysStatus);
          SetHardKeysMask(prefP,
                          oldHardKeyMask);
        }
        // Then do all as for Cancel

        case EmulatorSettingsFormCancelButton:
        {
#ifndef __RELEASE__
          if (CustomDataP == NULL)
            ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

          HardKeysFormClose(frmP,
                            CustomDataP);

          if (CustomDataP)
          {
            RestoreDIAState(&CustomDataP->DIAState);
            
            FrmSetGadgetData(frmP,
                             FrmGetObjectIndex(frmP,
                                               HardKeysFormCustomData),
                             NULL);
      	
            MemPtrFree(CustomDataP);
          }

          FrmGotoForm(MainForm);

          handled = true;
        }
        break;

      }
    }
    break;

    case frmOpenEvent:
    {
      frmP = FrmGetActiveForm();
#ifndef __RELEASE__
      if (frmP == NULL)
        ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

      CustomDataP = (tHardKeysFormCustomData*)MemPtrNew(sizeof(tHardKeysFormCustomData));
      if (CustomDataP == NULL)
      {
        FrmAlert(NotEnoughMemoryAlert);
        FrmGotoForm(MainForm);
        //handled = false; //remember to leave handled as false
        break;
      }
      
      FrmSetGadgetData(frmP,
                       FrmGetObjectIndex(frmP,
                                         HardKeysFormCustomData),
                       CustomDataP);
      
      // Initialize custom data
      SaveDIAState(&CustomDataP->DIAState);
      CustomDataP->prefP = prefP;

      SetDIAPolicy(frmP,
                   pinInputTriggerDisabled,
                   DexLibRefNum,
                   0 /*pinInputAreaOpen*/);

      HardKeysFormInit(frmP,
                       CustomDataP);

      FrmDrawForm(frmP);
      //handled = false; //remember to leave handled as false
    }
    break;
  }

  return handled;
}
/*----------------------------------------------------------------------------*/


static void HardKeysFormInit(FormPtr frmP,
                             tHardKeysFormCustomData* CustomDataP)
/***********************************************************************
 *
 *  HardKeysFormInit
 *
 ***********************************************************************/
#define CPCKEYCODE_STRING_SIZE  5
{
char** listContentPP;

#ifndef __RELEASE__
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */


  //
  // Center Rocker key
  //
  CustomDataP->CenterKeyListHandle = SysFormPointerArrayToStrings((char*)RockerCenterKeysList,
                                                                  MAX_ROCKERCENTERKEYS);
#ifndef __RELEASE__
  if (CustomDataP->CenterKeyListHandle == NULL)
    ErrDisplay("CustomDataP->CenterKeyListHandle == NULL");
#endif /* __RELEASE__ */

  listContentPP = (char**)MemHandleLock(CustomDataP->CenterKeyListHandle);
  
  InitListAndSetIndex(frmP,
                      HardKeysRockerCenterList,
                      HardKeysRockerCenterTrigger,
                      listContentPP,
                      MAX_ROCKERCENTERKEYS,
                      CustomDataP->prefP->RockerCenterKeyIndex);

  //
  // Hard keys
  //
  CustomDataP->HardKeyListHandle = SysFormPointerArrayToStrings((char*)HardKeysList,
                                                                MAX_HARDKEYS);
#ifndef __RELEASE__
  if (CustomDataP->HardKeyListHandle == NULL)
    ErrDisplay("CustomDataP->HardKeyListHandle == NULL");
#endif /* __RELEASE__ */

  listContentPP = (char**)MemHandleLock(CustomDataP->HardKeyListHandle);

  // Hard key 1
  InitListAndSetIndex(frmP,
                      HardKeysHardKey1List,
                      HardKeysHardKey1Trigger,
                      listContentPP,
                      MAX_HARDKEYS,
                      CustomDataP->prefP->HardKey1Index);

  // Hard key 2
  InitListAndSetIndex(frmP,
                      HardKeysHardKey2List,
                      HardKeysHardKey2Trigger,
                      listContentPP,
                      MAX_HARDKEYS,
                      CustomDataP->prefP->HardKey2Index);

  // Hard key 3
  InitListAndSetIndex(frmP,
                      HardKeysHardKey3List,
                      HardKeysHardKey3Trigger,
                      listContentPP,
                      MAX_HARDKEYS,
                      CustomDataP->prefP->HardKey3Index);

  // Hard key 4
  InitListAndSetIndex(frmP,
                      HardKeysHardKey4List,
                      HardKeysHardKey4Trigger,
                      listContentPP,
                      MAX_HARDKEYS,
                      CustomDataP->prefP->HardKey4Index);


  // CPC key code A
  SetFieldInteger(frmP,
                  HardKeysCPCKeyCodeAField,
                  CPCKEYCODE_STRING_SIZE,
                  CustomDataP->prefP->CPCKeycodeA);

  // CPC key code B
  SetFieldInteger(frmP,
                  HardKeysCPCKeyCodeBField,
                  CPCKEYCODE_STRING_SIZE,
                  CustomDataP->prefP->CPCKeycodeB);

  // CPC key code C
  SetFieldInteger(frmP,
                  HardKeysCPCKeyCodeCField,
                  CPCKEYCODE_STRING_SIZE,
                  CustomDataP->prefP->CPCKeycodeC);

  // CPC key code D
  SetFieldInteger(frmP,
                  HardKeysCPCKeyCodeDField,
                  CPCKEYCODE_STRING_SIZE,
                  CustomDataP->prefP->CPCKeycodeD);
}
/*----------------------------------------------------------------------------*/


static void HardKeysFormApply(FormPtr frmP,
                              tHardKeysFormCustomData* CustomDataP)
/***********************************************************************
 *
 *  HardKeysFormApply
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

  
  // Center Rocker
  CustomDataP->prefP->RockerCenterKeyIndex = GetListIndex(frmP,
                                                          HardKeysRockerCenterList);
  // Hard key 1
  CustomDataP->prefP->HardKey1Index = GetListIndex(frmP,
                                                   HardKeysHardKey1List);
  // Hard key 2
  CustomDataP->prefP->HardKey2Index = GetListIndex(frmP,
                                                   HardKeysHardKey2List);
  // Hard key 3
  CustomDataP->prefP->HardKey3Index = GetListIndex(frmP,
                                                   HardKeysHardKey3List);
  // Hard key 4
  CustomDataP->prefP->HardKey4Index = GetListIndex(frmP,
                                                   HardKeysHardKey4List);

  //
  // CPC Keycodes
  //
  // CPC key code A
  CustomDataP->prefP->CPCKeycodeA = GetFieldInteger(frmP,
                                                           HardKeysCPCKeyCodeAField);
  // CPC key code B
  CustomDataP->prefP->CPCKeycodeB = GetFieldInteger(frmP,
                                                           HardKeysCPCKeyCodeBField);
  // CPC key code C
  CustomDataP->prefP->CPCKeycodeC = GetFieldInteger(frmP,
                                                           HardKeysCPCKeyCodeCField);
  // CPC key code D
  CustomDataP->prefP->CPCKeycodeD = GetFieldInteger(frmP,
                                                           HardKeysCPCKeyCodeDField);


  // Ask for Preferences save
  CustomDataP->prefP->PreferencesChanged = 1;
}
/*----------------------------------------------------------------------------*/


static void HardKeysFormClose(FormPtr frmP,
                              tHardKeysFormCustomData* CustomDataP)
/***********************************************************************
 *
 *  HardKeysFormClose
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (CustomDataP == NULL)
    ErrDisplay("CustomDataP == NULL");
#endif /* __RELEASE__ */

  
  // Rocker Center list
#ifndef __RELEASE__
  if (CustomDataP->CenterKeyListHandle == NULL)
    ErrDisplay("CustomDataP->CenterKeyListHandle == NULL");
#endif /* __RELEASE__ */

  MemHandleUnlock(CustomDataP->CenterKeyListHandle);
  MemHandleFree(CustomDataP->CenterKeyListHandle);

  // Hard keys list
#ifndef __RELEASE__
  if (CustomDataP->HardKeyListHandle == NULL)
    ErrDisplay("CustomDataP->HardKeyListHandle == NULL");
#endif /* __RELEASE__ */

  MemHandleUnlock(CustomDataP->HardKeyListHandle);
  MemHandleFree(CustomDataP->HardKeyListHandle);
}
/*----------------------------------------------------------------------------*/


void ChangeDriveAContent(void)
/***********************************************************************
 *
 *  ChangeDriveAContent
 *
 ***********************************************************************/
{
  DriveSelected = DriveA;

  // Open Drive form
  FrmGotoForm(DriveForm);
}
/*----------------------------------------------------------------------------*/


void ChangeDriveBContent(void)
/***********************************************************************
 *
 *  ChangeDriveBContent
 *
 ***********************************************************************/
{
  DriveSelected = DriveB;

  // Open Drive form
  FrmGotoForm(DriveForm);
}
/*----------------------------------------------------------------------------*/
