/*
    CaPriCe for Palm OS - Amstrad CPC 464/664/6128 emulator for Palm devices
    Copyright (C) 2009  Frdric Coste

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <PalmOS.h>
#include <VFSMgr.h>
#include <PceNativeCall.h>
#include <ByteOrderUtils.h>

#include "Files.h"
#include "Routines.h"
#include "Trace.h"
#include "Forms.h"



//===================
// PATCH begin
#ifdef _PATCH_ENABLE

#endif /* _PATCH_ENABLE */
// PATCH end
//===================



#define MISC_SECTION_SIZE          ( CONTEXT_SIZE_Z80_SZ + CONTEXT_SIZE_Z80_SZ_BIT + CONTEXT_SIZE_Z80_SZP + \
                                     CONTEXT_SIZE_Z80_SZHV_inc + CONTEXT_SIZE_Z80_SZHV_dec + \
                                     CONTEXT_SIZE_MODE0_TABLE + CONTEXT_SIZE_MODE1_TABLE + \
                                     CONTEXT_SIZE_GATEARRAY + \
                                     CONTEXT_SIZE_PPI + \
                                     CONTEXT_SIZE_PSG_LEVEL_PP + \
                                     CONTEXT_SIZE_PSG_LEVEL_AR + CONTEXT_SIZE_PSG_LEVEL_AL + \
                                     CONTEXT_SIZE_PSG_LEVEL_BR + CONTEXT_SIZE_PSG_LEVEL_BL + \
                                     CONTEXT_SIZE_PSG_LEVEL_CR + CONTEXT_SIZE_PSG_LEVEL_CL + \
                                     CONTEXT_SIZE_GPBUFFER + \
                                     CONTEXT_SIZE_RAM + CONTEXT_SIZE_SND_BUFFER + \
                                     CONTEXT_SIZE_SESSIONFILENAME )

typedef struct
{
  UInt32 Offset;
  UInt32 Size;
  UInt32 StaticAreaSize;
  UInt32 DynamicAreaSize;
} tRestoreCPCSection;

static const tRestoreCPCSection RestoreCPCSections[] =
{
	// Z80 section
	{ CONTEXT_OFFSET_Z80, CONTEXT_SIZE_Z80, STATIC_SIZE_Z80, DYNAMIC_SIZE_Z80 },
	// PSG section
	{ CONTEXT_OFFSET_PSG, CONTEXT_SIZE_PSG, STATIC_SIZE_PSG, DYNAMIC_SIZE_PSG },
	// FDC section
	{ CONTEXT_OFFSET_FDC, CONTEXT_SIZE_FDC, STATIC_SIZE_FDC, DYNAMIC_SIZE_FDC },
	// VDU section
	{ CONTEXT_OFFSET_VDU, CONTEXT_SIZE_VDU, STATIC_SIZE_VDU, DYNAMIC_SIZE_VDU },
	// CRTC section
	{ CONTEXT_OFFSET_CRTC, CONTEXT_SIZE_CRTC, STATIC_SIZE_CRTC, DYNAMIC_SIZE_CRTC },
	// Misc section
	{ CONTEXT_OFFSET_Z80_SZ, MISC_SECTION_SIZE, 0, 0 },
	// Sound Callback Param section
	{ CONTEXT_OFFSET_SOUND_CB_PARAM, CONTEXT_SIZE_SOUND_CB_PARAM, STATIC_SIZE_SOUND_CB_PARAM, DYNAMIC_SIZE_SOUND_CB_PARAM },
};
#define NB_RESTORECPC_SECTIONS    NUMBER_OF_ITEMS(RestoreCPCSections)


//
// BMP FORMAT
//
typedef struct
{
	tUShort Signature;
	tULong FileSize;
	tULong Reserved;
	tULong Offset;
}tBitmapFileHeader;

typedef struct
{
	tULong Size;
	tULong ImageWidth;
	tULong ImageHeight;
	tUShort NbPlanes;
	tUShort BitsPerPixel;
	tULong CompressionType;
	tULong ImageSize;
	tULong HorizontalResolution;
	tULong VerticalResolution;
	tULong NbColors;
	tULong NbImportantColors;
}tBitmapHeader;


#define SCREENSHOT_FILE_HEADER_SIZE     sizeof(tBitmapFileHeader)
#define SCREENSHOT_BITMAP_HEADER_SIZE   sizeof(tBitmapHeader)
#define SCREENSHOT_NB_COLORS            32
#define SCREENSHOT_COLOR_SIZE           4
#define SCREENSHOT_IMAGE_WIDTH          384L
#define SCREENSHOT_IMAGE_HEIGHT         272L
#define SCREENSHOT_IMAGE_HEIGHT_OFFSET  0  // To keep top and bottom borders with same height
#define SCREENSHOT_HEADER               ( SCREENSHOT_FILE_HEADER_SIZE + SCREENSHOT_BITMAP_HEADER_SIZE + \
                                          (SCREENSHOT_NB_COLORS * SCREENSHOT_COLOR_SIZE) )
#define SCREENSHOT_SIZE_IMAGE           (SCREENSHOT_IMAGE_WIDTH * SCREENSHOT_IMAGE_HEIGHT)
#define SCREENSHOT_SIZE                 (SCREENSHOT_HEADER + SCREENSHOT_SIZE_IMAGE)


static const tBitmapFileHeader BitmapFileHeader =
{
  'BM',
  EndianSwap32(SCREENSHOT_SIZE),
  0,
  EndianSwap32(SCREENSHOT_HEADER)
};

static const tBitmapHeader BitmapHeader =
{
  EndianSwap32(SCREENSHOT_BITMAP_HEADER_SIZE),
  EndianSwap32(SCREENSHOT_IMAGE_WIDTH),
  EndianSwap32(SCREENSHOT_IMAGE_HEIGHT),
  EndianSwap16(1),                               // Number of planes
  EndianSwap16(8),                               // Bits per pixel
  0,                                             // Type of compression
  EndianSwap32(SCREENSHOT_SIZE_IMAGE),
  0,                                             // Horizontal resolution
  0,                                             // Vertical resolution
  EndianSwap32(SCREENSHOT_NB_COLORS),            // Number of colors
  0                                              // Number of important colors
};


static Err OpenDirectory(const char* pathP,
                         FileRef* dirRefP,
                         UInt16* volRefNumP) SECTION_FILES;

static Err OpenFile(const char* pathP,
                    const char* filenameP,
                    FileRef* fileRefP,
                    UInt16 openMode,
                    Boolean create) SECTION_FILES;
                    
#ifndef _USE_MACROS
static Err CloseFile(FileRef fileRef) SECTION_FILES;
#else /* _USE_MACROS */
#define CloseFile(fileRef) \
  VFSFileClose((FileRef)fileRef);
#endif /* _USE_MACROS */

static tCPCRestoreReturnCode RestoreCPC_Section(FileRef ContextFileRef,
                                                tUChar* ContextP,
                                                tUChar* FileContextP,
                                                const tRestoreCPCSection* sectionP) SECTION_FILES;
static tCPCRestoreReturnCode RestoreCPC_Drive(FileRef ContextFileRef,
                                              tDrive* NativeDriveP,
                                              tNativeCPC* NativeCPC,
                                              tBool* LoadDiskImageP);
static tCPCRestoreReturnCode RestoreCPC_DiskImage(FileRef ContextFileRef,
                                                  tDrive* NativeDriveP,
                                                  tNativeCPC* NativeCPC) SECTION_FILES;
static tCPCRestoreReturnCode RestoreCPC_MemBank(FileRef ContextFileRef,
                                                tUChar* ContextP,
                                                tUChar* FileContextP,
                                                tUChar* readNativeCPCP) SECTION_FILES;

#ifndef _USE_MACROS

tBool IsDriveFilenameExist(tDrive* nativeDriveP)
/***********************************************************************
 *
 *  IsDriveFilenameExist
 *
 ***********************************************************************/
{
  return (((tDrive*)EndianSwap32(nativeDriveP))->filename[0] == 0 ? cFalse : cTrue);
}
/*----------------------------------------------------------------------------*/

void SetDriveFilename(tDrive* nativeDriveP,
                      const char* filenameP)
/***********************************************************************
 *
 *  SetDriveFilename
 *
 ***********************************************************************/
{
  StrCopy(((tDrive*)EndianSwap32(nativeDriveP))->filename,
          filenameP);
}
/*----------------------------------------------------------------------------*/

void GetDriveFilename(tDrive* nativeDriveP,
                      char* filenameP)
/***********************************************************************
 *
 *  GetDriveFilename
 *
 ***********************************************************************/
{
  StrCopy(filenameP,
          ((tDrive*)EndianSwap32(nativeDriveP))->filename);
}
/*----------------------------------------------------------------------------*/

#endif /* _USE_MACROS */


static Err OpenDirectory(const char* pathP,
                         FileRef* dirRefP,
                         UInt16* volRefNumP)
/***********************************************************************
 *
 *  OpenDirectory
 *
 ***********************************************************************/
{
Err Result = errNone;
UInt32 volIterator = vfsIteratorStart;

#ifndef __RELEASE__
  if (pathP == NULL)
    ErrDisplay("pathP == NULL");
  if (dirRefP == NULL)
    ErrDisplay("dirRefP == NULL");
  if (volRefNumP == NULL)
    ErrDisplay("volRefNumP == NULL");
#endif /* __RELEASE__ */

  //
  // Search directory among volumes
  //
  while (volIterator != vfsIteratorStop)
  {
    Result = VFSVolumeEnumerate(volRefNumP,
                                &volIterator);
    if (Result != errNone)
      return Result;

    if (VFSFileOpen(*volRefNumP,
                    pathP,
                    vfsModeRead,
                    dirRefP) == errNone)
    {
    	return errNone;
    }
  }

	return vfsErrFileNotFound;
}
/*----------------------------------------------------------------------------*/


static Err OpenFile(const char* pathP,
                    const char* filenameP,
                    FileRef* fileRefP,
                    UInt16 openMode,
                    Boolean create)
/***********************************************************************
 *
 *  OpenFile
 *
 ***********************************************************************/
#undef OPENFILE_DEBUG_ENABLED
//#define OPENFILE_DEBUG_ENABLED
#undef OPENFILE_TRACE_ENABLED
//#define OPENFILE_TRACE_ENABLED

#ifdef OPENFILE_TRACE_ENABLED
#  define OPENFILE_TRACE_SHOW_INT(value) TRACE_SHOW_INT("OpenFile", value)
#else /* OPENFILE_TRACE_ENABLED */
#  define OPENFILE_TRACE_SHOW_INT(value)
#endif /* OPENFILE_TRACE_ENABLED */
{
Err Result = errNone;
UInt16 volRefNum;
UInt32 volIterator = vfsIteratorStart;
FileRef dirRef;
Boolean dirFound = false;
Boolean dirCreated = false;
char* fullpathnameP;

  OPENFILE_TRACE_SHOW_INT(1);
  
#ifndef __RELEASE__
  if (pathP == NULL)
    ErrDisplay("pathP == NULL");
  if (filenameP == NULL)
    ErrDisplay("filenameP == NULL");
  if (fileRefP == NULL)
    ErrDisplay("fileRefP == NULL");
#endif /* __RELEASE__ */

#ifdef OPENFILE_DEBUG_ENABLED
  TRACE_SHOW_TEXT("pathP", pathP);
  TRACE_SHOW_TEXT("filenameP", filenameP);
#endif /* OPENFILE_DEBUG_ENABLED */

  //
  // Search directory among volumes
  //
  while (volIterator != vfsIteratorStop)
  {
    Result = VFSVolumeEnumerate(&volRefNum,
                                &volIterator);
    if (Result != errNone)
    {
#ifdef OPENFILE_DEBUG_ENABLED
      TRACE_SHOW_HEX("VFSVolumeEnumerate FAILURE Result=", Result);
#endif /* OPENFILE_DEBUG_ENABLED */
      return Result;
    }

    Result = VFSFileOpen(volRefNum,
                         pathP,
                         vfsModeRead,
                         &dirRef);
    if (Result == errNone)
    {
      dirFound = true;
      break;
    }
  }

  OPENFILE_TRACE_SHOW_INT(2);
  
  Result = VFSFileClose(dirRef);
  if (Result != errNone)
  {
#ifdef OPENFILE_DEBUG_ENABLED
    TRACE_SHOW_HEX("VFSFileClose FAILURE Result=", Result);
#endif /* OPENFILE_DEBUG_ENABLED */
    return Result;
  }

  OPENFILE_TRACE_SHOW_INT(3);
  
  if (dirFound == false)
  {
    if (create == false)
      return vfsErrFileNotFound;

    OPENFILE_TRACE_SHOW_INT(4);

    //
    // Create directory on first volume
    //
    while (volIterator != vfsIteratorStop)
    {
      Result = VFSVolumeEnumerate(&volRefNum,
                                  &volIterator);
      if (Result != errNone)
        return Result;

      if (VFSDirCreate(volRefNum,
                       pathP) == errNone)
      {
        dirCreated = true;
        break;
      }
    }

    OPENFILE_TRACE_SHOW_INT(5);

    if (dirCreated == false)
      return vfsErrFileNotFound;
  }

  OPENFILE_TRACE_SHOW_INT(6);

  //
  // Create filename
  //
  fullpathnameP = (char*)MemPtrNew(PATHNAME_MAXSIZE + SIZETAB_FILENAME + 1);
  if (fullpathnameP == NULL)
    return memErrNotEnoughSpace;

  StrCopy(fullpathnameP,
          pathP);
  StrCat(fullpathnameP,
         filenameP);
  
  OPENFILE_TRACE_SHOW_INT(7);

  //
  // Open file
  //
  Result = VFSFileOpen(volRefNum,
                       fullpathnameP,
                       openMode,
                       fileRefP);
  if (Result == vfsErrFileNotFound)
  {
    if (create == true)
    {
      //
      // Create file
      //
      Result = VFSFileCreate(volRefNum,
                             fullpathnameP);
      if (Result == errNone)
      {
        Result = VFSFileOpen(volRefNum,
                             fullpathnameP,
                             openMode,
                             fileRefP);
      }
    }
  }

  OPENFILE_TRACE_SHOW_INT(8);

  MemPtrFree(fullpathnameP);

  return Result;
}
/*----------------------------------------------------------------------------*/


#ifndef _USE_MACROS

static Err CloseFile(FileRef fileRef)
/***********************************************************************
 *
 *  CloseFile
 *
 ***********************************************************************/
{
  return VFSFileClose(fileRef);
}
/*----------------------------------------------------------------------------*/

#endif /* _USE_MACROS */


tBool IsFileExist(const char* pathP,
                  const char* filenameP)
/***********************************************************************
 *
 *  IsFileExist
 *
 ***********************************************************************/
{
FileRef ref;

  if ( (pathP == NULL) || (filenameP == NULL) )
    return cFalse;
  
  if (OpenFile(pathP,
               filenameP,
               &ref,
               vfsModeRead,
               false) == errNone)
  {
    CloseFile(ref);
    return cTrue;
  }
  
  return cFalse;
}
/*----------------------------------------------------------------------------*/


tBool IsDirectoryExist(const char* pathP)
/***********************************************************************
 *
 *  IsDirectoryExist
 *
 ***********************************************************************/
{
FileRef dirRef;
UInt16 volRefNum;

#ifndef __RELEASE__
  if (pathP == NULL)
    ErrDisplay("pathP == NULL");
#endif /* __RELEASE__ */

  if (OpenDirectory(pathP,
                    &dirRef,
                    &volRefNum) == errNone)
  {
  	VFSFileClose(dirRef);
  	return cTrue;
  }
  
  return cFalse;
}
/*----------------------------------------------------------------------------*/


Err DeleteFile(const char* pathP,
               const char* filenameP)
/***********************************************************************
 *
 *  DeleteFile
 *
 ***********************************************************************/
#undef DELETEFILE_TRACE_ENABLED
//#define DELETEFILE_TRACE_ENABLED

#ifdef DELETEFILE_TRACE_ENABLED
#  define DELETEFILE_TRACE_SHOW_INT(value) TRACE_SHOW_INT("DeleteFile", value)
#else
#  define DELETEFILE_TRACE_SHOW_INT(value)
#endif /* DELETEFILE_TRACE_ENABLED */

{
Err Result = errNone;
UInt16 volRefNum;
UInt32 Attributes;
FileRef dirRef = 0;
FileRef fileRef = 0;
char* fullfilenameP;

  DELETEFILE_TRACE_SHOW_INT(1);

  //
  // Remove Read-only attributes
  //
  do
  {
    // Open File
    Result = OpenFile(pathP,
                      filenameP,
                      &fileRef,
                      vfsModeReadWrite,
                      false);
    if (Result != errNone) continue;

    DELETEFILE_TRACE_SHOW_INT(2);
    
    Result = VFSFileGetAttributes(fileRef,
                                  &Attributes);
    if (Result != errNone) continue;

    DELETEFILE_TRACE_SHOW_INT(3);
    	
    // Remove Write-only attributes
    if (Attributes & vfsFileAttrReadOnly)
    {
      DELETEFILE_TRACE_SHOW_INT(4);

    	Result = VFSFileSetAttributes(fileRef,
    	                              0);
      if (Result != errNone) continue;

      DELETEFILE_TRACE_SHOW_INT(5);
    }
  }
  while (0);

  DELETEFILE_TRACE_SHOW_INT(6);
    
  if (fileRef)
  {
  	VFSFileClose(fileRef);
  }

  DELETEFILE_TRACE_SHOW_INT(7);

  fullfilenameP = (char*)MemPtrNew(PATHNAME_MAXSIZE + SIZETAB_FILENAME + 1);
  if (fullfilenameP == NULL)
    return memErrNotEnoughSpace;

  DELETEFILE_TRACE_SHOW_INT(8);
  
  //
  // Delete File
  //
  do
  {
  	// Open directory
    Result = OpenDirectory(pathP,
                           &dirRef,
                           &volRefNum);
    if (Result != errNone) continue;

    DELETEFILE_TRACE_SHOW_INT(9);

    // Create full filename
    StrCopy(fullfilenameP,
            pathP);
    StrCat(fullfilenameP,
           filenameP);

  	Result = VFSFileDelete(volRefNum,
  	                       fullfilenameP);
    if (Result != errNone) continue;

    DELETEFILE_TRACE_SHOW_INT(10);
  }
  while (0);

  DELETEFILE_TRACE_SHOW_INT(11);

  if (dirRef)
  {
  	VFSFileClose(dirRef);
  }

  DELETEFILE_TRACE_SHOW_INT(12);

  MemPtrFree(fullfilenameP);
  
  return Result;
}
/*----------------------------------------------------------------------------*/


Err CreateDirectories(const char* palmPathP,
                      const char* capricePathP,
                      const char* diskPathP,
                      const char* screenshotPathP)
/***********************************************************************
 *
 *  CreateDirectories
 *
 ***********************************************************************/
{
Err Result = errNone;
UInt16 volRefNum;
FileRef dirRef;
Boolean dirFound = false;

  // Search palm directory among volumes
  Result = OpenDirectory(palmPathP,
                         &dirRef,
                         &volRefNum);
  if (Result != errNone)
    return Result;

  VFSFileClose(dirRef);

  // Open CaPriCe directory
  if (VFSFileOpen(volRefNum,
                  capricePathP,
                  vfsModeRead,
                  &dirRef) == errNone)
  {
    VFSFileClose(dirRef);
  }
  else
  {
    if (VFSDirCreate(volRefNum,
                     capricePathP) != errNone)
    {
      return vfsErrFileNotFound;
    }
  }

  // Open Disc directory
  if (VFSFileOpen(volRefNum,
                  diskPathP,
                  vfsModeRead,
                  &dirRef) == errNone)
  {
    VFSFileClose(dirRef);
  }
  else
  {
    if (VFSDirCreate(volRefNum,
                     diskPathP) != errNone)
    {
      return vfsErrFileNotFound;
    }
  }

  // Open Screenshot directory
  if (VFSFileOpen(volRefNum,
                  screenshotPathP,
                  vfsModeRead,
                  &dirRef) == errNone)
  {
    VFSFileClose(dirRef);
  }
  else
  {
    if (VFSDirCreate(volRefNum,
                     screenshotPathP) != errNone)
    {
      return vfsErrFileNotFound;
    }
  }

  return errNone;
}
/*----------------------------------------------------------------------------*/



Err EjectDisk(tDrive* nativedriveP,
              tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  EjectDisk
 *
 ***********************************************************************/
{
tDiskOperation* DiskOperationP;
Err Result;

  DiskOperationP = (tDiskOperation*)(EndianSwap32(NativeCPC->contextP) + CONTEXT_OFFSET_DISKOPERATION);

  DiskOperationP->NativeCPC = NativeCPC;
  DiskOperationP->Drive = (tDrive*)EndianSwap32(nativedriveP);

  Result = PceNativeCallResource(Native_DiskEject,
#ifdef __SIMU__
                                 "Native_DiskEject.dll\0PNOMain",
#endif /* __SIMU__ */
                                 DiskOperationP);

#if defined(_TESTU) && defined(_TRACE)
  if (Result >= appErrorClass)
  {
    TRACE_SHOW_HEX("TestU Native_DiskEject",
                   Result-testUErrorClass);
  }
#endif

  return Result;
}
/*----------------------------------------------------------------------------*/


Err LoadDiskImage(const char* pathP,
                  tDrive* nativedriveP,
                  tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  LoadDiskImage
 *
 ***********************************************************************/
{
tDrive* Drive = (tDrive*)EndianSwap32(nativedriveP);
tDiskOperation* DiskOperationP = NULL;
FileRef DiskImageRef;
UInt32 Attributes;
UInt32 numBytesRead;
Err Result;
char* memP = NULL;

  Result = OpenFile(pathP,
                    Drive->filename,
                    &DiskImageRef,
                    vfsModeRead,
                    false);
  if (Result != errNone)
    return Result;
    
  do
  {
    memP = (char*)MemPtrNew(sizeof(Drive->filename));
    if (memP == NULL)
    {
    	Result = memErrNotEnoughSpace;
    	continue;
    }
    MemMove(memP,
            Drive->filename,
            sizeof(Drive->filename));

    Result = SaveAndEjectDiskImage(pathP,
                                   nativedriveP,
                                   NativeCPC);
    if (Result != errNone) continue;

    MemMove(Drive->filename,
            memP,
            sizeof(Drive->filename));

    //
    // Prepare disk load operation
    //
    DiskOperationP = (tDiskOperation*)(EndianSwap32(NativeCPC->contextP) + CONTEXT_OFFSET_DISKOPERATION);

    DiskOperationP->Drive = Drive;
    DiskOperationP->NativeCPC = NativeCPC;

    //
    // Load entire disk image
    //
    // Get File size
    Result = VFSFileSize(DiskImageRef,
                         &DiskOperationP->disk_size);
    if (Result != errNone) continue;

    // Allocate memory for whole disk data
    MemPtrNewLarge(DiskOperationP->disk_size,
                   (void**)&DiskOperationP->DiskContentP);
    if (DiskOperationP->DiskContentP == NULL)
    {
      FrmAlert(NotEnoughMemoryAlert);
      Result = memErrNotEnoughSpace;
      continue;
    }

    // Read entire disk image file
    VFSFileRead(DiskImageRef,
                DiskOperationP->disk_size,
                (void*)DiskOperationP->DiskContentP,
                &numBytesRead);
    // Check read
    if (numBytesRead != DiskOperationP->disk_size)
    {
      FrmAlert(InvalidImageFormatAlert);
      Result = vfsErrBadData;
      continue;
    }

    //
    // Disk load operation
    //
    Result = PceNativeCallResource(Native_DiskLoad,
#ifdef __SIMU__
                                   "Native_DiskLoad.dll\0PNOMain",
#endif /* __SIMU__ */
                                   DiskOperationP);

#if defined(_TESTU) && defined(_TRACE)
    if (Result >= appErrorClass)
    {
      TRACE_SHOW_HEX("TestU Native_DiskLoad",
                     Result-testUErrorClass);
    }
#endif

    if (Result != errNone)
    {
      FrmAlert(InvalidImageFormatAlert);
      continue;
    }
    
    // Set write protected state
    Result = VFSFileGetAttributes(DiskImageRef,
                                  &Attributes);
    if (Result != errNone) continue;

    if (Attributes & vfsFileAttrReadOnly)
    {
    	Drive->write_protected = ~0; // Write protected is set
    }
  }
  while (0);

  if (memP != NULL)
  {
    MemPtrFree(memP);
  }

  // Fermer le fichier image
  CloseFile(DiskImageRef);

  return Result;
}
/*----------------------------------------------------------------------------*/


Err LoadDiskImageFromMemory(tDrive* nativedriveP,
                            tNativeCPC* NativeCPC,
                            tULong DiskSize,
                            tUChar* DiskContentP)
/***********************************************************************
 *
 *  LoadDiskImageFromMemory
 *
 ***********************************************************************/
{
tDrive* Drive = (tDrive*)EndianSwap32(nativedriveP);
tDiskOperation* DiskOperationP;
Err Result;

  //
  // Prepare disk load operation
  //
  DiskOperationP = (tDiskOperation*)(EndianSwap32(NativeCPC->contextP) + CONTEXT_OFFSET_DISKOPERATION);

  DiskOperationP->Drive = Drive;
  DiskOperationP->NativeCPC = NativeCPC;
  DiskOperationP->disk_size = EndianSwap32(DiskSize);
  DiskOperationP->DiskContentP = (tUChar*)EndianSwap32(DiskContentP);
  
  //
  // Disk load operation
  //
  Result = PceNativeCallResource(Native_DiskLoad,
#ifdef __SIMU__
                                 "Native_DiskLoad.dll\0PNOMain",
#endif /* __SIMU__ */
                                 DiskOperationP);

#ifdef _TESTU
  if (Result >= appErrorClass)
  {
    TRACE_SHOW_HEX("TestU Native_DiskLoad", Result-testUErrorClass);
  }
#endif /* _TESTU */

  if (Result != errNone)
  {
    FrmAlert(InvalidImageFormatAlert);
  }

  return Result;
}
/*----------------------------------------------------------------------------*/


Err SaveDiskImage(const char* pathP,
                  tDrive* nativedriveP,
                  tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  SaveDiskImage
 *
 ***********************************************************************/
{
tDrive* DriveP = (tDrive*)EndianSwap32(nativedriveP);
tUChar* diskdataP;
Err Result;
UInt32 attributs;
UInt32 disksize;
UInt32 numBytesWritten;
FileRef DiskImageRef;
tDiskOperation* DiskOperationP = NULL;


  if ( (DriveP == NULL) || (pathP == NULL) || (!DriveP->filename[0]))
    return vfsErrFileNotFound;

  // Si le disque n'a pas t modifi, ne rien faire
  if (DriveP->altered == 0)
    return errNone;

  // Ouvrir le fichier image
  Result = OpenFile(pathP,
                    DriveP->filename,
                    &DiskImageRef,
                    vfsModeReadWrite,
                    true);
  if (Result != errNone)
    return Result;
    
  do
  {
    // Rcuprer les attributs du fichier.
    Result = VFSFileGetAttributes(DiskImageRef,
                                  &attributs);
    if (Result != errNone) continue;

    // Si le fichier est en lecture seule, abandonner
    if ((attributs & vfsFileAttrReadOnly) != 0)
    {
      FrmCustomAlert(WriteProtectedDiskAlert,
                     DriveP->filename,
                     "",
                     "");
      Result = vfsErrFilePermissionDenied;
      continue;
    }

    // Sauvegarder le fichier image
    DiskOperationP = (tDiskOperation*)(EndianSwap32(NativeCPC->contextP) + CONTEXT_OFFSET_DISKOPERATION);

    DiskOperationP->Drive = DriveP;
#ifdef _TRACE
    DiskOperationP->NativeCPC = NativeCPC;
#endif /* _TRACE */

    Result = PceNativeCallResource(Native_DiskSave,
#ifdef __SIMU__
                                   "Native_DiskSave.dll\0PNOMain",
#endif /* __SIMU__ */
                                   DiskOperationP);

#ifdef _TESTU
    if (Result >= appErrorClass)
    {
      TRACE_SHOW_HEX("TestU Native_DiskSave", Result-testUErrorClass);
    }
#endif

    if (Result != errNone)
    {
      FrmAlert(BadWriteImageAlert);
      continue;
    }

    //
    // Save entire disk file
    //
    diskdataP = (tUChar*)EndianSwap32(DriveP->dataP);
    disksize = (UInt32)EndianSwap32(DriveP->data_size);
    Result = VFSFileWrite(DiskImageRef,
                          disksize,
                          diskdataP,
                          &numBytesWritten);
#ifndef __RELEASE__
    if (Result != errNone)
      ErrDisplay("VFSFileWrite != errNone");
#endif /* __RELEASE__ */
    if (numBytesWritten != disksize)
    {
      Result = BadWriteImageAlert;
    }
    else
    {
      DriveP->altered = 0;
    }
  }
  while (0);

  // Fermer le fichier image
  CloseFile(DiskImageRef);

  return Result;
}
/*----------------------------------------------------------------------------*/


Err SaveAndEjectDiskImage(const char* pathP,
                          tDrive* nativedriveP,
                          tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  SaveAndEjectDiskImage
 *
 ***********************************************************************/
{
	// Save disk if possible before eject
  SaveDiskImage(pathP,
                nativedriveP,
                NativeCPC);
  
  return (EjectDisk(nativedriveP,
                    NativeCPC));
}
/*----------------------------------------------------------------------------*/


void SwapDrive(tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  SwapDrive
 *
 ***********************************************************************/
{
tDrive* driveTempP;

  // Swap Drives not regarding endianess
  driveTempP = NativeCPC->DriveA;
  NativeCPC->DriveA = NativeCPC->DriveB;
  NativeCPC->DriveB = driveTempP;

  FrmUpdateForm(MainForm,
                SubPanelRedrawUpdateCode);
}
/*----------------------------------------------------------------------------*/


Boolean SaveAndEjectBoth(tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  SaveAndEjectBoth
 *
 ***********************************************************************/
{
Boolean Change = false;

  // Drive A
  if (IsDriveFilenameExist(NativeCPC->DriveA) == cTrue)
  {
    SaveAndEjectDiskImage(prefP->FilesPathname,
                          NativeCPC->DriveA,
                          NativeCPC);
    Change = true;
  }

  // Drive B
  if (IsDriveFilenameExist(NativeCPC->DriveB) == cTrue)
  {
    SaveAndEjectDiskImage(prefP->FilesPathname,
                          NativeCPC->DriveB,
                          NativeCPC);
    Change = true;
  }
  
  return (Change);
}
/*----------------------------------------------------------------------------*/


Err CreateDisk(tDrive* nativedriveP,
               tUChar FormatType,
               tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  CreateDisk
 *
 ***********************************************************************/
{
tDiskOperation* DiskOperationP;
Err Result;
MemHandle resH;

  DiskOperationP = (tDiskOperation*)(EndianSwap32(NativeCPC->contextP) + CONTEXT_OFFSET_DISKOPERATION);

  resH = DmGetResource(TABLE_ID_TYPE, TABLE_ID_DISK_FORMAT);

  DiskOperationP->Drive = (tDrive*)EndianSwap32(nativedriveP);
  DiskOperationP->NativeCPC = NativeCPC;
  DiskOperationP->FormatType = FormatType;
  DiskOperationP->DiskFormatTableP = (tDiskFormat*)MemHandleLock(resH);

  Result = PceNativeCallResource(Native_DiskFormat,
#ifdef __SIMU__
                                 "Native_DiskFormat.dll\0PNOMain",
#endif /* __SIMU__ */
                                 DiskOperationP);

#ifdef _TESTU
  if (Result >= appErrorClass)
  {
    TRACE_SHOW_HEX("TestU Native_DiskFormat",
                   Result-testUErrorClass);
  }
#endif

  MemHandleUnlock(resH);
  DmReleaseResource(resH);

  return Result;
}
/*----------------------------------------------------------------------------*/


UInt16 RetreiveFolderContent(const char* pathP,
                             char*** filelistPP,
                             const char* extensionP,
                             tCPCFolderContentOptions Options)
/***********************************************************************
 *
 *  RetreiveFolderContent
 *
 ***********************************************************************/
{
UInt16 volRefNum;
UInt32 vfsIterator;
FileRef dirRef;
FileInfoType FileInfo;
char* FilenameP;
char* EntryP;
char** FilenameEntry;
UInt16 Tri1, Tri2;
UInt16 NbFiles;
UInt16 FilenameSize;
UInt16 ExtensionSize;
UInt8 IsNotRoot;
UInt8 IsDirectory;

  // Librer la liste prcdente
  if (*filelistPP)
  {
    MemPtrFree(*filelistPP);
    *filelistPP = NULL;
  }
  
  if (OpenDirectory(pathP,
                    &dirRef,
                    &volRefNum) != errNone)
  {
  	return 0;
  }
  
  do
  {
    FilenameP = MemPtrNew(MAX_FILE_NAME);
    if (FilenameP == NULL) continue;

    // Exploration de la racine ?
    IsNotRoot = StrCompareAscii(pathP,
                                ROOT_PATH) ? 1 : 0;

    // Parcourir la liste pour connaitre le nombre d'entres dans ce dossier.
    NbFiles = IsNotRoot;
    FileInfo.nameP = NULL;
    FileInfo.nameBufLen = 0;
    vfsIterator = vfsIteratorStart;
    while (vfsIterator != vfsIteratorStop)
    {
      if (VFSDirEntryEnumerate(dirRef,
                               &vfsIterator,
                               &FileInfo) != errNone)
      {
        break;
      }

      NbFiles++;
    }

    if (!NbFiles)
    {
      MemPtrFree(FilenameP);
      continue;
    }

    // Allouer la taille de la liste pour accueillir le nom de tous les fichiers du dossier.
    *filelistPP = (char**)MemPtrNew(sizeof(char*) * NbFiles);
    if (*filelistPP == NULL)
    {
      MemPtrFree(FilenameP);
      continue;
    }
    
    NbFiles = 0;
    FilenameEntry = *filelistPP;
    FileInfo.nameP = FilenameP;
    FileInfo.nameBufLen = MAX_FILE_NAME;

    // Si pathP n'est pas la racine
    if ((Options & Show_Folder) && IsNotRoot)
    {
      // Ajouter retour au dossier parent
      *FilenameEntry = (char*)MemPtrNew(sizeof(PARENT_PATH));
      if (*FilenameEntry != NULL)
      {
        StrCopy(*(FilenameEntry++), PARENT_PATH);
        NbFiles++;
      }
    }
  
    // Rcuprer et stocker le nom de tous les fichiers du dossier.
    vfsIterator = vfsIteratorStart;
    while (vfsIterator != vfsIteratorStop)
    {
      if (VFSDirEntryEnumerate(dirRef,
                               &vfsIterator,
                               &FileInfo) != errNone)
      {
        break;
      }
      
      // Dossier ?
      IsDirectory = FileInfo.attributes & vfsFileAttrDirectory ? 1 : 0;
    
      FilenameSize = StrLen(FileInfo.nameP);
      ExtensionSize = StrLen(extensionP);

      EntryP = (char*)MemPtrNew(FilenameSize + IsDirectory + 1);
      if (EntryP != NULL)
      {
        if (IsDirectory)
        {
          if (Options & Show_Folder)
          {
            *EntryP = '/';
            StrCopy(EntryP+1,
                    FileInfo.nameP);
          }
          else
          {
            MemPtrFree(EntryP);
            continue;
          }
        }
        else
        {
          // Si l'extension est diffrente, ne pas ajouter ce fichier
          if (extensionP != NULL)
          {
            if (StrCaselessCompare(&FileInfo.nameP[FilenameSize-ExtensionSize],
                                   extensionP))
            {
              MemPtrFree(EntryP);
              continue;
            }
          }
        
          StrCopy(EntryP,
                  FileInfo.nameP);
        }
      
        *(FilenameEntry++) = EntryP;
        NbFiles++;
      }
    }

    MemPtrFree(FilenameP);

    //
    // Tri alphabtique des noms des fichiers
    //
    if (NbFiles >= 2) // Au moins 2 fichiers  trier
    {
      for(Tri1=0; Tri1 < (NbFiles-1); Tri1++)
      {
        for(Tri2=0; Tri2 < (NbFiles-1-Tri1); Tri2++)
        {
          if (StrCaselessCompare((*filelistPP)[Tri2],
                                 (*filelistPP)[Tri2+1]) > 0)
          {
            FilenameP = (*filelistPP)[Tri2];
            (*filelistPP)[Tri2] = (*filelistPP)[Tri2+1];
            (*filelistPP)[Tri2+1] = FilenameP;
          }
        }
      }
    }
  }
  while (0);

  VFSFileClose(dirRef);
  
  return NbFiles;
}
/*----------------------------------------------------------------------------*/


tCPCSaveReturnCode SaveCPC(const char* pathP,
                           const char* filenameP,
                           tUChar* contextP)
/***********************************************************************
 *
 *  SaveCPC
 *
 ***********************************************************************/
{
tCPCSaveReturnCode ReturnCode = Save_OK;
Err Result;
FileRef DiskImageRef;
UInt32 numBytesWritten;
tUChar* memoryP;
UInt32 size;
tDrive* driveP;
tContextHeader* contextHeaderP;
tULong Flags = 0;
tNativeCPC* NativeCPC = (tNativeCPC*)(contextP + CONTEXT_OFFSET_NATIVECPC);

  // Create result file
  Result = OpenFile(pathP,
                    filenameP,
                    &DiskImageRef,
                    vfsModeReadWrite,
                    true);
  if (Result != errNone)
    return File_Not_Created;
    
  // Prepare flags
#ifdef _PROFILE
  Flags |= CONTEXT_FLAG_PROFILE;
#endif /* _PROFILE */
#ifdef _DEBUG
  Flags |= CONTEXT_FLAG_DEBUG;
#endif /* _DEBUG */
#ifdef _TESTU
  Flags |= CONTEXT_FLAG_TESTU;
#endif /* _TESTU */
#ifdef _TRACE
  Flags |= CONTEXT_FLAG_TRACE;
#endif /* _TRACE */

  // Prepare context header
  contextHeaderP = (tContextHeader*)(contextP + CONTEXT_OFFSET_HEADER);
  contextHeaderP->Flags = EndianSwap32(Flags);
  contextHeaderP->CPCModel = prefP->CPCModel;
  contextHeaderP->ScreenType = prefP->ScreenType;
  contextHeaderP->UseParados = prefP->UseParados;
  contextHeaderP->Use64kMemoryExtension = prefP->Use64kMemoryExtension;
  contextHeaderP->EmuKeyFxState = EmulatorKeys[KEYINDEX_FX].KeyStatus == KeyPressed ? 1 : 0;
  contextHeaderP->EmuKeyJoystickState = EmulatorKeys[KEYINDEX_JOYSTICK].KeyStatus == KeyPressed ? 1 : 0;
  contextHeaderP->EmuKeyAutoToggleState = EmulatorKeys[KEYINDEX_AUTOTOGGLE].KeyStatus == KeyPressed ? 1 : 0;
  contextHeaderP->Mode2AntiAliasing = prefP->Mode2AntiAliasing;

  do
  {
    // Save of the context
    Result = VFSFileWrite(DiskImageRef,
                          SIZETAB_CONTEXT,
                          contextP,
                          &numBytesWritten);
#ifndef __RELEASE__
    if (Result != errNone)
      ErrDisplay("VFSFileWrite != errNone");
#endif /* __RELEASE__ */
    if (numBytesWritten != SIZETAB_CONTEXT)
    {
      ReturnCode = Write_Error;
      continue;
    }

    // Save Drive A Data
    driveP = (tDrive*)(contextP + CONTEXT_OFFSET_DRIVE_A);
    memoryP = (tUChar*)EndianSwap32(driveP->dataP);
    size = EndianSwap32(driveP->data_size);
    if (size)
    {
      Result = VFSFileWrite(DiskImageRef,
                            size,
                            memoryP,
                            &numBytesWritten);
#ifndef __RELEASE__
      if (Result != errNone)
        ErrDisplay("VFSFileWrite != errNone");
#endif /* __RELEASE__ */
      if (numBytesWritten != size)
      {
        ReturnCode = Write_Error;
        continue;
      }
    }

    // Save Drive B Data
    driveP = (tDrive*)(contextP + CONTEXT_OFFSET_DRIVE_B);
    memoryP = (tUChar*)EndianSwap32(driveP->dataP);
    size = EndianSwap32(driveP->data_size);
    if (size)
    {
      Result = VFSFileWrite(DiskImageRef,
                            size,
                            memoryP,
                            &numBytesWritten);
#ifndef __RELEASE__
      if (Result != errNone)
        ErrDisplay("VFSFileWrite != errNone");
#endif /* __RELEASE__ */
      if (numBytesWritten != size)
      {
        ReturnCode = Write_Error;
        continue;
      }
    }
  }
  while (0);

  VFSFileClose(DiskImageRef);

  // Something wrong...Delete potential invalid file
  if (Result != errNone)
  {
    DeleteFile(pathP, filenameP);
  }

#ifdef _TRACE
  if (Result != errNone) TRACE_SHOW_HEX("SaveCPC End",
                                        Result);
#endif /* _TRACE */

  return ReturnCode;
}
/*----------------------------------------------------------------------------*/


tCPCRestoreReturnCode RestoreCPC(const char* pathP,
                                 const char* filenameP,
                                 tEmulatorKeysStatus* KeyStatusP,
                                 tPreferences* requestedPrefP,
                                 tUChar* contextP)
/***********************************************************************
 *
 *  RestoreCPC
 *
 ***********************************************************************/
#undef RESTORECPC_TRACE_ENABLED
//#define RESTORECPC_TRACE_ENABLED

#ifdef RESTORECPC_TRACE_ENABLED
#  define RESTORECPC_TRACE_SHOW_INT(value) TRACE_SHOW_INT("RestoreCPC", value)
#else
#  define RESTORECPC_TRACE_SHOW_INT(value)
#endif /* RESTORECPC_TRACE_ENABLED */
{
tNativeCPC* NativeCPC = (tNativeCPC*)(contextP + CONTEXT_OFFSET_NATIVECPC);
const tRestoreCPCSection* sectionP;
tCPCRestoreReturnCode ReturnCode = Restore_OK;
Err Result;
FileRef ContextFileRef;
tUChar* memP = NULL;
tUChar* readNativeCPCP = NULL;
tUChar* readContextP;
tContextHeader* contextHeaderP;
UInt32 numBytesRead;
tReindexOperationParameters ReindexParam;
UInt8 Loop;
tBool loadDiskImageA;
tBool loadDiskImageB;

  RESTORECPC_TRACE_SHOW_INT(1);

  Result = OpenFile(pathP,
                    filenameP,
                    &ContextFileRef,
                    vfsModeRead,
                    false);
  if (Result != errNone)
    return File_Not_Found;

  RESTORECPC_TRACE_SHOW_INT(2);

  contextHeaderP = (tContextHeader*)(contextP + CONTEXT_OFFSET_HEADER);

  do
  {
    //
    // Header
    //
    memP = (tUChar*)MemPtrNew(CONTEXT_SIZE_HEADER);
    if (memP == NULL) 
    { 
    	Result = memErrNotEnoughSpace; 
    	continue;
    }
    Result = VFSFileRead(ContextFileRef, 
                         CONTEXT_SIZE_HEADER, 
                         (void*)memP, 
                         &numBytesRead);
#ifndef __RELEASE__
    if (Result != errNone)
      ErrDisplay("VFSFileRead != errNone");
#endif /* __RELEASE__ */
    if (numBytesRead != CONTEXT_SIZE_HEADER)
    { 
      Result = vfsErrBadData;
      continue;
    }
  
    // Retreive requested parameters
    requestedPrefP->CPCModel  = ((tContextHeader*)memP)->CPCModel;
    requestedPrefP->ScreenType  = ((tContextHeader*)memP)->ScreenType;
    requestedPrefP->UseParados  = ((tContextHeader*)memP)->UseParados;
    requestedPrefP->Use64kMemoryExtension  = ((tContextHeader*)memP)->Use64kMemoryExtension;
    requestedPrefP->Mode2AntiAliasing  = ((tContextHeader*)memP)->Mode2AntiAliasing;

    // Check CaPriCe version
    if (StrCompare(((tContextHeader*)memP)->version,
                   contextHeaderP->version))
    {
      ReturnCode = Wrong_Version;
      continue;
    }
    // Check CPC Model
    if (requestedPrefP->CPCModel != prefP->CPCModel)
    {
      ReturnCode = Wrong_CPC_Model;
      continue;
    }
    // Restore keyboard status
    KeyStatusP->FxKeyStatus = ((tContextHeader*)memP)->EmuKeyFxState ? KeyPressed : KeyReleased;
    KeyStatusP->JoystickKeyStatus = ((tContextHeader*)memP)->EmuKeyJoystickState ? KeyPressed : KeyReleased;
    KeyStatusP->AutoToggleKeyStatus = ((tContextHeader*)memP)->EmuKeyAutoToggleState ? KeyPressed : KeyReleased;

    RESTORECPC_TRACE_SHOW_INT(3);

    //
    // Pass through Resources + DiskOperation
    //
    Result = VFSFileSeek(ContextFileRef,
                         vfsOriginCurrent,
                         CONTEXT_SIZE_RESOURCES + CONTEXT_SIZE_DISKOPERATION);
#ifndef __RELEASE__
    if (Result != errNone)
      ErrDisplay("VFSFileSeek != errNone");
#endif /* __RELEASE__ */

    RESTORECPC_TRACE_SHOW_INT(4);

    //
    // NativeCPC
    //
#define RESTORECPC_NATIVE_CPC_SIZE    (STATIC_SIZE_NATIVECPC + DYNAMIC_SIZE_NATIVECPC)
#define RESTORECPC_NATIVE_CPC_OFFSET  (CONTEXT_OFFSET_NATIVECPC + STATIC_SIZE_NATIVECPC)

    readNativeCPCP = (tUChar*)MemPtrNew(RESTORECPC_NATIVE_CPC_SIZE);
    if (readNativeCPCP == NULL)
    { 
  	  Result = memErrNotEnoughSpace;
  	  continue;
  	}
  
    Result = VFSFileRead(ContextFileRef, 
                         RESTORECPC_NATIVE_CPC_SIZE, 
                         (void*)readNativeCPCP, 
                         &numBytesRead);
#ifndef __RELEASE__
    if (Result != errNone)
      ErrDisplay("VFSFileRead != errNone");
#endif /* __RELEASE__ */

    if ( (numBytesRead != RESTORECPC_NATIVE_CPC_SIZE) ||
         (Result != errNone) )
    { 
      ReturnCode = Bad_Data; 
      continue;
    }
    
    // Save read context pointer
    readContextP = (tUChar*)EndianSwap32(((tNativeCPC*)readNativeCPCP)->contextP);
  
    // Prepare reindex parameters
    ReindexParam.srcContextBase = (UInt32)readContextP;
    ReindexParam.dstContextBase = (UInt32)contextP;
    ReindexParam.contextSize = SIZETAB_CONTEXT;
    ReindexParam.native = cTrue;

    // Re-index dynamic area
    ReindexParam.srcAreaP = (void**)(readNativeCPCP + STATIC_SIZE_NATIVECPC);
    ReindexParam.dstAreaP = (void**)(contextP + RESTORECPC_NATIVE_CPC_OFFSET);
    ReindexParam.areaSize = DYNAMIC_SIZE_NATIVECPC;
    ReindexPointersArea(&ReindexParam);
  
    // Restore NativeCPC Variables
#define RESTORECPC_NATIVE_CPC_VAR_SIZE   (CONTEXT_SIZE_NATIVECPC - STATIC_SIZE_NATIVECPC - DYNAMIC_SIZE_NATIVECPC)
#define RESTORECPC_NATIVE_CPC_VAR_OFFSET (CONTEXT_OFFSET_NATIVECPC + STATIC_SIZE_NATIVECPC + DYNAMIC_SIZE_NATIVECPC)
    Result = VFSFileRead(ContextFileRef,
                         RESTORECPC_NATIVE_CPC_VAR_SIZE,
                         (void*)(contextP + RESTORECPC_NATIVE_CPC_VAR_OFFSET),
                         &numBytesRead);
#ifndef __RELEASE__
    if (Result != errNone)
      ErrDisplay("VFSFileRead != errNone");
#endif /* __RELEASE__ */
    if (Result != errNone) continue;

    RESTORECPC_TRACE_SHOW_INT(5);

    //
    // Read sections
    //
    for (Loop=NB_RESTORECPC_SECTIONS, sectionP=RestoreCPCSections; 
         Loop; 
         Loop--, sectionP++)
    {
      RESTORECPC_TRACE_SHOW_INT(500 + NB_RESTORECPC_SECTIONS - Loop);

      ReturnCode = RestoreCPC_Section(ContextFileRef,
                                      contextP,
                                      readContextP,
                                      sectionP);
      if (ReturnCode != Restore_OK) break;
    }
    if (ReturnCode != Restore_OK) continue;

    RESTORECPC_TRACE_SHOW_INT(6);
    
    //
    // Drive A
    //
    ReturnCode = RestoreCPC_Drive(ContextFileRef,
                                  (tDrive*)EndianSwap32(contextP + CONTEXT_OFFSET_DRIVE_A),
                                  NativeCPC,
                                  &loadDiskImageA);
    if (ReturnCode != Restore_OK) continue;

    RESTORECPC_TRACE_SHOW_INT(7);
    
    //
    // Drive B
    //
    ReturnCode = RestoreCPC_Drive(ContextFileRef,
                                  (tDrive*)EndianSwap32(contextP + CONTEXT_OFFSET_DRIVE_B),
                                  NativeCPC,
                                  &loadDiskImageB);
    if (ReturnCode != Restore_OK) continue;

    RESTORECPC_TRACE_SHOW_INT(8);
    
    //
    // Mem Banks
    //
    ReturnCode = RestoreCPC_MemBank(ContextFileRef,
                                    contextP,
                                    readContextP,
                                    readNativeCPCP);
    if (ReturnCode != Restore_OK) continue;
  
    RESTORECPC_TRACE_SHOW_INT(9);
    
    //
    // Drive A Disk Image
    //
    if (loadDiskImageA == cTrue)
    {
      RESTORECPC_TRACE_SHOW_INT(10);
  
      ReturnCode = RestoreCPC_DiskImage(ContextFileRef,
                                        (tDrive*)EndianSwap32(contextP + CONTEXT_OFFSET_DRIVE_A),
                                        NativeCPC);
      if (ReturnCode != Restore_OK) continue;
    }

    RESTORECPC_TRACE_SHOW_INT(11);
  
    //
    // Drive B Disk Image
    //
    if (loadDiskImageB == cTrue)
    {
      RESTORECPC_TRACE_SHOW_INT(12);

      ReturnCode = RestoreCPC_DiskImage(ContextFileRef,
                                        (tDrive*)EndianSwap32(contextP + CONTEXT_OFFSET_DRIVE_B),
                                        NativeCPC);
      if (ReturnCode != Restore_OK) continue;
    }

    RESTORECPC_TRACE_SHOW_INT(13);

    NativeCPC->RestorationPerformed = 1;
  }
  while (0);

  RESTORECPC_TRACE_SHOW_INT(14);

  if (memP != NULL)
  {
    MemPtrFree(memP);
  }

  if (readNativeCPCP != NULL)
  {
    MemPtrFree(readNativeCPCP);
  }
  
  VFSFileClose(ContextFileRef);

  RESTORECPC_TRACE_SHOW_INT(15);

  return (ReturnCode);
}
/*----------------------------------------------------------------------------*/


static tCPCRestoreReturnCode RestoreCPC_Section(FileRef ContextFileRef,
                                                tUChar* ContextP,
                                                tUChar* FileContextP,
                                                const tRestoreCPCSection* sectionP)
/***********************************************************************
 *
 *  RestoreCPC_Section
 *
 ***********************************************************************/
#undef RESTORECPC_SECTION_TRACE_ENABLED
//#define RESTORECPC_SECTION_TRACE_ENABLED

#ifdef RESTORECPC_SECTION_TRACE_ENABLED
#  define RESTORECPC_SECTION_TRACE_SHOW_INT(value) TRACE_SHOW_INT("RestoreCPC_Section", value)
#else
#  define RESTORECPC_SECTION_TRACE_SHOW_INT(value)
#endif /* RESTORECPC_SECTION_TRACE_ENABLED */
{
tReindexOperationParameters ReindexParam;
UInt32 numBytesRead;
UInt32 Size;
Err Result;

  RESTORECPC_SECTION_TRACE_SHOW_INT(1);

  // Pass through STATIC areaSize
  if (sectionP->StaticAreaSize)
  {
    Result = VFSFileSeek(ContextFileRef,
                         vfsOriginCurrent,
                         sectionP->StaticAreaSize);
#ifndef __RELEASE__
    if (Result != errNone)
      ErrDisplay("VFSFileSeek != errNone");
#endif /* __RELEASE__ */
  }
  
  RESTORECPC_SECTION_TRACE_SHOW_INT(2);

  // Read dynamic and variables areas
  Size = sectionP->Size - sectionP->StaticAreaSize;
  if (Size)
  {
    RESTORECPC_SECTION_TRACE_SHOW_INT(3);

    Result = VFSFileRead(ContextFileRef,
                         Size,
                         (void*)(ContextP + sectionP->Offset + sectionP->StaticAreaSize),
                         &numBytesRead);
#ifndef __RELEASE__
    if (Result != errNone)
      ErrDisplay("VFSFileRead != errNone");
#endif /* __RELEASE__ */
    if (numBytesRead != Size) 
    { 
      return (Bad_Data);
    }

    RESTORECPC_SECTION_TRACE_SHOW_INT(4);
    
    // Reindex dynamic area
    if (sectionP->DynamicAreaSize)
    {
      RESTORECPC_SECTION_TRACE_SHOW_INT(5);

      // Prepare reindex parameters
      ReindexParam.srcContextBase = (UInt32)FileContextP;
      ReindexParam.dstContextBase = (UInt32)ContextP;
      ReindexParam.contextSize = SIZETAB_CONTEXT;
      ReindexParam.native = cTrue;

      // Re-index dynamic area
      ReindexParam.srcAreaP = (void**)(ContextP + sectionP->Offset + sectionP->StaticAreaSize);
      ReindexParam.dstAreaP = ReindexParam.srcAreaP;
      ReindexParam.areaSize = sectionP->DynamicAreaSize;
      ReindexPointersArea(&ReindexParam);
    }
  }

  RESTORECPC_SECTION_TRACE_SHOW_INT(6);

  return (Restore_OK);
}
/*----------------------------------------------------------------------------*/


static tCPCRestoreReturnCode RestoreCPC_Drive(FileRef ContextFileRef,
                                              tDrive* NativeDriveP,
                                              tNativeCPC* NativeCPC,
                                              tBool* LoadDiskImageP)
/***********************************************************************
 *
 *  RestoreCPC_Drive
 *
 ***********************************************************************/
#undef RESTORECPC_DRIVE_TRACE_ENABLED
//#define RESTORECPC_DRIVE_TRACE_ENABLED

#ifdef RESTORECPC_DRIVE_TRACE_ENABLED
#  define RESTORECPC_DRIVE_TRACE_SHOW_INT(value) TRACE_SHOW_INT("RestoreCPC_Drive", value)
#else
#  define RESTORECPC_DRIVE_TRACE_SHOW_INT(value)
#endif /* RESTORECPC_DRIVE_TRACE_ENABLED */
{
tDrive* driveP = (tDrive*)EndianSwap32(NativeDriveP);
tDrive* readDriveP;
UInt32 size;
UInt32 numBytesRead;
Err Result;
tCPCRestoreReturnCode ReturnCode = Restore_OK;

#ifndef __RELEASE__
  if (NativeCPC == cNull)
    ErrDisplay("NativeCPC == cNull");
  if (driveP == cNull)
    ErrDisplay("driveP == cNull");
  if (LoadDiskImageP == cNull)
    ErrDisplay("LoadDiskImageP == cNull");
#endif /* __RELEASE__ */

  *LoadDiskImageP = cFalse;

  RESTORECPC_DRIVE_TRACE_SHOW_INT(1);

  // Allocate memory to read Drive data
  MemPtrNewLarge(CONTEXT_SIZE_DRIVE, (void**)&readDriveP);
  if (readDriveP == NULL) 
  { 
    return (Not_Enough_Memory);
  }

  RESTORECPC_DRIVE_TRACE_SHOW_INT(4);
    
  do
  {
    // Read drive data
    Result = VFSFileRead(ContextFileRef,
                         CONTEXT_SIZE_DRIVE,
                         (void*)readDriveP,
                         &numBytesRead);
#ifndef __RELEASE__
    if (Result != errNone)
      ErrDisplay("VFSFileRead != errNone");
#endif /* __RELEASE__ */
    if (numBytesRead != CONTEXT_SIZE_DRIVE)
    { 
    	ReturnCode = Bad_Data;
    	continue;
    }

    RESTORECPC_DRIVE_TRACE_SHOW_INT(5);
  
    if (readDriveP->dataP)
    {
      RESTORECPC_DRIVE_TRACE_SHOW_INT(6);
    
  	  MemMove(driveP,
  	          readDriveP,
  	          CONTEXT_SIZE_DRIVE);
  	        
  	  *LoadDiskImageP = cTrue;

      RESTORECPC_DRIVE_TRACE_SHOW_INT(7);
    }
  }
  while(0);

  MemPtrFree(readDriveP);
  
  RESTORECPC_DRIVE_TRACE_SHOW_INT(8);

  return ReturnCode;
}
/*----------------------------------------------------------------------------*/


static tCPCRestoreReturnCode RestoreCPC_DiskImage(FileRef ContextFileRef,
                                                  tDrive* NativeDriveP,
                                                  tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  RestoreCPC_DiskImage
 *
 ***********************************************************************/
#undef RESTORECPC_DISKIMAGE_TRACE_ENABLED
//#define RESTORECPC_DISKIMAGE_TRACE_ENABLED

#ifdef RESTORECPC_DISKIMAGE_TRACE_ENABLED
#  define RESTORECPC_DISKIMAGE_TRACE_SHOW_INT(value) TRACE_SHOW_INT("RestoreCPC_Drive", value)
#else
#  define RESTORECPC_DISKIMAGE_TRACE_SHOW_INT(value)
#endif /* RESTORECPC_DISKIMAGE_TRACE_ENABLED */
{
tDrive* driveP = (tDrive*)EndianSwap32(NativeDriveP);
tUChar* memP;
UInt32 size;
UInt32 numBytesRead;
Err Result;

  RESTORECPC_DISKIMAGE_TRACE_SHOW_INT(1);

  // Get size of disk image
  size = EndianSwap32(driveP->data_size);
  
  // Something to read...
  if (size)
  {
    RESTORECPC_DISKIMAGE_TRACE_SHOW_INT(2);

    MemPtrNewLarge(size, (void**)&memP);
    if (memP == NULL) 
    { 
      return (Not_Enough_Memory);
    }

    RESTORECPC_DISKIMAGE_TRACE_SHOW_INT(3);
    
    // Read disk image
    driveP->dataP = (tUChar*)EndianSwap32(memP);
    Result = VFSFileRead(ContextFileRef,
                         size,
                         (void*)memP,
                         &numBytesRead);
#ifndef __RELEASE__
    if (Result != errNone)
      ErrDisplay("VFSFileRead != errNone");
#endif /* __RELEASE__ */
    if (numBytesRead != size)
    { 
      return (Bad_Data);
    }

    RESTORECPC_DISKIMAGE_TRACE_SHOW_INT(4);

    // Initialize native drive structure
    if (LoadDiskImageFromMemory(NativeDriveP,
                                NativeCPC,
                                driveP->data_size,
                                driveP->dataP) != errNone)
    {
      RESTORECPC_DISKIMAGE_TRACE_SHOW_INT(5);

      EjectDisk(NativeDriveP,
                NativeCPC);
                
      SetDriveFilename(NativeDriveP,
                       NODISK_FILENAME);
    }

    RESTORECPC_DISKIMAGE_TRACE_SHOW_INT(6);
  }

  RESTORECPC_DISKIMAGE_TRACE_SHOW_INT(7);

  return (Restore_OK);
}
/*----------------------------------------------------------------------------*/


static tCPCRestoreReturnCode RestoreCPC_MemBank(FileRef ContextFileRef,
                                                tUChar* ContextP,
                                                tUChar* FileContextP,
                                                tUChar* readNativeCPCP)
/***********************************************************************
 *
 *  RestoreCPC_MemBank
 *
 ***********************************************************************/
#undef RESTORECPC_MEMBANK_TRACE_ENABLED
//#define RESTORECPC_MEMBANK_TRACE_ENABLED

#ifdef RESTORECPC_MEMBANK_TRACE_ENABLED
#  define RESTORECPC_MEMBANK_TRACE_SHOW_INT(value) TRACE_SHOW_INT("RestoreCPC_MemBank", value)
#else
#  define RESTORECPC_MEMBANK_TRACE_SHOW_INT(value)
#endif /* RESTORECPC_MEMBANK_TRACE_ENABLED */
{
tCPCRestoreReturnCode ReturnCode = Restore_OK;
Err Result;
UInt32 Loop;
UInt32 numBytesRead;
tNativeCPC* NativeCPC = (tNativeCPC*)(ContextP + CONTEXT_OFFSET_NATIVECPC);
tUChar** MemmapROM = (tUChar**)(ContextP + CONTEXT_OFFSET_MEMMAP_ROM);
tUChar** FileMemmapROM;
tUChar* FileExpansionROM;

  RESTORECPC_MEMBANK_TRACE_SHOW_INT(1);

  FileExpansionROM = ((tNativeCPC*)readNativeCPCP)->pbExpansionROM;

  do
  {
    //
    // Read MEMMAP_ROM
    //
    FileMemmapROM = (tUChar**)MemPtrNew(CONTEXT_SIZE_MEMMAP_ROM);
    if (FileMemmapROM == NULL) 
    { 
  	  ReturnCode = Not_Enough_Memory;
  	  continue;
    }

    Result = VFSFileRead(ContextFileRef, 
                         CONTEXT_SIZE_MEMMAP_ROM, 
                         (void*)FileMemmapROM, 
                         &numBytesRead);
#ifndef __RELEASE__
    if (Result != errNone)
      ErrDisplay("VFSFileRead != errNone");
#endif /* __RELEASE__ */

    if ( (numBytesRead != CONTEXT_SIZE_MEMMAP_ROM) ||
         (Result != errNone) )
    { 
      ReturnCode = Bad_Data;
      continue;
    }

    //
    // EXPANSION ROM
    //
    for (Loop=0; Loop < MEMMAP_ROM_COUNT; Loop++)
    {
  	  if (FileExpansionROM == FileMemmapROM[Loop])
  	  {
        RESTORECPC_MEMBANK_TRACE_SHOW_INT(200 + Loop);
  		  NativeCPC->pbExpansionROM = ((tUChar**)EndianSwap32(NativeCPC->memmap_ROM))[Loop];
  		  break;
  	  }
    }
  }
  while (0);
  
  if (FileMemmapROM != NULL)
  {
    MemPtrFree(FileMemmapROM);
  }

  return (ReturnCode);
}
/*----------------------------------------------------------------------------*/


tCPCSaveReturnCode WriteFile(const char* pathP, 
                             const char* filenameP, 
                             const tUChar* fileContentP,
                             UInt32 fileSize)
/***********************************************************************
 *
 *  WriteFile
 *
 ***********************************************************************/
#undef WRITEFILE_TRACE_ENABLED
//#define WRITEFILE_TRACE_ENABLED

#ifdef WRITEFILE_TRACE_ENABLED
#  define WRITEFILE_TRACE_SHOW_INT(value) TRACE_SHOW_INT("WriteFile", value)
#else
#  define WRITEFILE_TRACE_SHOW_INT(value)
#endif /* WRITEFILE_TRACE_ENABLED */
{
tCPCSaveReturnCode ReturnCode = Save_OK;
FileRef fileRef;
UInt32 numBytesWritten;
Err Result;

  WRITEFILE_TRACE_SHOW_INT(1);

  // Create result file
  Result = OpenFile(pathP,
                    filenameP,
                    &fileRef,
                    vfsModeReadWrite,
                    true);
  if (Result != errNone)
    return File_Not_Created;

  WRITEFILE_TRACE_SHOW_INT(2);

  // Write file content
  Result = VFSFileWrite(fileRef,
                        fileSize,
                        fileContentP,
                        &numBytesWritten);
#ifndef __RELEASE__
  if (Result != errNone)
    ErrDisplay("VFSFileWrite != errNone");
#endif /* __RELEASE__ */
  if (numBytesWritten != fileSize)
  {
    ReturnCode = Write_Error;

    WRITEFILE_TRACE_SHOW_INT(3);
  }

  WRITEFILE_TRACE_SHOW_INT(4);

  VFSFileClose(fileRef);
    
  WRITEFILE_TRACE_SHOW_INT(5);

  return ReturnCode;
}
/*----------------------------------------------------------------------------*/


Err PrepareScreenshot(const tUChar* pScreen,
                      const tUChar* pCPCColors,
                      const tUChar* pPalmPalette,
                      tUChar** ppScreenshot,
                      UInt32* pSize)
/***********************************************************************
 *
 *  PrepareScreenshot
 *
 ***********************************************************************/
#undef PREPARESCREENSHOT_TRACE_ENABLED
//#define PREPARESCREENSHOT_TRACE_ENABLED

#ifdef PREPARESCREENSHOT_TRACE_ENABLED
#  define PREPARESCREENSHOT_TRACE_SHOW_INT(value) TRACE_SHOW_INT("PrepareScreenshot", value)
#else
#  define PREPARESCREENSHOT_TRACE_SHOW_INT(value)
#endif /* PREPARESCREENSHOT_TRACE_ENABLED */
{
Err Result = errNone;
tUChar* pScreenshot;
tUChar* pBuffer;
tUChar* pContent;
tUChar* pPalmtoCPCColours;
UInt32 Loop;
UInt32 lineIndex;
UInt32 rowIndex;

  *ppScreenshot = NULL;
  *pSize = 0;

  PREPARESCREENSHOT_TRACE_SHOW_INT(1);

  do
  {
    MemPtrNewLarge(SCREENSHOT_SIZE, (tVoid**)&pScreenshot);
    pPalmtoCPCColours = (tUChar*)MemPtrNew(256);
    if ( (pScreenshot == NULL) ||
         (pPalmtoCPCColours == NULL) )
    {
      FrmAlert(NotEnoughMemoryAlert);
      Result = memErrNotEnoughSpace;
      continue;
    }
  
    *ppScreenshot = pBuffer = pScreenshot;
    *pSize = SCREENSHOT_SIZE;

    PREPARESCREENSHOT_TRACE_SHOW_INT(2);

    //  
    // Prepare file header
    //
    MemMove(pBuffer,
            &BitmapFileHeader,
            sizeof(tBitmapFileHeader));
    pBuffer+=sizeof(tBitmapFileHeader);

    PREPARESCREENSHOT_TRACE_SHOW_INT(3);

    //  
    // Prepare image header
    //
    MemMove(pBuffer,
            &BitmapHeader,
            sizeof(tBitmapHeader));
    pBuffer+=sizeof(tBitmapHeader);

    PREPARESCREENSHOT_TRACE_SHOW_INT(4);

    //  
    // Prepare palette
    //
    for (Loop=0; Loop < SCREENSHOT_NB_COLORS; Loop++, pCPCColors+=sizeof(colours_rgb_entry))
    {
    	*(pBuffer++) = *(pCPCColors+2); // Blue
    	*(pBuffer++) = *(pCPCColors+1); // Green
    	*(pBuffer++) = *pCPCColors;     // Red
    	*(pBuffer++) = 0;               // Reserved
    }
  
    PREPARESCREENSHOT_TRACE_SHOW_INT(5);

    //
    // Prepare image content
    //
    // Conversion from Palm Palette to CPC Palette
    for (Loop=0; Loop < SCREENSHOT_NB_COLORS; Loop++)
    {
      pPalmtoCPCColours[pPalmPalette[Loop]] = Loop;
    }

    PREPARESCREENSHOT_TRACE_SHOW_INT(6);

    // Transfer bitmap content from bottom left to top right using CPC Palette conversion
    for (lineIndex=SCREENSHOT_IMAGE_HEIGHT; lineIndex; lineIndex--)
    {
      const tUChar* lineP = pScreen + ((lineIndex + SCREENSHOT_IMAGE_HEIGHT_OFFSET) * SCREENSHOT_IMAGE_WIDTH);
      for (rowIndex=0; rowIndex < SCREENSHOT_IMAGE_WIDTH; rowIndex++)
      {
    	  *(pBuffer++) = pPalmtoCPCColours[*(lineP++)];
      }
    }
  }
  while (0);
  
  PREPARESCREENSHOT_TRACE_SHOW_INT(7);

  if (Result != errNone)
  {
  	if (pScreenshot)
  	{
  	  MemPtrFree(pScreenshot);
  	}
  }

  if (pPalmtoCPCColours)
  {
    MemPtrFree(pPalmtoCPCColours);
  }
    
  return Result;
}
/*----------------------------------------------------------------------------*/


Err ExtractExtensionFromFilename(char* filenameP,
                                 const char* extensionP)
/***********************************************************************
 *
 *  ExtractExtensionFromFilename
 *
 ***********************************************************************/
#undef EXTRACTEXTENSIONFROMFILENAME_TRACE_ENABLED
//#define EXTRACTEXTENSIONFROMFILENAME_TRACE_ENABLED

#ifdef EXTRACTEXTENSIONFROMFILENAME_TRACE_ENABLED
#  define EXTRACTEXTENSIONFROMFILENAME_TRACE_SHOW_INT(value) TRACE_SHOW_INT("ExtractExtensionFromFilename", value)
#else /* EXTRACTEXTENSIONFROMFILENAME_TRACE_ENABLED */
#  define EXTRACTEXTENSIONFROMFILENAME_TRACE_SHOW_INT(value)
#endif /* EXTRACTEXTENSIONFROMFILENAME_TRACE_ENABLED */
{
char* tempP;
char* extP;

#ifndef __RELEASE__
  if (filenameP == NULL)
    ErrDisplay("filenameP == NULL");
  if (extensionP == NULL)
    ErrDisplay("extensionP == NULL");
#endif /* __RELEASE__ */

  EXTRACTEXTENSIONFROMFILENAME_TRACE_SHOW_INT(1);

  tempP = (char*)MemPtrNew(StrLen(filenameP)+1);
  if (tempP == NULL)
  {
    FrmAlert(NotEnoughMemoryAlert);
    return memErrNotEnoughSpace;
  }

  EXTRACTEXTENSIONFROMFILENAME_TRACE_SHOW_INT(2);
  
  // Suppress extension to filename
  extP = StrStr(StrToLower(tempP,
                           filenameP),
                extensionP);
  if (extP != NULL)
  {
    *(filenameP + (extP - tempP)) = 0;
  }

  EXTRACTEXTENSIONFROMFILENAME_TRACE_SHOW_INT(3);

  MemPtrFree(tempP);

  EXTRACTEXTENSIONFROMFILENAME_TRACE_SHOW_INT(4);

  return errNone;
}
/*----------------------------------------------------------------------------*/


tUChar GenerateFilename(tChar* filenameP)
/***********************************************************************
 *
 *  GenerateFilename
 *
 ***********************************************************************/
{
DateTimeType DateTime;

#ifndef __RELEASE__
  if (filenameP == NULL)
    ErrDisplay("filenameP == NULL");
#endif /* __RELEASE__ */

  // Get Actual Date and Time
  TimSecondsToDateTime(TimGetSeconds(),
                       &DateTime);

  // Create filename
  StrPrintF(filenameP,
            "%d-%02d-%02d_%02dh%02d",
            DateTime.year,
            DateTime.month,
            DateTime.day,
            DateTime.hour,
            DateTime.minute);

  return (StrLen(filenameP));
}
/*----------------------------------------------------------------------------*/


UInt16 GetSaveFilename(char* filenameP,
                       char* extensionP,
                       char* formTitle)
/***********************************************************************
 *
 *  GetSaveFilename
 *
 ***********************************************************************/
{
FormPtr frmP;
FieldPtr fieldP;
MemHandle memH;
MemPtr memP;
UInt16 buttonID;
UInt16 filenameLength;
char* stringP;
tDIAState DIAState;

#ifndef __RELEASE__
  if (filenameP == NULL)
    ErrDisplay("filenameP == NULL");
  if (extensionP == NULL)
    ErrDisplay("extensionP == NULL");
#endif /* __RELEASE__ */

  // Create Save Form
  frmP = FrmInitForm(SaveForm);
#ifndef __RELEASE__
  if (frmP == NULL)
    ErrDisplay("frmP == NULL");
#endif /* __RELEASE__ */

  SaveDIAState(&DIAState);

  // Init Save Form
  SetDIAPolicy(frmP,
               pinInputTriggerDisabled,
               DexLibRefNum,
               0 /*pinInputAreaOpen*/);
  FrmSetTitle(frmP,
              formTitle);

  // Display extension
  if (StrLen(extensionP))
  {
    fieldP = (FieldPtr)FrmGetObjectPtr(frmP,
                                       FrmGetObjectIndex(frmP,
                                                         SaveFormExtensionField));
    FldSetTextPtr(fieldP,
                  extensionP);
  }

  // Display filename
  fieldP = (FieldPtr)FrmGetObjectPtr(frmP,
                                     FrmGetObjectIndex(frmP,
                                                       SaveFormNameField));
#ifndef __RELEASE__
  if (fieldP == NULL)
    ErrDisplay("fieldP == NULL");
#endif /* __RELEASE__ */

  FldSetMaxChars(fieldP, SIZETAB_FILENAME - 5); // 4 for Extension, 1 for terminating '\0'
  memH = MemHandleNew(SIZETAB_FILENAME);

#ifndef __RELEASE__
  if (memH == NULL)
    ErrDisplay("memH == NULL");
#endif /* __RELEASE__ */

  memP = MemHandleLock(memH);

#ifndef __RELEASE__
  if (memP == NULL)
    ErrDisplay("memP == NULL");
#endif /* __RELEASE__ */

  if (StrLen(filenameP) == 0)
  {
    GenerateFilename(filenameP);
  }

  StrNCopy(memP,
           filenameP,
           SIZETAB_FILENAME - 5);
  // Supprimer l'extension
  stringP = StrStr(memP,
                   extensionP);
  if (stringP)
  {
    *stringP = 0;
  }
  MemHandleUnlock(memH);
  FldSetTextHandle(fieldP,
                   memH);

  // Ouvrir la boite de dialogue
  buttonID = FrmDoDialog(frmP);

  // Retreive filename
  memH = FldGetTextHandle(fieldP);

#ifndef __RELEASE__
  if (memH == NULL)
    ErrDisplay("memH == NULL");
#endif /* __RELEASE__ */

  memP = MemHandleLock(memH);

#ifndef __RELEASE__
  if (memP == NULL)
    ErrDisplay("memP == NULL");
#endif /* __RELEASE__ */

  // Save
  if (buttonID == SaveFormSaveButton)
  {
    StrCopy(filenameP,
            memP);
    StrCat(filenameP,
           extensionP);

    filenameLength = StrLen(memP);
  }
  // Cancel
  else
  {
    filenameLength = 0;
  }

  // Free resources
  MemHandleUnlock(memH);
  FrmDeleteForm(frmP);

  RestoreDIAState(&DIAState);

  return filenameLength;
}
/*----------------------------------------------------------------------------*/


void SaveScreenshot(tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  SaveScreenShot
 *
 ***********************************************************************/
#undef SAVESCREENSHOT_DEBUG_ENABLE
//#define SAVESCREENSHOT_DEBUG_ENABLE
#undef SAVESCREENSHOT_TRACE_ENABLED
//#define SAVESCREENSHOT_TRACE_ENABLED

#ifdef SAVESCREENSHOT_TRACE_ENABLED
#  define SAVESCREENSHOT_TRACE_SHOW_INT(value) TRACE_SHOW_INT("SaveScreenshot", value)
#else
#  define SAVESCREENSHOT_TRACE_SHOW_INT(value)
#endif /* SAVESCREENSHOT_TRACE_ENABLED */


#undef SAVESCREENSHOT_FILE_NOT_CREATED
//#define SAVESCREENSHOT_FILE_NOT_CREATED
#undef SAVESCREENSHOT_WRITE_ERROR
//#define SAVESCREENSHOT_WRITE_ERROR
#undef SAVESCREENSHOT_SAVE_SUCCESS
//#define SAVESCREENSHOT_SAVE_SUCCESS
{
Char* filenameP = NULL;
Char* stringP = NULL;
tUChar* fileContentP = NULL;
UInt32 fileSize;
tCPCSaveReturnCode ReturnCode;

  SAVESCREENSHOT_TRACE_SHOW_INT(1);
  
  do
  {
 	  filenameP = (char*)MemPtrNew(MAX_FILE_NAME * sizeof(Char));
 	  stringP = (char*)MemPtrNew(MAX_FILE_NAME * sizeof(Char));
    if ( (filenameP == NULL) || (stringP == NULL) )
    {
      FrmAlert(NotEnoughMemoryAlert);
      continue;
    }
  
    SAVESCREENSHOT_TRACE_SHOW_INT(2);
  
  	if (IsDriveFilenameExist(NativeCPC->DriveA) == cTrue)
  	{
  		//
  		// Build filename using portion of filename and current time
  		//
  		GetDriveFilename(NativeCPC->DriveA,
  		                 filenameP);

      ExtractExtensionFromFilename(filenameP,
                                   DISK_EXTENSION);
                                   
      filenameP[12]=chrNull; // Keep the 12 first characters of filename

      // Add Number of seconds since the beginning...
      StrPrintF(stringP,
                "-%lx",
                TimGetSeconds());
      StrCat(filenameP, stringP);
  	}
  	else
  	{
      // Force filename generation
      filenameP[0] = 0;
    }

    SAVESCREENSHOT_TRACE_SHOW_INT(3);

#ifdef SAVESCREENSHOT_DEBUG_ENABLE
    TRACE_SHOW_TEXT("filenameP=", filenameP);
#endif
    		
    // Get Screenshot filename
    if (GetSaveFilename(filenameP,
                        SCREENSHOT_EXTENSION,
                        SAVE_SCREENSHOT_TITLE) == 0)
    {
    	continue;
    }

    SAVESCREENSHOT_TRACE_SHOW_INT(4);

#ifdef SAVESCREENSHOT_DEBUG_ENABLE
    TRACE_SHOW_TEXT("filenameP=", filenameP);
#endif
  
    // File already exists, Request confirmation
    if (IsFileExist(DEFAULT_SCREENSHOT_PATH,
    	              filenameP) == cTrue)
    {
    	if (FrmCustomAlert(ConfirmReplaceFileAlert,
    	                   filenameP,
    	                   "",
    	                   "") == 0)
    	{
    		continue;
    	}
    }

    SAVESCREENSHOT_TRACE_SHOW_INT(5);
    
    // Prepare file content
    if (PrepareScreenshot((const tUChar*)EndianSwap32(NativeCPC->BmpOffScreenBits),
      	                  (const tUChar*)EndianSwap32(NativeCPC->active_colours),
      	                  (const tUChar*)NativeCPC->colours,
      	                  &fileContentP,
      	                  &fileSize) != errNone)
    {
    	continue;
    }
    
    SAVESCREENSHOT_TRACE_SHOW_INT(6);
    
    // Create file
    ReturnCode = WriteFile(DEFAULT_SCREENSHOT_PATH,
      		                 filenameP,
      		                 fileContentP,
      		                 fileSize);
      		                                    
#ifdef SAVESCREENSHOT_FILE_NOT_CREATED
    ReturnCode = File_Not_Created;
#endif /* SAVESCREENSHOT_FILE_NOT_CREATED */
#ifdef SAVESCREENSHOT_WRITE_ERROR
    ReturnCode = Write_Error;
#endif /* SAVESCREENSHOT_WRITE_ERROR */
#ifdef SAVESCREENSHOT_SAVE_SUCCESS
    ReturnCode = Save_OK;
#endif /* SAVESCREENSHOT_SAVE_SUCCESS */
    if (ReturnCode == File_Not_Created)
    {
      FrmAlert(ScreenshotCreationAlert);
    }
    else if (ReturnCode == Write_Error)
    {
      FrmAlert(BadCPCSaveAlert);
    }

    SAVESCREENSHOT_TRACE_SHOW_INT(7);

    MemPtrFree(fileContentP);
  }
  while (0);
  
  if (filenameP) MemPtrFree(filenameP);
  if (stringP) MemPtrFree(stringP);

  SAVESCREENSHOT_TRACE_SHOW_INT(8);
}
/*----------------------------------------------------------------------------*/


void SaveSession(tNativeCPC* NativeCPC,
                 tUChar* contextP,
                 char* sessionFilenameP)
/***********************************************************************
 *
 *  SaveSession
 *
 ***********************************************************************/
#undef SAVESESSION_FILE_NOT_CREATED
//#define SAVESESSION_FILE_NOT_CREATED
#undef SAVESESSION_WRITE_ERROR
//#define SAVESESSION_WRITE_ERROR
#undef SAVESESSION_SAVE_SUCCESS
//#define SAVESESSION_SAVE_SUCCESS
{
char* filenameP;
tCPCSaveReturnCode ReturnCode;

 	filenameP = (char*)MemPtrNew(MAX_FILE_NAME);
  if (filenameP == NULL)
  {
    FrmAlert(NotEnoughMemoryAlert);
    return;
  }

  // Get last session filename
  StrCopy(filenameP,
          sessionFilenameP);
  
  do
  {
  	if (IsDriveFilenameExist(NativeCPC->DriveA) == cTrue)
  	{
  		// Get Disk A filename as default session filename
  		GetDriveFilename(NativeCPC->DriveA,
  		                 filenameP);

      ExtractExtensionFromFilename(filenameP,
                                   DISK_EXTENSION);
  	}
  	else
  	{
      // Force filename generation
      filenameP[0] = 0;
    }

    // Get Screenshot filename
    if (GetSaveFilename(filenameP,
                        CONTEXT_EXTENSION,
                        SAVE_SESSION_TITLE) == 0)
    {
    	continue;
    }

    // File already exists, Request confirmation
    if (IsFileExist(DEFAULT_CAPRICE_PATH,
    	              filenameP) == cTrue)
    {
    	if (FrmCustomAlert(ConfirmReplaceFileAlert,
    	                   filenameP,
    	                   "",
    	                   "") == 0)
    	{
    		continue;
    	}
    }
    
    // Save session filename
    StrCopy(sessionFilenameP,
            filenameP);

    // Save session
    ReturnCode = SaveCPC(DEFAULT_CAPRICE_PATH,
                         sessionFilenameP,
                         contextP);

#ifdef SAVESESSION_FILE_NOT_CREATED
    ReturnCode = File_Not_Created;
#endif /* SAVESESSION_FILE_NOT_CREATED */
#ifdef SAVESESSION_WRITE_ERROR
    ReturnCode = Write_Error;
#endif /* SAVESESSION_WRITE_ERROR */
#ifdef SAVESESSION_SAVE_SUCCESS
    ReturnCode = Save_OK;
#endif /* SAVESESSION_SAVE_SUCCESS */

    if (ReturnCode == File_Not_Created)
    {
      FrmAlert(CPCSaveCreationAlert);
    }
    else if (ReturnCode == Write_Error)
    {
      FrmAlert(BadCPCSaveAlert);
    }
  }
  while (0);
  
  MemPtrFree(filenameP);
}
/*----------------------------------------------------------------------------*/


UInt32 DiskAutostart(tNativeCPC* NativeCPC,
                     tDrive* nativedriveP,
                     tUChar* commandP)
/***********************************************************************
 *
 *  DiskAutostart
 *
 ***********************************************************************/
{
tDiskOperation* DiskOperationP;
tAutoStartReturnCode returnCode;
Err Result;

  DiskOperationP = (tDiskOperation*)(EndianSwap32(NativeCPC->contextP) + CONTEXT_OFFSET_DISKOPERATION);

  DiskOperationP->NativeCPC = NativeCPC;
  DiskOperationP->Drive = (tDrive*)EndianSwap32(nativedriveP);
  DiskOperationP->Param = (tVoid*)commandP;

  returnCode = (tAutoStartReturnCode)PceNativeCallResource(Native_AutoStart,
#ifdef __SIMU__
                                                           "Native_AutoStart.dll\0PNOMain",
#endif /* __SIMU__ */
                                                           DiskOperationP);

  return (UInt32)returnCode;
}
/*----------------------------------------------------------------------------*/
