/*
    CaPriCe for Palm OS - Amstrad CPC 464/664/6128 emulator for Palm devices
    Copyright (C) 2009  Frdric Coste

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <PalmOS.h>
#include <ByteOrderUtils.h>
#include "..\Palm\PalmDisplayExtent.h"

#include "CaPrice.h"
#include "Display.h"
#include "CPC.h"
#include "Resources.h"
#include "Forms.h"
#include "Trace.h"
#include "Routines.h"
#include "Keyboard.h"


//===================
// PATCH begin
#ifdef _PATCH_ENABLE

#endif /* _PATCH_ENABLE */
// PATCH end
//===================


#define FILENAME_MAX_PIXELWIDTH         220

#define Y_OFFSET_EMULATOR_PANEL_320     SCREEN_CPC_HEIGHT
#define Y_OFFSET_SUB_PANEL_320          (Y_OFFSET_EMULATOR_PANEL_320 + Y_SIZE_EMULATOR_PANEL)
#define Y_OFFSET_KEYBOARD_PANEL_320     (Y_OFFSET_SUB_PANEL_320 + Y_SIZE_SUB_PANEL)

#define Y_OFFSET_KEYBOARD_PANEL_480     SCREEN_CPC_HEIGHT
#define Y_OFFSET_EMULATOR_PANEL_480     (Y_OFFSET_KEYBOARD_PANEL_480 + Y_SIZE_KEYBOARD_PANEL)
#define Y_OFFSET_SUB_PANEL_480          (Y_OFFSET_EMULATOR_PANEL_480 + Y_SIZE_EMULATOR_PANEL)

#define Y_OFFSET_KEYBOARD_PANEL_FS      320

#define X_AUTO_TOGGLE_SPEED_BITMAP      230
#define Y_AUTO_TOGGLE_SPEED_BITMAP      204

#define X_SOUND_VOLUME_ADJUST_BITMAP    190
#define Y_SOUND_VOLUME_ADJUST_BITMAP    198
#define X_SOUND_VOLUME_ADJUST_MESSAGE   225
#define Y_SOUND_VOLUME_ADJUST_MESSAGE   211

#define X_KEY_STATUS_BITMAP             290
#define Y_SHIFT_STATUS_BITMAP           210
#define Y_CTRL_STATUS_BITMAP            220

#define X_DISK_BITMAP                   280
#define Y_DISK_BITMAP                   10

#define X_TRACE_DISPLAY_PC              8
#define Y_TRACE_INTERVAL                20
#define Y_TRACE_DISPLAY_TRACE           0
#define Y_TRACE_DISPLAY_OPCODE          (Y_TRACE_DISPLAY_TRACE + Y_TRACE_INTERVAL)
#define Y_TRACE_DISPLAY_PC              (Y_TRACE_DISPLAY_OPCODE + Y_TRACE_INTERVAL)
#define Y_TRACE_DISPLAY_REGISTERS       (Y_TRACE_DISPLAY_PC + Y_TRACE_INTERVAL)
#define Y_TRACE_DISPLAY_BREAKPOINT      (Y_TRACE_DISPLAY_REGISTERS + Y_TRACE_INTERVAL)

#define X_DEBUG_DISPLAY                 (OFFSCREEN_OFFSET_X + 160)
#define Y_DEBUG_INTERVAL                Y_TRACE_INTERVAL
#define Y_DEBUG_DISPLAY_SAMPLES         OFFSCREEN_OFFSET_Y
#define Y_DEBUG_DISPLAY_CB              (Y_DEBUG_DISPLAY_SAMPLES + Y_DEBUG_INTERVAL)
#define Y_DEBUG_DISPLAY_KEY             (Y_DEBUG_DISPLAY_CB + Y_DEBUG_INTERVAL)
#define Y_DEBUG_DISPLAY_KEY_MASK        (Y_DEBUG_DISPLAY_KEY + Y_DEBUG_INTERVAL)
#define Y_DEBUG_DISPLAY_PRESSED_KEY     (Y_DEBUG_DISPLAY_KEY_MASK + Y_DEBUG_INTERVAL)
#define Y_DEBUG_DISPLAY_KEYBOARD        (Y_DEBUG_DISPLAY_PRESSED_KEY + Y_DEBUG_INTERVAL)

#define X_FPS_DISPLAY                   OFFSCREEN_OFFSET_X
#define Y_FPS_DISPLAY                   (SCREEN_CPC_HEIGHT_FS - Y_DEBUG_INTERVAL)


static tResult DisplayOffscreenKey(tKey* keyP) SECTION_DISPLAY;
static tResult DisplayKeyboard(tKeyboard* keyboardP,
                               DisplayBackgroundFunctionPtr displayBackgroundP,
                               IndexedColorType textColor,
                               Coord YSize,
                               Coord YBase) SECTION_DISPLAY;
static void Display464KeyboardBackground(void) SECTION_DISPLAY;
static void Display664KeyboardBackground(void) SECTION_DISPLAY;
static void Display6128KeyboardBackground(void) SECTION_DISPLAY;
static void DisplayMiniKeyboardBackground(void) SECTION_DISPLAY;
static tResult DisplayDrivesPanel(tVoid) SECTION_DISPLAY;
static tResult DisplaySubCommandPanel(tVoid) SECTION_DISPLAY;


void PrepareLandscapeScreenCopy(tNativeCPC* NativeCPC,
                                UInt16 LibraryRef)
/***********************************************************************
 *
 *  PrepareLandscapeScreenCopy
 *
 ***********************************************************************/
{
BitmapType* OffscreenBitmap = WinGetBitmap(OffScreen);
UInt32* OffscreenBits;
UInt32* DisplayBits;
Coord OffscreenWidth;
Coord ActiveWidth;
Coord ActiveHeight;

#ifndef __RELEASE__
  if (NativeCPC == NULL)
    ErrDisplay("NativeCPC == NULL");
  if (OffscreenBitmap == NULL)
    ErrDisplay("OffscreenBitmap == NULL");
#endif /* __RELEASE__ */

  if (!LibraryRef)
    return;

  OffscreenBits = (UInt32*)BmpGetBits(OffscreenBitmap);
  DisplayBits = (UInt32*)DexGetDisplayAddress(LibraryRef);

  // Get bitmaps dimensions
  BmpGetDimensions(OffscreenBitmap,
                   &OffscreenWidth,
                   NULL,
                   NULL);
  DexGetDisplayDimensions(LibraryRef,
                          &ActiveWidth,
                          &ActiveHeight,
                          NULL);

  // Prepare Copy parameters
  NativeCPC->OffscreenStartBits = (tULong*)((tULong)OffscreenBits + (OFFSCREEN_OFFSET_Y_FS * (tULong)OffscreenWidth));
  NativeCPC->OffscreenCopyHeight = CPC_VISIBLE_SCR_HEIGHT;
  NativeCPC->OffscreenCopyWidth = CPC_SCR_WIDTH;
  NativeCPC->OffscreenAlignGap = 0;
  
  NativeCPC->OnscreenStartBits  = (tULong*)((tULong)DisplayBits + ONSCREEN_OFFSET_LANDSCAPE_Y);
  NativeCPC->OnscreenStartBits += ((((tULong)ActiveHeight - (tULong)OffscreenWidth) / 2) + (tULong)OffscreenWidth) * (tULong)ActiveWidth / 4 /* tULong* */;
  NativeCPC->OnscreenPixelGap = (tULong)ActiveWidth;
  NativeCPC->OnscreenAlignGap = ((tULong)ActiveWidth * (tULong)OffscreenWidth) + 1;

  // Endianness conversion
  NativeCPC->OffscreenStartBits = (tULong*)EndianSwap32(NativeCPC->OffscreenStartBits);
  NativeCPC->OffscreenCopyHeight = EndianSwap32(NativeCPC->OffscreenCopyHeight);
  NativeCPC->OffscreenCopyWidth = EndianSwap32(NativeCPC->OffscreenCopyWidth);
  NativeCPC->OffscreenAlignGap = EndianSwap32(NativeCPC->OffscreenAlignGap);
  NativeCPC->OnscreenStartBits = (tULong*)EndianSwap32(NativeCPC->OnscreenStartBits);
  NativeCPC->OnscreenPixelGap = EndianSwap32(NativeCPC->OnscreenPixelGap);
  NativeCPC->OnscreenAlignGap = EndianSwap32(NativeCPC->OnscreenAlignGap);
}
/*----------------------------------------------------------------------------*/


tResult DisplayCPCScreen(const tUChar offsetX,
                         const tUChar offsetY)
/***********************************************************************
 *
 *  DisplayCPCScreen
 *
 ***********************************************************************/
{
RectangleType Source;

#ifndef __RELEASE__
  if (WinGetDrawWindow() == NULL)
    ErrDisplay("WinGetDrawWindow() == NULL");
#endif /* __RELEASE__ */

  if (LandscapeActive && prefP->Display320x480)
  {
#ifndef __RELEASE__

    RctSetRectangle(&Source,
                    0,
                    OFFSCREEN_OFFSET_Y_FS,
                    CPC_SCR_WIDTH,
                    CPC_VISIBLE_SCR_HEIGHT);
    WinCopyRectangle(OffScreen,
                     NULL,
                     &Source,
                     ONSCREEN_OFFSET_LANDSCAPE_X,
                     ONSCREEN_OFFSET_LANDSCAPE_Y,
                     winPaint);
                     
#else /* __RELEASE__ */

  PceNativeCallResource(Native_WinCopyScreen,
#ifdef __SIMU__
                        "Native_WinCopyScreen.dll\0PNOMain",
#endif /* __SIMU__ */
                        NativeCPC);
                        
  NOT_USED(offsetX);
  NOT_USED(offsetY);
                        
#endif /* __RELEASE__ */
  }
  else
  {
    if (!FullscreenActive)
    {
      RctSetRectangle(&Source,
                      offsetX,
                      offsetY,
                      SCREEN_CPC_WIDTH,
                      SCREEN_CPC_HEIGHT);
      WinCopyRectangle(OffScreen,
                       NULL,
                       &Source,
                       ONSCREEN_OFFSET_X,
                       ONSCREEN_OFFSET_Y,
                       winPaint);
    }
    else
    {
      RctSetRectangle(&Source, 
                      offsetX, // Apply X offset also in fullscreen
                      OFFSCREEN_OFFSET_Y_FS,
                      SCREEN_CPC_WIDTH_FS,
                      SCREEN_CPC_HEIGHT_FS);
      WinCopyRectangle(OffScreen,
                       NULL,
                       &Source,
                       ONSCREEN_OFFSET_X_FS,
                       ONSCREEN_OFFSET_Y_FS,
                       winPaint);
    }
  }
  
  return ResultSucceed;
}
/*----------------------------------------------------------------------------*/


tVoid ClearFullscreen(tVoid)
/***********************************************************************
 *
 *  ClearFullscreen
 *
 ***********************************************************************/
{
IndexedColorType oldBackColor;
RectangleType Rectangle;
Coord extentX, extentY;

#ifndef __RELEASE__
  if (WinGetDrawWindow() == NULL)
    ErrDisplay("WinGetDrawWindow() == NULL");
#endif /* __RELEASE__ */

  oldBackColor = WinSetBackColor(colorIndexP[COLORINDEX_BLACK]);

  WinGetDisplayExtent(&extentX,
                      &extentY);
  RctSetRectangle(&Rectangle,
                  0,
                  0,
                  (LandscapeActive && prefP->Display320x480) ? extentX : SCREEN_PALM_WIDTH,
                  (LandscapeActive && prefP->Display320x480) ? extentY : SCREEN_PALM_HEIGHT_FS);

  WinFillRectangle(&Rectangle,
                   0 /* Black*/);

  WinSetBackColor(oldBackColor);
}
/*----------------------------------------------------------------------------*/


tVoid DisplayStartupScreen(tVoid)
/***********************************************************************
 *
 *  DisplayStartupScreen
 *
 ***********************************************************************/
{
IndexedColorType oldBackColor;
IndexedColorType oldTextColor;
RectangleType Rectangle;
MemHandle resH;
BitmapPtr resP;
FontID oldFont;
UInt8 Inc;
char* TextCapriceAP[] =
{
  " CCC    A   PPPP  RRRR  III  CCC  EEEEE",
  "C   C  A A  P   P R   R  I  C   C E    ",
  "C     A   A P   P R   R  I  C     E    ",
  "C     AAAAA PPPP  RRRR   I  C     EEEE ",
  "C     A   A P     R R    I  C     E    ",
  "C   C A   A P     R  R   I  C   C E    ",
  " CCC  A   A P     R   R III  CCC  EEEEE",
};
#define NB_CAPRICE_TEXT         sizeof(TextCapriceAP) / sizeof(TextCapriceAP[0])
char* TextInitAP = "Initializing...";
char* TextHelpAP = "Try \"About...\" for Help !!";
    

#ifndef __RELEASE__
  if (WinGetDrawWindow() == NULL) 
    ErrDisplay("WinGetDrawWindow() == NULL");
#endif /* __RELEASE__ */

  oldBackColor = WinSetBackColor(colorIndexP[COLORINDEX_DARK_BLUE]);
  oldTextColor = WinSetTextColor(colorIndexP[COLORINDEX_YELLOW]);

  RctSetRectangle(&Rectangle,
                  ONSCREEN_OFFSET_X,
                  ONSCREEN_OFFSET_Y,
                  SCREEN_CPC_WIDTH,
                  SCREEN_CPC_HEIGHT);
  WinFillRectangle(&Rectangle, 0);
  
  // Get CPC Font
  resH = DmGetResource(FONT_ID_TYPE,
                       FONT_ID_CPC);
#ifndef __RELEASE__
  if (resH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */
  resP = MemHandleLock(resH);
#ifndef __RELEASE__
  if (resP == NULL)
    ErrDisplay("resP == NULL");
#endif /* __RELEASE__ */

  if (FntDefineFont(FontCPC,
                    (FontPtr)resP) == errNone)
  {
    oldFont = FntSetFont(FontCPC);
  
    // CAPRICE Title    
    for (Inc=0; Inc < NB_CAPRICE_TEXT; Inc++)
    {
      WinPaintChars(TextCapriceAP[Inc],
                    StrLen(TextCapriceAP[Inc]),
                    4,
                    24 + ((Inc)*8));
    }

    // Initialising...
    WinPaintChars(TextInitAP,
                  StrLen(TextInitAP),
                  16,
                  120);
  
    // Help
    WinPaintChars(TextHelpAP,
                  StrLen(TextHelpAP),
                  16,
                  210);
  
    FntSetFont(oldFont);
  }
  
  WinSetTextColor(oldTextColor);
  WinSetBackColor(oldBackColor);
  
  MemHandleUnlock(resH);
  DmReleaseResource(resH);
}
/*----------------------------------------------------------------------------*/


tVoid DisplayShutdownScreen(tVoid)
/***********************************************************************
 *
 *  DisplayShutdownScreen
 *
 ***********************************************************************/
{
IndexedColorType oldBackColor;
IndexedColorType oldTextColor;
RectangleType Rectangle;
MemHandle resH;
BitmapPtr resP;
FontID oldFont;
UInt8 Inc;
WinHandle oldDrawWindowH;
char* TextInitAP = "Shutting down...";
Coord extentX, extentY;

#ifndef __RELEASE__
  if (WinGetDrawWindow() == NULL)
    ErrDisplay("WinGetDrawWindow() == NULL");
#endif /* __RELEASE__ */

  oldBackColor = WinSetBackColor(colorIndexP[COLORINDEX_DARK_BLUE]);

  if (LandscapeActive)
  {
    WinGetDisplayExtent(&extentX,
                        &extentY);
    RctSetRectangle(&Rectangle,
                    0,
                    0,
                    (LandscapeActive && prefP->Display320x480) ? extentX : SCREEN_PALM_WIDTH,
                    (LandscapeActive && prefP->Display320x480) ? extentY : SCREEN_PALM_HEIGHT_FS);
  }
  else if (!FullscreenActive)
  {
    RctSetRectangle(&Rectangle,
                    ONSCREEN_OFFSET_X,
                    ONSCREEN_OFFSET_Y,
                    SCREEN_CPC_WIDTH,
                    SCREEN_CPC_HEIGHT);
  }
  else
  {
    RctSetRectangle(&Rectangle,
                    ONSCREEN_OFFSET_X_FS,
                    ONSCREEN_OFFSET_Y_FS,
                    SCREEN_CPC_WIDTH_FS,
                    SCREEN_CPC_HEIGHT_FS);
  }
  
  WinFillRectangle(&Rectangle, 0);
  
  // Get CPC Font
  resH = DmGetResource(FONT_ID_TYPE,
                       FONT_ID_CPC);
#ifndef __RELEASE__
  if (resH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */
  resP = MemHandleLock(resH);
#ifndef __RELEASE__
  if (resP == NULL)
    ErrDisplay("resP == NULL");
#endif /* __RELEASE__ */

  oldTextColor = WinSetTextColor(colorIndexP[COLORINDEX_YELLOW]);

  if (FntDefineFont(FontCPC,
                    (FontPtr)resP) == errNone)
  {
    oldFont = FntSetFont(FontCPC);
  
    // Shutting down...
    WinPaintChars(TextInitAP,
                  StrLen(TextInitAP),
                  16,
                  120);
  
    FntSetFont(oldFont);
  }
  
  WinSetTextColor(oldTextColor);
  WinSetBackColor(oldBackColor);
  
  MemHandleUnlock(resH);
  DmReleaseResource(resH);
}
/*----------------------------------------------------------------------------*/


tResult DisplayEmulatorCommandPanel(tVoid)
/***********************************************************************
 *
 *  DisplayEmulatorCommandPanel
 *
 ***********************************************************************/
{
MemHandle resH;
BitmapPtr resP;
UInt8 Loop;
tKey* keyP;
WinHandle oldDrawWindow;
WinHandle DrawWindow;
UInt16 Error;

#ifndef __RELEASE__
  if (WinGetDrawWindow() == NULL)
    ErrDisplay("WinGetDrawWindow() == NULL");
#endif /* __RELEASE__ */

  // Offscreen window to prepare bitmap of panel
  DrawWindow = WinCreateOffscreenWindow(SCREEN_CPC_WIDTH,
                                        Y_SIZE_EMULATOR_PANEL,
                                        nativeFormat,
                                        &Error);
  if (DrawWindow == NULL)
    return ResultFailed;
    
  oldDrawWindow = WinSetDrawWindow(DrawWindow);

  // Display background Emulator Buttons Bar
  resH = DmGetResource(bitmapRsc,
                       ButtonsAreaBitmap);
#ifndef __RELEASE__
  if (resH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */
  resP = MemHandleLock(resH);
#ifndef __RELEASE__
  if (resP == NULL)
    ErrDisplay("resP == NULL");
#endif /* __RELEASE__ */
  WinDrawBitmap(resP, 0, 0);
  MemHandleUnlock(resH);
  DmReleaseResource(resH);

  // Display emulator command buttons
  keyP = EmulatorKeys;
#ifndef __RELEASE__
  if (keyP == NULL)
    ErrDisplay("keyP == NULL");
#endif /* __RELEASE__ */

  for (Loop=0; Loop < NB_EMULATORS_KEYS; Loop++)
  { 
    DisplayOffscreenKey(keyP++);
  }

  // Copy Offscreen bitmap to main screen
  WinSetDrawWindow(oldDrawWindow);
  WinDrawBitmap(WinGetBitmap(DrawWindow),
                0,
                GetYBaseEmulatorPanel());
  WinDeleteWindow(DrawWindow,
                  false);
  
  return ResultSucceed;
}
/*----------------------------------------------------------------------------*/


static tResult DisplayDrivesPanel(tVoid)
/***********************************************************************
 *
 *  DisplayDrivesPanel
 *
 ***********************************************************************/
{
MemHandle resH;
BitmapPtr resP;
FontID oldFont;
WinDrawOperation oldDrawOperation;
IndexedColorType oldBackColor;
Int16 MaxWidth;
Int16 TruncWidth;
Boolean StringNotTruncated;
UInt8 Loop;
char* filenameP;
char* tempP;
Coord y_base;

#ifndef __RELEASE__
  if (WinGetDrawWindow() == NULL)
    ErrDisplay("WinGetDrawWindow() == NULL");
#endif /* __RELEASE__ */

  filenameP = MemPtrNew(SIZETAB_FILENAME+1);
  if (filenameP == NULL)
  {
    FrmAlert(NotEnoughMemoryAlert);
    return ResultFailed;
  }

  // Display background Drives Bar
  resH = DmGetResource(bitmapRsc,
                       DrivesAreaBitmap);
#ifndef __RELEASE__
  if (resH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */
  resP = MemHandleLock(resH);
#ifndef __RELEASE__
  if (resP == NULL)
    ErrDisplay("resP == NULL");
#endif /* __RELEASE__ */

  y_base = GetYBaseSubPanel();
    
  WinDrawBitmap(resP, 0, y_base);
  MemHandleUnlock(resH);
  DmReleaseResource(resH);
  
  // Save context
  oldDrawOperation = WinSetDrawMode(winMask);
  oldFont = FntSetFont(stdFont);

  oldBackColor = WinSetBackColor(colorIndexP[COLORINDEX_BLUE]);
  
  // Affichage du nom de l'image de la face A
  if (NativeCPC && IsDriveFilenameExist(NativeCPC->DriveA))
  {
    GetDriveFilename(NativeCPC->DriveA,
                     filenameP);
    
    ExtractExtensionFromFilename(filenameP,
                                 DISK_EXTENSION);

    // Truncate filename to fit title bitmap
    MaxWidth = FILENAME_MAX_PIXELWIDTH;
    TruncWidth = StrLen(filenameP);
    FntCharsInWidth(filenameP,
                    &MaxWidth,
                    &TruncWidth,
                    &StringNotTruncated);
    if (StringNotTruncated == false)
    {
      filenameP[TruncWidth] = 0;
      filenameP[TruncWidth-1] = '.';
      filenameP[TruncWidth-2] = '.';
      filenameP[TruncWidth-3] = '.';
    }
  
    // Display disk images titles
    WinPaintChars(filenameP,
                  StrLen(filenameP),
                  68,
                  y_base+0);
  }

  // Affichage du nom de l'image de la face B
  if (NativeCPC && IsDriveFilenameExist(NativeCPC->DriveB))
  {
    GetDriveFilename(NativeCPC->DriveB,
                     filenameP);

    ExtractExtensionFromFilename(filenameP,
                                 DISK_EXTENSION);

      
    // Truncate filename to fit title bitmap
    MaxWidth = FILENAME_MAX_PIXELWIDTH;
    TruncWidth = StrLen(filenameP);
    FntCharsInWidth(filenameP,
                    &MaxWidth,
                    &TruncWidth,
                    &StringNotTruncated);
    if (StringNotTruncated == false)
    {
      filenameP[TruncWidth] = 0;
      filenameP[TruncWidth-1] = '.';
      filenameP[TruncWidth-2] = '.';
      filenameP[TruncWidth-3] = '.';
    }
  
    // Display disk images titles
    WinPaintChars(filenameP,
                  StrLen(filenameP),
                  68,
                  y_base+18);
  }

  // Restore context
  FntSetFont(oldFont);
  WinSetBackColor(oldBackColor);
  WinSetDrawMode(oldDrawOperation);

  // Display drive command buttons
  DisplayKey(&DriveKeys[KEYINDEX_SWAPDISK]);
  DisplayKey(&DriveKeys[KEYINDEX_EJECTDISK]);
  
  // Display FDC Led
  if (NativeCPC != NULL)
  {
    DisplayOnScreenFDCLed(NativeCPC->drive_led);
  }

  MemPtrFree(filenameP);

  return ResultSucceed;
}
/*----------------------------------------------------------------------------*/


static tResult DisplaySubCommandPanel(tVoid)
/***********************************************************************
 *
 *  DisplaySubCommandPanel
 *
 ***********************************************************************/
{
MemHandle resH;
BitmapPtr resP;
UInt8 Loop;
tKey* keyP;
WinHandle oldDrawWindow;
WinHandle DrawWindow;
UInt16 Error;

#ifndef __RELEASE__
  if (WinGetDrawWindow() == NULL)
    ErrDisplay("WinGetDrawWindow() == NULL");
#endif /* __RELEASE__ */

  // Offscreen window to prepare bitmap of panel
  DrawWindow = WinCreateOffscreenWindow(SCREEN_CPC_WIDTH,
                                        Y_SIZE_SUB_PANEL,
                                        nativeFormat,
                                        &Error);
  if (DrawWindow == NULL)
    return ResultFailed;
    
  oldDrawWindow = WinSetDrawWindow(DrawWindow);

  // Display background Sub Buttons Bar
  resH = DmGetResource(bitmapRsc,
                       SubButtonsAreaBitmap);
#ifndef __RELEASE__
  if (resH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */
  resP = MemHandleLock(resH);
#ifndef __RELEASE__
  if (resP == NULL)
    ErrDisplay("resP == NULL");
#endif /* __RELEASE__ */
  WinDrawBitmap(resP, 0, 0);
  MemHandleUnlock(resH);
  DmReleaseResource(resH);

  // Display emulator command buttons
  keyP = EmulatorKeysPanelP;
#ifndef __RELEASE__
  if (keyP == NULL)
    ErrDisplay("keyP == NULL");
#endif /* __RELEASE__ */

  for (Loop=0; Loop < NbEmulatorKeys; Loop++)
  { 
    DisplayOffscreenKey(keyP++);
  }

  // Copy Offscreen bitmap to main screen
  WinSetDrawWindow(oldDrawWindow);
  WinDrawBitmap(WinGetBitmap(DrawWindow),
                0,
                GetYBaseSubPanel());
  WinDeleteWindow(DrawWindow,
                  false);
  
  return ResultSucceed;
}
/*----------------------------------------------------------------------------*/


tResult DisplaySubPanel(UInt8 SubButtonsActive)
/***********************************************************************
 *
 *  DisplaySubPanel
 *
 ***********************************************************************/
{
  if (!SubButtonsActive)
  {
    DisplayDrivesPanel();
  }
  else
  {
    DisplaySubCommandPanel();
  }
}
/*----------------------------------------------------------------------------*/


tResult DisplayKey(tKey* keyP)
/***********************************************************************
 *
 *  DisplayKey
 *
 ***********************************************************************/
{
DmResID resID;
MemHandle resH;
BitmapPtr resP;

#ifndef __RELEASE__
  if (WinGetDrawWindow() == NULL)
    ErrDisplay("WinGetDrawWindow() == NULL");
#endif /* __RELEASE__ */

  resID = keyP->KeyStatus == KeyReleased ? keyP->settingP->resKeyUp : keyP->settingP->resKeyDown;

  if (resID == NULL)
    return ResultFailed;

  resH = DmGetResource(bitmapRsc,
                       resID);

#ifndef __RELEASE__
  if (resH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */

  resP = MemHandleLock(resH);

#ifndef __RELEASE__
  if (resP == NULL)
    ErrDisplay("resP == NULL");
#endif /* __RELEASE__ */

  WinDrawBitmap(resP,
                keyP->Left,
                keyP->Top);
  
  MemHandleUnlock(resH);
  DmReleaseResource(resH);

  return ResultSucceed;
}
/*----------------------------------------------------------------------------*/


static tResult DisplayOffscreenKey(tKey* keyP)
/***********************************************************************
 *
 *  DisplayOffscreenKey
 *
 ***********************************************************************/
{
DmResID resID;
MemHandle resH;
BitmapPtr resP;

#ifndef __RELEASE__
  if (WinGetDrawWindow() == NULL)
    ErrDisplay("WinGetDrawWindow() == NULL");
#endif /* __RELEASE__ */

  resID = keyP->KeyStatus == KeyReleased ? keyP->settingP->resKeyUp : keyP->settingP->resKeyDown;

  if (resID == NULL)
    return ResultFailed;

  resH = DmGetResource(bitmapRsc,
                       resID);

#ifndef __RELEASE__
  if (resH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */

  resP = MemHandleLock(resH);

#ifndef __RELEASE__
  if (resP == NULL)
    ErrDisplay("resP == NULL");
#endif /* __RELEASE__ */

  WinDrawBitmap(resP,
                keyP->settingP->Left,
                keyP->settingP->Top);
  
  MemHandleUnlock(resH);
  DmReleaseResource(resH);

  return ResultSucceed;
}
/*----------------------------------------------------------------------------*/


tResult DisplayOnScreenFDCLed(tUChar led)
/***********************************************************************
 *
 *  DisplayOnScreenFDCLed
 *
 ***********************************************************************/
{
MemHandle resH;
BitmapPtr resP;

#ifndef __RELEASE__
  if (WinGetDrawWindow() == NULL)
    ErrDisplay("WinGetDrawWindow() == NULL");
#endif /* __RELEASE__ */

    
  resH = DmGetResource(bitmapRsc,
                       led ? FDC_Led_ON : FDC_Led_OFF);

#ifndef __RELEASE__
  if (resH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */

  resP = MemHandleLock(resH);

#ifndef __RELEASE__
  if (resP == NULL)
    ErrDisplay("resP == NULL");
#endif /* __RELEASE__ */
  
  WinDrawBitmap(resP,
                6,
                GetYBaseSubPanel() + 5);
  
  MemHandleUnlock(resH);
  DmReleaseResource(resH);
  
  return ResultSucceed;
}
/*----------------------------------------------------------------------------*/


tResult DisplayOffScreenDisk(tVoid)
/***********************************************************************
 *
 *  DisplayOffScreenDisk
 *
 ***********************************************************************/
{
MemHandle resH;
BitmapPtr resP;
WinHandle oldDrawWindow;

  resH = DmGetResource(bitmapRsc,
                       Disk_Active);
  resP = MemHandleLock(resH);

  // Save context
  oldDrawWindow = WinSetDrawWindow(OffScreen);

  WinDrawBitmap(resP, 
                prefP->ScreenDisplayOffsetX + X_DISK_BITMAP, 
                prefP->ScreenDisplayOffsetY + Y_DISK_BITMAP);

  MemHandleUnlock(resH);
  DmReleaseResource(resH);

  // Restore context
  WinSetDrawWindow(oldDrawWindow);
}
/*----------------------------------------------------------------------------*/


tVoid DisplayOffScreenEmulatorSpeed(tULong percentage)
/***********************************************************************
 *
 *  DisplayOffScreenEmulatorSpeed
 *
 ***********************************************************************/
{
char string[20];
IndexedColorType oldColor;
FontID oldFont;
WinDrawOperation oldDrawMode;
WinHandle oldDrawWindow;

#ifndef __RELEASE__
  if (WinGetDrawWindow() == NULL)
    ErrDisplay("WinGetDrawWindow() == NULL");
#endif /* __RELEASE__ */

  if (!percentage)
    return;

  // Save context
  oldDrawWindow = WinSetDrawWindow(OffScreen);
  oldDrawMode = WinSetDrawMode(winMask);
  oldFont = FntSetFont(stdFont);

  oldColor = WinSetBackColor(colorIndexP[COLORINDEX_WHITE]);
  
  // Display text
  StrPrintF(string,
            "%ld %%",
            percentage);
  WinPaintChars(string, 
                StrLen(string), 
                prefP->ScreenDisplayOffsetX + MESSAGE_START_X, 
                prefP->ScreenDisplayOffsetY + MESSAGE_START_Y);

  // Restore context
  FntSetFont(oldFont);
  WinSetBackColor(oldColor);
  WinSetDrawMode(oldDrawMode);
  WinSetDrawWindow(oldDrawWindow);
}
/*----------------------------------------------------------------------------*/


tVoid DisplayOffscreenMessage(const char* messageP)
/***********************************************************************
 *
 *  DisplayOffscreenMessage
 *
 ***********************************************************************/
{
IndexedColorType oldColor;
FontID oldFont;
WinDrawOperation oldDrawMode;
WinHandle oldDrawWindow;

#ifndef __RELEASE__
  if (WinGetDrawWindow() == NULL)
    ErrDisplay("WinGetDrawWindow() == NULL");

  if (messageP == NULL)
    ErrDisplay("messageP == NULL");
#endif /* __RELEASE__ */

  // Save context
  oldDrawWindow = WinSetDrawWindow(OffScreen);
  oldDrawMode = WinSetDrawMode(winMask);
  oldFont = FntSetFont(stdFont);
  oldColor = WinSetBackColor(colorIndexP[COLORINDEX_WHITE]);
  
  // Display message
  WinPaintChars(messageP, 
                StrLen(messageP), 
                prefP->ScreenDisplayOffsetX + MESSAGE_START_X, 
                prefP->ScreenDisplayOffsetY + MESSAGE_START_Y);

  // Restore context
  WinSetBackColor(oldColor);
  FntSetFont(oldFont);
  WinSetDrawMode(oldDrawMode);
  WinSetDrawWindow(oldDrawWindow);
}
/*----------------------------------------------------------------------------*/


static tResult DisplayKeyboard(tKeyboard* keyboardP,
                               DisplayBackgroundFunctionPtr displayBackgroundP,
                               IndexedColorType textColor,
                               Coord YSize,
                               Coord YBase)
/***********************************************************************
 *
 *  DisplayKeyboard
 *
 ***********************************************************************/
{
IndexedColorType oldColor;
MemHandle resH;
BitmapPtr resP;
WinHandle oldDrawWindow;
WinHandle DrawWindow;
WinDrawOperation oldDrawMode;
FontID oldFont;
UInt16 Error;
UInt8 Loop;
char String[3];
Boolean ShiftPressed;


#ifndef __RELEASE__
  if (WinGetDrawWindow() == NULL)
    ErrDisplay("WinGetDrawWindow() == NULL");
#endif /* __RELEASE__ */

  // Offscreen window to prepare bitmap of panel
  DrawWindow = WinCreateOffscreenWindow(SCREEN_CPC_WIDTH,
                                        YSize,
                                        nativeFormat,
                                        &Error);
  if (DrawWindow == NULL)
    return ResultFailed;
    
  oldDrawWindow = WinSetDrawWindow(DrawWindow);

  //
  // Draw keyboard background
  //    
  if (displayBackgroundP)
  {
  	displayBackgroundP();
  }
 
  //
  // Draw pushed keys bitmap
  //
#ifndef __RELEASE__
  if (keyboardP->keyRShiftP == NULL)
    ErrDisplay("keyboardP->keyRShiftP == NULL");
#endif /* __RELEASE__ */
  DisplayOffscreenKey(keyboardP->keyRShiftP);
#ifndef __RELEASE__
  if (keyboardP->keyLShiftP == NULL)
    ErrDisplay("keyboardP->keyLShiftP == NULL");
#endif /* __RELEASE__ */
  DisplayOffscreenKey(keyboardP->keyLShiftP);
#ifndef __RELEASE__
  if (keyboardP->keyCTRLP == NULL)
    ErrDisplay("keyboardP->keyCTRLP == NULL");
#endif /* __RELEASE__ */
  DisplayOffscreenKey(keyboardP->keyCTRLP);

  //
  // Get CPC Font
  //
  resH = DmGetResource(FONT_ID_TYPE,
                       FONT_ID_CPC);
#ifndef __RELEASE__
  if (resH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */
  resP = MemHandleLock(resH);
#ifndef __RELEASE__
  if (resP == NULL)
    ErrDisplay("resP == NULL");
#endif /* __RELEASE__ */

  //
  // Draw key labels
  //
  ShiftPressed = IsShiftPressed() == cFalse ? false : true;
  
  if (FntDefineFont(FontCPC, (FontPtr)resP) == errNone)
  {
    // Save context
    oldDrawMode = WinSetDrawMode(winMask);
    oldFont = FntSetFont(FontCPC);

    oldColor = WinSetBackColor(textColor);
  
    //
    // Normal keyboard display
    //
    if (!GetShowCPCKeycodes())
    {
      String[1] = 0;
      for (Loop = 0; Loop < keyboardP->headerP->NbTotalKeys; Loop++)
      {
        const tKeySetting* settingP = keyboardP->keysP[Loop].settingP;

        String[0] = (ShiftPressed == false) ? settingP->charCode_Normal : settingP->charCode_Shift;
        if (String[0])
        {
          // Display key char
          WinPaintChars(String, 
                        1,
                        settingP->Left + ((settingP->Width - FONT_CPC_WIDTH) / 2),
                        settingP->Top + (settingP->Height - FONT_CPC_HEIGHT) / 2);
        }
      }
    }
    else
    //
    // CPC Keycodes display
    //
    {
      WinSetBackColor(colorIndexP[COLORINDEX_WHITE]);
      WinSetDrawMode(winPaintInverse);
    	
      String[2] = 0;
      for (Loop = 0; Loop < keyboardP->headerP->NbTotalKeys; Loop++)
      {
        const tKeySetting* settingP = keyboardP->keysP[Loop].settingP;

        String[0] = '0' + keyboardP->keysP[Loop].CPCKeycode / 10;
        String[1] = '0' + keyboardP->keysP[Loop].CPCKeycode % 10;

        if (String[0] == '0')
        {
          // Display key char
          WinPaintChars(&String[1], 
                        1,
                        settingP->Left + ((settingP->Width - FONT_CPC_WIDTH) / 2),
                        settingP->Top + (settingP->Height - FONT_CPC_HEIGHT) / 2);
        }
        else
        {
          // Display key char
          WinPaintChars(String, 
                        2,
                        settingP->Left + ((settingP->Width - (FONT_CPC_WIDTH * 2)) / 2),
                        settingP->Top + (settingP->Height - FONT_CPC_HEIGHT) / 2);
        }
      }
    }
  
    WinSetDrawMode(oldDrawMode);
    WinSetBackColor(oldColor);
    FntSetFont(oldFont);
  }

  // Free font resource
  MemHandleUnlock(resH);
  DmReleaseResource(resH);

  // Copy Offscreen bitmap to main screen by replacing Emulator and Sub panels
  WinSetDrawWindow(oldDrawWindow);
  WinDrawBitmap(WinGetBitmap(DrawWindow),
                0,
                YBase);
  WinDeleteWindow(DrawWindow,
                  false);
  
  return ResultSucceed;
}
/*----------------------------------------------------------------------------*/


static void Display464KeyboardBackground(void)
/***********************************************************************
 *
 *  Display464KeyboardBackground
 *
 ***********************************************************************/
{
MemHandle bitmapH;
BitmapPtr bitmapP;

  bitmapH = DmGetResource(bitmapRsc,
                          Keyboard_464);
#ifndef __RELEASE__
  if (bitmapH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */

  bitmapP = MemHandleLock(bitmapH);
#ifndef __RELEASE__
  if (bitmapP == NULL)
    ErrDisplay("bitmapP == NULL");
#endif /* __RELEASE__ */
  
  WinDrawBitmap(bitmapP,
                0,
                0);
  
  MemHandleUnlock(bitmapH);
}
/*----------------------------------------------------------------------------*/


static void Display664KeyboardBackground(void)
/***********************************************************************
 *
 *  Display664KeyboardBackground
 *
 ***********************************************************************/
{
MemHandle bitmapH;
BitmapPtr bitmapP;

  bitmapH = DmGetResource(bitmapRsc,
                          Keyboard_664);
#ifndef __RELEASE__
  if (bitmapH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */

  bitmapP = MemHandleLock(bitmapH);
#ifndef __RELEASE__
  if (bitmapP == NULL)
    ErrDisplay("bitmapP == NULL");
#endif /* __RELEASE__ */
  
  WinDrawBitmap(bitmapP,
                0,
                0);
  
  MemHandleUnlock(bitmapH);
}
/*----------------------------------------------------------------------------*/


static void Display6128KeyboardBackground(void)
/***********************************************************************
 *
 *  Display6128KeyboardBackground
 *
 ***********************************************************************/
{
MemHandle bitmapH;
BitmapPtr bitmapP;

	//
	// Top of keyboard
	//
  bitmapH = DmGetResource(bitmapRsc,
                          Keyboard_6128);
#ifndef __RELEASE__
  if (bitmapH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */

  bitmapP = MemHandleLock(bitmapH);
#ifndef __RELEASE__
  if (bitmapP == NULL)
    ErrDisplay("bitmapP == NULL");
#endif /* __RELEASE__ */
  
  WinDrawBitmap(bitmapP,
                0,
                0);
  
  MemHandleUnlock(bitmapH);

	//
	// Keyboard
	//
  bitmapH = DmGetResource(bitmapRsc,
                          Mini_Keyboard);
#ifndef __RELEASE__
  if (bitmapH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */

  bitmapP = MemHandleLock(bitmapH);
#ifndef __RELEASE__
  if (bitmapP == NULL)
    ErrDisplay("bitmapP == NULL");
#endif /* __RELEASE__ */
  
  WinDrawBitmap(bitmapP,
                0,
                38);
  
  MemHandleUnlock(bitmapH);
}
/*----------------------------------------------------------------------------*/


static void DisplayMiniKeyboardBackground(void)
/***********************************************************************
 *
 *  DisplayMiniKeyboardBackground
 *
 ***********************************************************************/
{
MemHandle bitmapH;
BitmapPtr bitmapP;

  bitmapH = DmGetResource(bitmapRsc,
                          Mini_Keyboard);
#ifndef __RELEASE__
  if (bitmapH == NULL)
    ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */

  bitmapP = MemHandleLock(bitmapH);
#ifndef __RELEASE__
  if (bitmapP == NULL)
    ErrDisplay("bitmapP == NULL");
#endif /* __RELEASE__ */
  
  WinDrawBitmap(bitmapP,
                0,
                0);
  
  MemHandleUnlock(bitmapH);
}
/*----------------------------------------------------------------------------*/


tResult DisplayDIAKeyboard(tUChar CPCModel)
/***********************************************************************
 *
 *  DisplayDIAKeyboard
 *
 ***********************************************************************/
{
IndexedColorType TextColor;
DisplayBackgroundFunctionPtr fctP;

  switch(CPCModel)
  {
    case CPC_MODEL_664:
      fctP = Display664KeyboardBackground;
      TextColor = colorIndexP[COLORINDEX_BLACK];
      break;

    case CPC_MODEL_6128:
      fctP = Display6128KeyboardBackground;
      TextColor = colorIndexP[COLORINDEX_BLACK];
      break;

    default:
      fctP = Display464KeyboardBackground;
      TextColor = colorIndexP[COLORINDEX_WHITE];
      break;
  }

	DisplayKeyboard(DIAKeyboardP,
	                fctP,
	                TextColor,
	                Y_SIZE_KEYBOARD_PANEL,
	                GetYBaseKeyBoardPanel());
  
  return ResultSucceed;
}
/*----------------------------------------------------------------------------*/


tResult DisplayMiniKeyboard(tVoid)
/***********************************************************************
 *
 *  DisplayMiniKeyboard
 *
 ***********************************************************************/
{
	DisplayKeyboard(MiniKeyboardP,
	                DisplayMiniKeyboardBackground,
	                colorIndexP[COLORINDEX_BLACK],
	                Y_SIZE_MINI_KEYBOARD_PANEL,
	                GetYBaseEmulatorPanel());
  
  return ResultSucceed;
}
/*----------------------------------------------------------------------------*/


tVoid DisplayOffScreenAutoToggleSpeed(UInt8 index)
/***********************************************************************
 *
 *  DisplayOffScreenAutoToggleSpeed
 *
 ***********************************************************************/
{
MemHandle resH;
BitmapPtr resP;
WinHandle oldDrawWindow;

  resH = DmGetResource(bitmapRsc,
                       AutoToggleSpeed_First+index-1);
  resP = MemHandleLock(resH);

  // Save context
  oldDrawWindow = WinSetDrawWindow(OffScreen);

  WinDrawBitmap(resP, 
                prefP->ScreenDisplayOffsetX + X_AUTO_TOGGLE_SPEED_BITMAP, 
                prefP->ScreenDisplayOffsetY + Y_AUTO_TOGGLE_SPEED_BITMAP);

  MemHandleUnlock(resH);
  DmReleaseResource(resH);

  // Restore context
  WinSetDrawWindow(oldDrawWindow);
}
/*----------------------------------------------------------------------------*/


tVoid DisplayOffScreenSoundVolumeAdjust(tUChar volume)
/***********************************************************************
 *
 *  DisplayOffScreenSoundVolumeAdjust
 *
 ***********************************************************************/
{
MemHandle resH;
BitmapPtr resP;
WinHandle oldDrawWindow;
IndexedColorType oldColor;
FontID oldFont;
WinDrawOperation oldDrawMode;
char string[20];

  resH = DmGetResource(bitmapRsc,
                       SoundVolumeAdjust);
  resP = MemHandleLock(resH);

  // Save context
  oldDrawWindow = WinSetDrawWindow(OffScreen);
  oldDrawMode = WinSetDrawMode(winMask);
  oldFont = FntSetFont(stdFont);

  //
  // Bitmap drawing
  //
  WinDrawBitmap(resP, 
                prefP->ScreenDisplayOffsetX + X_SOUND_VOLUME_ADJUST_BITMAP, 
                prefP->ScreenDisplayOffsetY + Y_SOUND_VOLUME_ADJUST_BITMAP);

  //
  // Display text
  //
  oldColor = WinSetBackColor(colorIndexP[COLORINDEX_WHITE]);

  StrPrintF(string,
            "%d %%",
            volume);
  WinPaintChars(string, 
                StrLen(string), 
                prefP->ScreenDisplayOffsetX + X_SOUND_VOLUME_ADJUST_MESSAGE, 
                prefP->ScreenDisplayOffsetY + Y_SOUND_VOLUME_ADJUST_MESSAGE);

  MemHandleUnlock(resH);
  DmReleaseResource(resH);

  WinSetBackColor(oldColor);

  // Restore context
  FntSetFont(oldFont);
  WinSetDrawMode(oldDrawMode);
  WinSetDrawWindow(oldDrawWindow);
}
/*----------------------------------------------------------------------------*/


tVoid DisplayOffScreenRockerHelp(tVoid)
/***********************************************************************
 *
 *  DisplayOffScreenRockerHelp
 *
 ***********************************************************************/
{
WinHandle oldDrawWindow;
IndexedColorType oldForeColor;
IndexedColorType oldBackColor;
Coord OffsetX, OffsetY;

#define TOP_TEXT "Top"
#define TOP_TEXT_LEN ((sizeof(TOP_TEXT) / sizeof (TOP_TEXT[0])) - 1)
#define TOP_TEXT_X_OFF ((SCREEN_CPC_WIDTH / 3) + 35)
#define TOP_TEXT_Y_OFF (0)
#define BOTTOM_TEXT "Bottom"
#define BOTTOM_TEXT_LEN ((sizeof(BOTTOM_TEXT) / sizeof (BOTTOM_TEXT[0])) - 1)
#define BOTTOM_TEXT_X_OFF ((SCREEN_CPC_WIDTH / 3) + 22)
#define BOTTOM_TEXT_Y_OFF (SCREEN_CPC_HEIGHT - 24)
#define LEFT_TEXT "Left"
#define LEFT_TEXT_LEN ((sizeof(LEFT_TEXT) / sizeof (LEFT_TEXT[0])) - 1)
#define LEFT_TEXT_X_OFF (2)
#define LEFT_TEXT_Y_OFF ((SCREEN_CPC_HEIGHT / 3) + 24)
#define RIGHT_TEXT "Right"
#define RIGHT_TEXT_LEN ((sizeof(RIGHT_TEXT) / sizeof (RIGHT_TEXT[0])) - 1)
#define RIGHT_TEXT_X_OFF (SCREEN_CPC_WIDTH - 42)
#define RIGHT_TEXT_Y_OFF LEFT_TEXT_Y_OFF

  // Save context
  oldDrawWindow = WinSetDrawWindow(OffScreen);
  WinPushDrawState();
  
  oldForeColor = WinSetForeColor(colorIndexP[COLORINDEX_WHITE]);
  oldBackColor = WinSetBackColor(colorIndexP[COLORINDEX_DARK_BLUE]);
  WinSetPatternType(grayPattern);
  
  // Draw matrix
  if (!FullscreenActive)
  {
    // Vertical lines
    WinPaintLine(prefP->ScreenDisplayOffsetX + (SCREEN_CPC_WIDTH / 3),
                 prefP->ScreenDisplayOffsetY,
                 prefP->ScreenDisplayOffsetX + (SCREEN_CPC_WIDTH / 3),
                 prefP->ScreenDisplayOffsetY + SCREEN_CPC_HEIGHT);
    WinPaintLine(prefP->ScreenDisplayOffsetX + (SCREEN_CPC_WIDTH * 2 / 3),
                 prefP->ScreenDisplayOffsetY,
                 prefP->ScreenDisplayOffsetX + (SCREEN_CPC_WIDTH * 2 / 3),
                 prefP->ScreenDisplayOffsetY + SCREEN_CPC_HEIGHT);
    // Horizontal lines
    WinPaintLine(prefP->ScreenDisplayOffsetX,
                 prefP->ScreenDisplayOffsetY + (SCREEN_CPC_HEIGHT / 3),
                 prefP->ScreenDisplayOffsetX + SCREEN_CPC_WIDTH, 
                 prefP->ScreenDisplayOffsetY + (SCREEN_CPC_HEIGHT / 3));
    WinPaintLine(prefP->ScreenDisplayOffsetX,
                 prefP->ScreenDisplayOffsetY + (SCREEN_CPC_HEIGHT * 2 / 3),
                 prefP->ScreenDisplayOffsetX + SCREEN_CPC_WIDTH,
                 prefP->ScreenDisplayOffsetY + (SCREEN_CPC_HEIGHT * 2 / 3));
  }
  else // Fullscreen
  {
    // Vertical lines
    WinPaintLine(OFFSCREEN_OFFSET_X_FS + (SCREEN_CPC_WIDTH_FS / 3),
                 0,
                 OFFSCREEN_OFFSET_X_FS + (SCREEN_CPC_WIDTH_FS / 3),
                 CPC_SCR_HEIGHT);
    WinPaintLine(OFFSCREEN_OFFSET_X_FS + (SCREEN_CPC_WIDTH_FS * 2 / 3),
                 0,
                 OFFSCREEN_OFFSET_X_FS + (SCREEN_CPC_WIDTH_FS * 2 / 3),
                 CPC_SCR_HEIGHT);
    // Horizontal lines
    WinPaintLine(0,
                 (OFFSCREEN_OFFSET_Y_FS + SCREEN_CPC_HEIGHT_FS / 3),
                 CPC_SCR_WIDTH, 
                 OFFSCREEN_OFFSET_Y_FS + (SCREEN_CPC_HEIGHT_FS / 3));
    WinPaintLine(0,
                 OFFSCREEN_OFFSET_Y_FS + (SCREEN_CPC_HEIGHT_FS * 2 / 3),
                 CPC_SCR_WIDTH,
                 OFFSCREEN_OFFSET_Y_FS + (SCREEN_CPC_HEIGHT_FS * 2 / 3));
  }
                
  //
  // Display texts
  //
  if (!FullscreenActive)
  {
    OffsetX = prefP->ScreenDisplayOffsetX;
    OffsetY = prefP->ScreenDisplayOffsetY;
  }
  else
  {
    OffsetX = OFFSCREEN_OFFSET_X;
    OffsetY = OFFSCREEN_OFFSET_Y;
  }
  WinSetDrawMode(winMask);
  WinSetBackColor(colorIndexP[COLORINDEX_WHITE]);
  WinPaintChars(TOP_TEXT,
                TOP_TEXT_LEN,
                OffsetX + TOP_TEXT_X_OFF,
                OffsetY + TOP_TEXT_Y_OFF);
  WinPaintChars(BOTTOM_TEXT,
                BOTTOM_TEXT_LEN,
                OffsetX + BOTTOM_TEXT_X_OFF,
                OffsetY + BOTTOM_TEXT_Y_OFF);
  WinPaintChars(LEFT_TEXT,
                LEFT_TEXT_LEN,
                OffsetX + LEFT_TEXT_X_OFF,
                OffsetY + LEFT_TEXT_Y_OFF);
  WinPaintChars(RIGHT_TEXT,
                RIGHT_TEXT_LEN,
                OffsetX + RIGHT_TEXT_X_OFF,
                OffsetY + RIGHT_TEXT_Y_OFF);

  // Restore context
  WinPopDrawState();
  WinSetDrawWindow(oldDrawWindow);
  WinSetForeColor(oldForeColor);
  WinSetBackColor(oldBackColor);
}
/*----------------------------------------------------------------------------*/


tVoid DisplayOffScreenKeyStatus(tVoid)
/***********************************************************************
 *
 *  DisplayOffScreenKeyStatus
 *
 ***********************************************************************/
{
MemHandle resH;
BitmapPtr resP;
WinHandle oldDrawWindow;

  // Save context
  oldDrawWindow = WinSetDrawWindow(OffScreen);

  //
  // Display SHIFT Indicator
  //
  if ((NativeCPC->keyboard_matrix[0x25 >> 4] & (1 << (0x25 & 7))) == 0)
  {
    resH = DmGetResource(bitmapRsc,
                         Shift_Key_Pressed);
    resP = MemHandleLock(resH);

    WinDrawBitmap(resP, 
                  prefP->ScreenDisplayOffsetX + X_KEY_STATUS_BITMAP, 
                  prefP->ScreenDisplayOffsetY + Y_SHIFT_STATUS_BITMAP);

    MemHandleUnlock(resH);
    DmReleaseResource(resH);
  }
  
  //
  // Display CTRL Indicator
  //
  if ((NativeCPC->keyboard_matrix[0x27 >> 4] & (1 << (0x27 & 7))) == 0)
  {
    resH = DmGetResource(bitmapRsc,
                         Ctrl_Key_Pressed);
    resP = MemHandleLock(resH);

    WinDrawBitmap(resP, 
                  prefP->ScreenDisplayOffsetX + X_KEY_STATUS_BITMAP, 
                  prefP->ScreenDisplayOffsetY + Y_CTRL_STATUS_BITMAP);

    MemHandleUnlock(resH);
    DmReleaseResource(resH);
  }

  // Restore context
  WinSetDrawWindow(oldDrawWindow);
  
}
/*----------------------------------------------------------------------------*/



Coord GetYBaseSubPanel(tVoid)
/***********************************************************************
 *
 *  GetYBaseSubPanel
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (prefP == NULL)
    ErrDisplay("prefP == NULL");
#endif /* __RELEASE__ */

  return (prefP->Display320x480 ? Y_OFFSET_SUB_PANEL_480 : Y_OFFSET_SUB_PANEL_320);
}
/*----------------------------------------------------------------------------*/


Coord GetYBaseEmulatorPanel(tVoid)
/***********************************************************************
 *
 *  GetYBaseEmulatorPanel
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (prefP == NULL)
    ErrDisplay("prefP == NULL");
#endif /* __RELEASE__ */

  return (prefP->Display320x480 ? Y_OFFSET_EMULATOR_PANEL_480 : Y_OFFSET_EMULATOR_PANEL_320);
}
/*----------------------------------------------------------------------------*/


Coord GetYBaseKeyBoardPanel(tVoid)
/***********************************************************************
 *
 *  GetYBaseKeyBoardPanel
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (prefP == NULL)
    ErrDisplay("prefP == NULL");
#endif /* __RELEASE__ */

  if (FullscreenActive)
  {
    return (Y_OFFSET_KEYBOARD_PANEL_FS);
  }
  else
  {
    if (prefP->Display320x480)
      return (Y_OFFSET_KEYBOARD_PANEL_480);
    else
      return (Y_OFFSET_KEYBOARD_PANEL_320);
  }
}
/*----------------------------------------------------------------------------*/


void DisplayRestoreError(tCPCRestoreReturnCode ErrorCode,
                         tPreferences* ReqPrefP)
/***********************************************************************
 *
 *  DisplayRestoreError
 *
 ***********************************************************************/
{
  if (ErrorCode != File_Not_Found)
  {
    if (ErrorCode == Wrong_Version)
    {
      FrmAlert(WrongCaPriceRestoreAlert);
    }
    else
    {
      EventType newEvent;

      if (ErrorCode == Bad_Data) FrmAlert(BadCPCRestoreAlert);
      if (ErrorCode == Not_Enough_Memory) FrmAlert(NotEnoughMemoryAlert);
            
      // Terminate application without context saving
      AllowSaveSession = 0;
      newEvent.eType = appStopEvent;
      EvtAddEventToQueue(&newEvent);
    }
  }
}
/*----------------------------------------------------------------------------*/


#ifdef _TRACE
tVoid DisplayTraceInfo(tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  DisplayTraceInfo
 *
 ***********************************************************************/
{
tZ80* Z80 = (tZ80*)EndianSwap32(NativeCPC->Z80);
FontID oldFont;
WinDrawOperation oldDrawMode;
char string[40];
tUShort Z80_PC;
tUShort Z80_SP;
tUShort Z80_AF;
tUShort Z80_BC;
tUShort Z80_DE;
tUShort Z80_HL;
tUChar Opcode, Opcode1, Opcode2, Opcode3;
IndexedColorType oldTextColor;
IndexedColorType oldBackColor;

  // Save context
  oldDrawMode = WinSetDrawMode(winPaint);
  oldFont = FntSetFont(stdFont);

  oldTextColor = WinSetTextColor(colorIndexP[COLORINDEX_WHITE]);
  oldBackColor = WinSetBackColor(colorIndexP[COLORINDEX_BLACK]);
  
  // Display PC and SP (Display Short Only)
  Z80_PC = ((tRegister)EndianSwap32(_PCdword)).w.h;
  Z80_SP = ((tRegister)EndianSwap32(Z80->Regs.SP.d)).w.h;
  StrPrintF(string,
            "PC SP=%x %x",
            Z80_PC,
            Z80_SP);
  WinPaintChars(string,
                StrLen(string),
                X_TRACE_DISPLAY_PC,
                Y_TRACE_DISPLAY_PC);
  
  // Display Registers (Display Short Only)
  Z80_AF = ((tRegister)EndianSwap32(_AFdword)).w.h;
  Z80_BC = ((tRegister)EndianSwap32(_BCdword)).w.h;
  Z80_DE = ((tRegister)EndianSwap32(_DEdword)).w.h;
  Z80_HL = ((tRegister)EndianSwap32(_HLdword)).w.h;
  StrPrintF(string,
            "AF BC DE HL=%x %x %x %x",
            Z80_AF,
            Z80_BC,
            Z80_DE,
            Z80_HL);
  WinPaintChars(string,
                StrLen(string),
                X_TRACE_DISPLAY_PC,
                Y_TRACE_DISPLAY_REGISTERS);

  // Display Opcodes (Display Short Only)
  Opcode = *((tUChar*)EndianSwap32(NativeCPC->membank_read[Z80_PC >> 14]) + (Z80_PC & 0x3fff));
  Opcode1 = *((tUChar*)EndianSwap32(NativeCPC->membank_read[(Z80_PC+1) >> 14]) + ((Z80_PC+1) & 0x3fff));
  Opcode2 = *((tUChar*)EndianSwap32(NativeCPC->membank_read[(Z80_PC+2) >> 14]) + ((Z80_PC+2) & 0x3fff));
  Opcode3 = *((tUChar*)EndianSwap32(NativeCPC->membank_read[(Z80_PC+3) >> 14]) + ((Z80_PC+3) & 0x3fff));
  StrPrintF(string,
            "(PC)=%x%x",
            (((tUShort)Opcode<<8) | (tUShort)Opcode1),
            (((tUShort)Opcode2<<8) | (tUShort)Opcode3));
  WinPaintChars(string,
                StrLen(string),
                X_TRACE_DISPLAY_PC,
                Y_TRACE_DISPLAY_OPCODE);

  // Display Breakpoint (Display Short Only)
  StrPrintF(string,
            "Breakpoint=%x",
            EndianSwap32(Z80->Regs.breakpoint) << 16);
  WinPaintChars(string,
                StrLen(string),
                X_TRACE_DISPLAY_PC,
                Y_TRACE_DISPLAY_BREAKPOINT);

  // Restore context
  FntSetFont(oldFont);
  WinSetDrawMode(oldDrawMode);
  WinSetTextColor(oldTextColor);
  WinSetBackColor(oldBackColor);
}
/*----------------------------------------------------------------------------*/


tVoid DisplayNativeTrace(tULong Trace)
/***********************************************************************
 *
 *  DisplayNativeTrace
 *
 ***********************************************************************/
{
WinDrawOperation oldDrawMode;
char string[30];
FontID oldFont;
IndexedColorType oldTextColor;
IndexedColorType oldBackColor;

  // Save context
  oldDrawMode = WinSetDrawMode(winPaint);
  oldFont = FntSetFont(stdFont);

  oldTextColor = WinSetTextColor(colorIndexP[COLORINDEX_WHITE]);
  oldBackColor = WinSetBackColor(colorIndexP[COLORINDEX_BLACK]);

  // Display Trace
  StrPrintF(string,
            "Trace=%lxh",
            Trace);
  WinPaintChars(string,
                StrLen(string),
                X_TRACE_DISPLAY_PC,
                Y_TRACE_DISPLAY_TRACE);

  // Restore context
  FntSetFont(oldFont);
  WinSetDrawMode(oldDrawMode);
  WinSetTextColor(oldTextColor);
  WinSetBackColor(oldBackColor);
}
/*----------------------------------------------------------------------------*/

#endif /* _TRACE */


#ifdef _DEBUG

tVoid DisplayOffscreenSoundData(tULong NativeSamples,
                                tULong CbCount,
                                tULong CbSamples)
/***********************************************************************
 *
 *  DisplayOffscreenSoundData
 *
 ***********************************************************************/
{
char string[20];
FontID oldFont;
WinDrawOperation oldDrawMode;
WinHandle oldDrawWindow;
IndexedColorType oldBackColor;

  // Save context
  oldDrawWindow = WinSetDrawWindow(OffScreen);
  oldDrawMode = WinSetDrawMode(winMask);
  oldFont = FntSetFont(stdFont);

  oldBackColor = WinSetBackColor(colorIndexP[COLORINDEX_WHITE]);
  
  // Display Sound Samples
  StrPrintF(string,
            "SndNative=%ld",
            NativeSamples);
  WinPaintChars(string,
                StrLen(string),
                X_DEBUG_DISPLAY,
                Y_DEBUG_DISPLAY_SAMPLES);

  // Display Callback data text
  StrPrintF(string,
            "SndCb=%ld (%ld)",
            CbSamples,
            CbCount);
  WinPaintChars(string,
                StrLen(string),
                X_DEBUG_DISPLAY,
                Y_DEBUG_DISPLAY_CB);

  // Restore context
  FntSetFont(oldFont);
  WinSetDrawMode(oldDrawMode);
  WinSetDrawWindow(oldDrawWindow);
  WinSetBackColor(oldBackColor);
}

#endif /* _DEBUG */


#if defined(_DEBUG) || defined(_SHOW_KEYS)

tVoid DisplayOffscreenKeyState(tVoid)
/***********************************************************************
 *
 *  DisplayOffscreenKeyState
 *
 ***********************************************************************/
{
char string[20], convert[20];
FontID oldFont;
WinDrawOperation oldDrawMode;
WinHandle oldDrawWindow;
UInt32 keyMask;
IndexedColorType oldBackColor;
UInt8 Loop;

  // Save context
  oldDrawWindow = WinSetDrawWindow(OffScreen);
  oldDrawMode = WinSetDrawMode(winMask);
  oldFont = FntSetFont(stdFont);

  oldBackColor = WinSetBackColor(colorIndexP[COLORINDEX_WHITE]);
  
  // Display Hard Keys state
  StrIToH(convert,
          (tULong)KeyCurrentState());
  StrPrintF(string,
            "HKey=%s",
            convert);
  WinPaintChars(string,
                StrLen(string),
                X_DEBUG_DISPLAY,
                Y_DEBUG_DISPLAY_KEY);

  // Display Hard Keys mask
  keyMask = KeySetMask(0); 
  KeySetMask(keyMask);
  StrIToH(convert, (tULong)keyMask);
  StrPrintF(string,
            "HMask=%s",
            convert);
  WinPaintChars(string,
                StrLen(string),
                X_DEBUG_DISPLAY,
                Y_DEBUG_DISPLAY_KEY_MASK);
                
  // Display Current Pressed Key
  StrPrintF(string,
            "PressedKey=%x",
            GetCurrentPressedKeys());
  WinPaintChars(string,
                StrLen(string),
                X_DEBUG_DISPLAY,
                Y_DEBUG_DISPLAY_PRESSED_KEY);

  // Display Keyboard Matrix
  for (Loop=0; Loop < 2; Loop++)
  {
    StrPrintF(string,
              "%02x%02x%02x%02x",
              (NativeCPC->keyboard_matrix[0+(Loop*8)] << 8) + NativeCPC->keyboard_matrix[1+(Loop*8)],
              (NativeCPC->keyboard_matrix[2+(Loop*8)] << 8) + NativeCPC->keyboard_matrix[3+(Loop*8)],
              (NativeCPC->keyboard_matrix[4+(Loop*8)] << 8) + NativeCPC->keyboard_matrix[5+(Loop*8)],
              (NativeCPC->keyboard_matrix[6+(Loop*8)] << 8) + NativeCPC->keyboard_matrix[7+(Loop*8)]);
    WinPaintChars(string,
                  StrLen(string),
                  X_DEBUG_DISPLAY,
                  Y_DEBUG_DISPLAY_KEYBOARD + (Y_DEBUG_INTERVAL*Loop));
  }

  // Restore context
  FntSetFont(oldFont);
  WinSetDrawMode(oldDrawMode);
  WinSetDrawWindow(oldDrawWindow);
  WinSetBackColor(oldBackColor);
}

#endif /* _DEBUG || _SHOW_KEYS */


#ifdef _SHOW_FPS

tVoid DisplayOffscreenFPS(tULong FPS)
/***********************************************************************
 *
 *  DisplayOffscreenFPS
 *
 ***********************************************************************/
{
char string[20];
FontID oldFont;
WinDrawOperation oldDrawMode;
WinHandle oldDrawWindow;
IndexedColorType oldBackColor;

  // Save context
  oldDrawWindow = WinSetDrawWindow(OffScreen);
  oldDrawMode = WinSetDrawMode(winMask);
  oldFont = FntSetFont(stdFont);

  oldBackColor = WinSetBackColor(colorIndexP[COLORINDEX_WHITE]);
  
  // Display Sound Samples
  if (FPS)
  {
    StrPrintF(string,
              "FPS=%ld",
              FPS);
    WinPaintChars(string,
                  StrLen(string),
                  X_FPS_DISPLAY,
                  Y_FPS_DISPLAY);
  }

  // Restore context
  FntSetFont(oldFont);
  WinSetDrawMode(oldDrawMode);
  WinSetDrawWindow(oldDrawWindow);
  WinSetBackColor(oldBackColor);
}

#endif /* _SHOW_FPS */


#if defined(_DEBUG) || defined(_SHOW_DATA)

tVoid DisplayOffscreenDATA(tChar* titleP, 
                           tULong data,
                           tULong x,
                           tULong y)
/***********************************************************************
 *
 *  DisplayOffscreenDATA
 *
 ***********************************************************************/
{
char string[100];
FontID oldFont;
WinDrawOperation oldDrawMode;
WinHandle oldDrawWindow;
IndexedColorType oldBackColor;

  // Save context
  oldDrawWindow = WinSetDrawWindow(OffScreen);
  oldDrawMode = WinSetDrawMode(winInvert);
  oldFont = FntSetFont(stdFont);

  oldBackColor = WinSetBackColor(colorIndexP[COLORINDEX_WHITE]);
  
  // Display Sound Samples
  StrPrintF(string,
            "%s=%ld",
            titleP,
            data);
  WinPaintChars(string,
                StrLen(string),
                x,
                y);

  // Restore context
  FntSetFont(oldFont);
  WinSetDrawMode(oldDrawMode);
  WinSetDrawWindow(oldDrawWindow);
  WinSetBackColor(oldBackColor);
}

#endif /* _SHOW_DATA */
