/*
    CaPriCe for Palm OS - Amstrad CPC 464/664/6128 emulator for Palm devices
    Copyright (C) 2009  Frdric Coste

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <PalmOS.h>
#include <PceNativeCall.h>
#include <ByteOrderUtils.h>
#include <Rect.h>
#include <FeatureMgr.h>
#include <NotifyMgr.h>
#include <VFSMgr.h>
#include "..\Palm\PalmDisplayExtent.h"

#include "CaPriCe.h"
#include "Prefs.h"
#include "Routines.h"
#include "Resources.h"
#include "Keyboard.h"
#include "Display.h"
#include "Files.h"
#include "CPC.h"
#include "Forms.h"
#include "Sound.h"
#include "Trace.h"
#include "Sections.h"


#undef CRASH_WORKAROUND
#define CRASH_WORKAROUND


//===================
// PATCH begin
#ifdef _PATCH_ENABLE

// CPC true speed patch
#undef PATCH_CPC_TRUE_SPEED
//#define PATCH_CPC_TRUE_SPEED

#endif /* _PATCH_ENABLE */
// PATCH end
//===================


UInt8 SystemHalt;
UInt8 collapsibleIA;
UInt8 CPCStarted;
UInt32 u32DebounceInTicks;
UInt8 DisplayEmuSpeed;
UInt8 AllowSaveSession;
UInt8 OnScreenRockerHelpDisplay;
UInt8 FullscreenActive;
UInt8 LandscapeActive;
UInt16 DexLibRefNum;
UInt8 SubPanelActive;
UInt8 MiniKeyboardActive;
UInt32 oldHardKeyMask;
UInt32 emulatorHardKeyMask;
UInt8 FDCLedOffScreenDisplay;
UInt8 Navigation5WaysActive;
UInt8 ScreenshotRequested;
UInt8 VideoFrameInitialDelay;
UInt32 WestPhaserTriggerActiveDelay;

tPreferences* prefP = NULL;
RGBColorType *originalPaletteP = NULL;
IndexedColorType* colorIndexP = NULL;
CmdToGoLaunchParamHeader* CmdToGoParamsP = NULL;
tDIAState oldDIAState;


// Autostart
char AutoStartCommand[20];

char* sessionFilenameP = NULL;

static UInt32 showMessageTick = NULL;
static const char* messageToShowP = NULL;
static UInt16 oldAutoOffTime;
static UInt32 oldScreenDepth;

static UInt8 ShowCPCKeycodes;


static const UInt32 AllowedDeviceList[] =
{
  'Cct1', // PALM_Tungsten_E
  'D050', // PALM_TX
  // Users feedback validated devices
  'TnT5', // PALM_Tungsten_T5
  'Zir4', // PALM_Tungsten_E2
  'MT64', // PALM Tungsten_C
  'Arz1', // PALM_Tungsten_T3
  'D061', // PALM_Centro
  // Not validated devices
  //'Frg1', // Palm_Tungsten_T
  //'Frg2', // Palm_Tungsten_T2
  //'Zi21', // PALM_Zire_21
  //'Zpth', // PALM_Zire_71
  //'H101', // PALM_Treo_600
};
#define DEVICEID_MAX          NUMBER_OF_ITEMS(AllowedDeviceList)


static const char* EmulatorMessages[] =
{
  "Joystick Rocker",
  "Cursors Rocker",
  "CPC true speed",
  "Palm full speed",
  "Music & Sound",
  "Mute",
  "Function keys",
  "Numeric keys"
};


static const RGBColorType ColorsRGB[] =
{
  // { 0, RED, GREEN, BLUE }
  {    0,   0,     0,    0 }, // COLORINDEX_BLACK
  {    0,   0,     0,  128 }, // COLORINDEX_DARK_BLUE
  {    0, 255,   255,    0 }, // COLORINDEX_YELLOW
  {    0, 255,   255,  255 }, // COLORINDEX_WHITE
  {    0, 164,     0,    0 }, // COLORINDEX_RED
  {    0,   0,     0,  192 }, // COLORINDEX_BLUE
};


#define DISK_DISPLAY_COUNT            5

#define memNewChunkFlagAllowLarge     0x1000


#define AUTOSTARTHOLD_CYCLES          ((AUTOSTARTHOLDTIME_MS / CYCLES_MS) + 1)
#define AUTOSTARTRELEASE_CYCLES       ((AUTOSTARTRELEASETIME_MS / CYCLES_MS) + 1)
#define AUTOSTARTRESET_CYCLES         ((AUTOSTARTRESETTIME_MS / CYCLES_MS) + 1)


static void AppEventLoop(void);
static Err AppStart(void);
static void AppStop(void);


/***********************************************************************
 *
 *  Internal Functions
 *
 ***********************************************************************/

static Err RomVersionCompatible(UInt32 requiredVersion,
                                UInt16 launchFlags)
/***********************************************************************
 *
 *  DmGetLargeResource
 *
 *  !!! Function MUST do not use global variables !!
 *
 ***********************************************************************/
{
UInt32 romVersion;

  /* See if we're on in minimum required version of the ROM or later. */
  FtrGet(sysFtrCreator,
         sysFtrNumROMVersion,
         &romVersion);
  if (romVersion < requiredVersion)
  {
    if ( (launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
         (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp) )
    {
      FrmAlert(RomIncompatibleAlert);

      /* Palm OS versions before 2.0 will continuously relaunch this
       * app unless we switch to another safe one. */
      if (romVersion < kPalmOS20Version)
      {
        AppLaunchWithCommand(sysFileCDefaultApp,
                             sysAppLaunchCmdNormalLaunch,
                             NULL);
      }
    }

    return sysErrRomIncompatible;
  }

  return errNone;
}
/*----------------------------------------------------------------------------*/


UInt32 PilotMain(UInt16 cmd,
                 MemPtr cmdPBP,
                 UInt16 launchFlags)
/***********************************************************************
 *
 *  PilotMain
 *
 *  !!! Function MUST do not use global variables !!
 *
 ***********************************************************************/
{
Err error;

  error = RomVersionCompatible(ourMinVersion,
                               launchFlags);
  if (error) return (error);

  switch (cmd)
  {
    case sysAppLaunchCmdNotify:
    {
      EventType newEvent;

      switch (((SysNotifyParamType*)cmdPBP)->notifyType)
      {
        case sysNotifyDisplayChangeEvent:
        case sysNotifyDisplayResizedEvent:
        {
          newEvent.eType = winDisplayChangedEvent;
          EvtAddUniqueEventToQueue(&newEvent,
                                   0,
                                   true);
        }
        break;

        case sysNotifyVolumeUnmountedEvent:
        {
          newEvent.eType = volumeUnmountedEvent;
          EvtAddEventToQueue(&newEvent);
        }
        break;

        case sysNotifyVolumeMountedEvent:
        {
          ((SysNotifyParamType*)cmdPBP)->handled |= (vfsHandledStartPrc | vfsHandledUIAppSwitch);
          newEvent.eType = volumeMountedEvent;
          EvtAddEventToQueue(&newEvent);
        }
        break;
      }
    }
    break;


    case sysAppLaunchCmdGoTo:
    {
      GoToParamsType* paramsP = (GoToParamsType*)cmdPBP;

      // Application do not have to be started
      if ((launchFlags & sysAppLaunchFlagNewGlobals) == 0)
      {
        if (paramsP->matchCustom)
        {
          MemPtrFree((MemPtr)paramsP->matchCustom);
        }
        break;
      }

      // Save CmdToGo params
      CmdToGoParamsP = (CmdToGoLaunchParamHeader*)paramsP->matchCustom;

      // CAUTION: MUST be followed by sysAppLaunchCmdNormalLaunch
    }


    case sysAppLaunchCmdNormalLaunch:
    {
      if ((error = AppStart()) == errNone)
      {
        FrmGotoForm(MainForm);
        AppEventLoop();
      }
      AppStop();
    }
    break;

    case sysAppLaunchCmdExgAskUser:
    {
      // Automatically accept exchange
      ((ExgAskParamType*)cmdPBP)->result = exgAskOk;
    }
    break;

    // Receive image disk content
    case sysAppLaunchCmdExgReceiveData:
    {
      CmdToGoLaunchParamHeader Header;
      MemPtr bufferP;
      UInt32 receive;
      ExgSocketType* socketP = (ExgSocketType*)cmdPBP;
      Boolean IsAppActive = launchFlags & sysAppLaunchFlagSubCall;

      // Prepare Header
      Header.magicID = appFileCreator;
      Header.descriptionOffset = DWORD_UPPER_ALIGN(sizeof(CmdToGoLaunchParamHeader));
      Header.descriptionSize = StrLen(socketP->description) + 1; // Add 0
      Header.fileOffset = Header.descriptionOffset + DWORD_UPPER_ALIGN(Header.descriptionSize);
      Header.fileSize = socketP->length;

      // Allocate memory
      bufferP = MemChunkNew(0,
                            Header.fileOffset + Header.fileSize,
                            0 | memNewChunkFlagNonMovable | memNewChunkFlagAllowLarge);
      if (bufferP == NULL)
      {
        FrmAlert(NotEnoughMemoryAlert);
        break;
      }

      MemSet(bufferP,
             Header.fileOffset + Header.fileSize,
             0);

      socketP->goToCreator = appFileCreator;
      socketP->goToParams.matchCustom = (UInt32)bufferP;

      // Copy Header into Buffer
      MemMove(bufferP,
              &Header,
              sizeof(CmdToGoLaunchParamHeader));

      // Copy Description into Buffer
      MemMove(bufferP + Header.descriptionOffset,
              socketP->description,
              Header.descriptionSize);

      // Copy file content into Buffer
      if (ExgAccept(socketP) != errNone)
        break;

      receive = ExgReceive(socketP,
                           bufferP + Header.fileOffset,
                           Header.fileSize,
                           &error);

      ExgDisconnect(socketP,
                    errNone);
    }
    break;

    case sysAppLaunchCmdSyncNotify:
    {
      // Register file extension
      ExgRegisterDatatype(appFileCreator,
                          exgRegExtensionID,
                          DISK_EXTENSION_SHORT,
                          DISK_EXTENSION_DESCRIPTION,
                          0);
    }
    break;

  }

  return error;
}
/*----------------------------------------------------------------------------*/


static Err AppStart(void)
/***********************************************************************
 *
 *  AppStart
 *
 ***********************************************************************/
#undef APPSTART_TRACE_ENABLED
//#define APPSTART_TRACE_ENABLED

#ifdef APPSTART_TRACE_ENABLED
#  define APPSTART_TRACE_SHOW_INT(value) TRACE_SHOW_INT("AppStart", value)
#else
#  define APPSTART_TRACE_SHOW_INT(value)
#endif /* APPSTART_TRACE_ENABLED */

#undef APPSTART_BAD_LIREPREFERENCES
//#define APPSTART_BAD_LIREPREFERENCES
#undef APPSTART_BAD_STARTMINIKEYBOARD
//#define APPSTART_BAD_STARTMINIKEYBOARD
#undef APPSTART_BAD_STARTDIAKEYBOARD
//#define APPSTART_BAD_STARTDIAKEYBOARD
#undef APPSTART_BAD_WINSCREENGETATTRIBUTES
//#define APPSTART_BAD_WINSCREENGETATTRIBUTES
#undef APPSTART_BAD_WINSCREENMODE_GET
//#define APPSTART_BAD_WINSCREENMODE_GET
#undef APPSTART_BAD_WINSCREENMODE_SET
//#define APPSTART_BAD_WINSCREENMODE_SET
#undef APPSTART_PROCESSOR_IS_ARM
//#define APPSTART_PROCESSOR_IS_ARM
#undef APPSTART_PROCESSOR_IS_DRAGONBALL
//#define APPSTART_PROCESSOR_IS_DRAGONBALL
#undef APPSTART_UNKNOWN_DEVICEID
//#define APPSTART_UNKNOWN_DEVICEID
#undef APPSTART_BAD_SYSCURDATABASE
//#define APPSTART_BAD_SYSCURDATABASE
#undef APPSTART_NO_DIA
//#define APPSTART_NO_DIA
#undef APPSTART_NO_5_WAYS_NAVIGATION
//#define APPSTART_NO_5_WAYS_NAVIGATION
#undef APPSTART_BAD_SYSNOTIFYREGISTER
//#define APPSTART_BAD_SYSNOTIFYREGISTER
#undef APPSTART_NO_VFS_NAVIGATION
//#define APPSTART_NO_VFS_NAVIGATION
{
UInt32 attr;
UInt32 depth;
UInt32 value;
UInt32 DeviceID;
UInt32 ProcessorType;
Err Error;
char String[5];
Boolean DeviceFound;
tUChar DeviceIndex;
UInt16 cardNo;
LocalID dbID;
Boolean regError;
UInt32 Loop;

#ifdef _TESTU
  Error = Routines_PerformTestU();
  if (Error != errNone) TRACE_SHOW_HEX("Routines_PerformTestU", Error-testUErrorClass);
#endif /* _TESTU */

  SystemHalt = 1;
  collapsibleIA = 0;
  CPCStarted = 0;
  DisplayEmuSpeed = 0;
  showMessageTick = 0;
  AllowSaveSession = 1;
  AutoToggleActive = 0;
  OnScreenRockerHelpDisplay = 0;
  FullscreenActive = 0;
  LandscapeActive = 0;
  DexLibRefNum = 0;
  SubPanelActive = 0;
  MiniKeyboardActive = 0;
  Navigation5WaysActive = 0;
  ScreenshotRequested = 0;
  ShowCPCKeycodes = 0;
  u32DebounceInTicks = (SysTicksPerSecond() * DEBOUNCETIME_MS / 1000) + 1; // +1 = upper round
  AutoStartCommand[0] = 0;

  APPSTART_TRACE_SHOW_INT(1);

  InitKeys();
  CreateDirectories(DEFAULT_PALM_PATH,
                    DEFAULT_CAPRICE_PATH,
                    DEFAULT_DISK_PATH,
                    DEFAULT_SCREENSHOT_PATH);

  APPSTART_TRACE_SHOW_INT(2);

  //
  // !! Save context before any error return !!
  //
  // Sauvegarder l'tat avant modification
  WinScreenMode(winScreenModeGet,
                NULL,
                NULL,
                &oldScreenDepth,
                NULL);

  // Sauvegarder le masque des touches matrielles
  oldHardKeyMask = KeySetMask(0);
  KeySetMask(oldHardKeyMask);

  // Disable Auto Off Time
  oldAutoOffTime = SysSetAutoOffTime(0);

  APPSTART_TRACE_SHOW_INT(3);

  // Lire les prfrences
  Error = LirePreferences(&prefP);
#ifdef APPSTART_BAD_LIREPREFERENCES
  Error = ~errNone;
#endif /* APPSTART_BAD_LIREPREFERENCES */
  if (Error != errNone)
  {
    FrmAlert(NotEnoughMemoryAlert);
    return Error;
  }

  // Initialisation lie aux prfrences
  if (prefP->CPCTrueSpeed)
  {
    EmulatorKeys[KEYINDEX_TRUESPEED].KeyStatus = KeyPressed;
  }
  SetRenderingParameters(prefP->Rendering);

  APPSTART_TRACE_SHOW_INT(4);

  Error = StartMiniKeyboard();
#ifdef APPSTART_BAD_STARTMINIKEYBOARD
  Error = ~errNone;
#endif /* APPSTART_BAD_STARTMINIKEYBOARD */
  if (Error != errNone)
  {
    FrmAlert(NotEnoughMemoryAlert);
    return Error;
  }

  APPSTART_TRACE_SHOW_INT(5);

  Error = StartDIAKeyboard();
#ifdef APPSTART_BAD_STARTDIAKEYBOARD
  Error = ~errNone;
#endif /* APPSTART_BAD_STARTDIAKEYBOARD */
  if (Error != errNone)
  {
    FrmAlert(NotEnoughMemoryAlert);
    return Error;
  }
  ActiveKeyboardP = DIAKeyboardP;
  UpdateKeysPosition();

  APPSTART_TRACE_SHOW_INT(6);

  // Vrifier si affichage en densit double
  WinScreenGetAttribute(winScreenDensity,
                        &attr);
#ifdef APPSTART_BAD_WINSCREENGETATTRIBUTES
  attr = ~kDensityDouble;
#endif /* APPSTART_BAD_WINSCREENGETATTRIBUTES */
  if (attr != kDensityDouble)
  {
    FrmAlert(NoHDScreenAlert);
    return sysErrNotAllowed;
  }

  APPSTART_TRACE_SHOW_INT(7);

  // Vrifier la possibilit d'afficher en 256 couleurs.
  Error = WinScreenMode(winScreenModeGetSupportedDepths,
                        NULL,
                        NULL,
                        &depth,
                        NULL);
#ifdef APPSTART_BAD_WINSCREENMODE_GET
  Error = ~errNone;
#endif /* APPSTART_BAD_WINSCREENMODE_GET */
  if ( ((depth & 0x80) == 0) || (Error != errNone) )
  {
    FrmAlert(NoColorSupportAlert);
    return sysErrNotAllowed;
  }

  APPSTART_TRACE_SHOW_INT(8);

  // Passer en 256 couleurs
  depth = 8;
  Error = WinScreenMode(winScreenModeSet,
                        NULL,
                        NULL,
                        &depth,
                        NULL);
#ifdef APPSTART_BAD_WINSCREENMODE_SET
  Error = ~errNone;
#endif /* APPSTART_BAD_WINSCREENMODE_SET */
  if (Error != errNone)
  {
    FrmAlert(NoColorSupportAlert);
    return sysErrNotAllowed;
  }

  APPSTART_TRACE_SHOW_INT(9);

  // Sauvegarder la palette
  originalPaletteP = (RGBColorType*)MemPtrNew(sizeof(RGBColorType) * 256);
  if (originalPaletteP == NULL)
  {
    FrmAlert(NotEnoughMemoryAlert);
    return Error;
  }
  WinPalette(winPaletteGet,
             0,
             256,
             originalPaletteP);

  APPSTART_TRACE_SHOW_INT(10);

  // Prparer les index des couleurs utiliss
  colorIndexP = (IndexedColorType*)MemPtrNew(sizeof(IndexedColorType) * COLORINDEX_COUNT);
  if (colorIndexP == NULL)
  {
    FrmAlert(NotEnoughMemoryAlert);
    return Error;
  }
  for (Loop=0; Loop < COLORINDEX_COUNT; Loop++)
  {
#ifndef CRASH_WORKAROUND
    colorIndexP[Loop] = WinRGBToIndex(&ColorsRGB[Loop]); // Crash
#else /* CRASH_WORKAROUND */
    RGBColorType color = ColorsRGB[Loop];
    colorIndexP[Loop] = WinRGBToIndex(&color);
#endif /* CRASH_WORKAROUND */
  }

  APPSTART_TRACE_SHOW_INT(11);

  // Vrifier la compatibilit avec l'appareil
  // see SystemMgr.h
  DeviceFound = false;
  FtrGet(sysFileCSystem,
         sysFtrNumProcessorID,
         &ProcessorType);
#ifdef APPSTART_PROCESSOR_IS_ARM
  ProcessorType = sysFtrNumProcessorXscale;
#endif /* APPSTART_PROCESSOR_IS_ARM */
#ifdef APPSTART_PROCESSOR_IS_DRAGONBALL
  ProcessorType = sysFtrNumProcessor328; // Motorola 68328    (Dragonball)
#endif /* APPSTART_PROCESSOR_IS_DRAGONBALL */
  if (sysFtrNumProcessorIsARM(ProcessorType))
  {
    if (!FtrGet(sysFtrCreator,
                sysFtrNumOEMDeviceID,
                &DeviceID))
    {
#ifdef APPSTART_UNKNOWN_DEVICEID
      DeviceID = 'Fred';
#endif /* APPSTART_UNKNOWN_DEVICEID */
      for (DeviceIndex=0; DeviceIndex < DEVICEID_MAX; DeviceIndex++)
      {
        if (DeviceID == AllowedDeviceList[DeviceIndex])
        {
          DeviceFound = true;
        }
      }

      // Avertir l'utilisateur des risques.
      if (DeviceFound == false)
      {
        StrNCopy(String, (char*)&DeviceID, 4);
        String[4] = 0;
        if (FrmCustomAlert(UnknownDeviceIDAlert,
                           String,
                           "",
                           "") == 0)
        {
          // L'utilisation ne dsire pas tester CaPriCe.
          return sysErrNotAllowed;
        }
      }
    }
  }
  // Ni ARM, Ni x86 = Erreur Cible
  else if (ProcessorType != sysFtrNumProcessorx86)
  {
    FrmAlert(ARMProcessorAlert);
    return sysErrNotAllowed;
  }

  APPSTART_TRACE_SHOW_INT(12);

  Error = SysCurAppDatabase(&cardNo,
                            &dbID);
#ifdef APPSTART_BAD_SYSCURDATABASE
  Error = ~kDensityDouble;
#endif /* APPSTART_BAD_SYSCURDATABASE */
  if (Error != errNone)
  {
    FrmAlert(UnableToRegisterAlert);
    return Error;
  }

  APPSTART_TRACE_SHOW_INT(13);

  // Vrifier la prsence de la zone DIA
  Error = FtrGet(pinCreator,
                 pinFtrAPIVersion,
                 &value);
#ifdef APPSTART_NO_DIA
  value = 0;
#endif /* APPSTART_NO_DIA */
  if ( (Error == errNone) && value )
  {
    collapsibleIA = 1;

    UpdateKeysPosition();
    
    // Sauvegarder l'tat de la DIA
    SaveDIAState(&oldDIAState);

    if (SysNotifyRegister(cardNo,
                          dbID,
                          sysNotifyDisplayResizedEvent,
                          NULL,
                          sysNotifyNormalPriority,
                          NULL ) != errNone)
    {
      FrmAlert(UnableToRegisterAlert);
      return sysErrNoFreeResource;
    }

#ifdef __RELEASE__
    // Open Display Extent Library
    if (SysLibLoad(dexLibType,
                   dexLibCreator,
                   &DexLibRefNum) == errNone)
	  {
	 		SysLibOpen(DexLibRefNum);
	  }
	  else
	  {
	    DexLibRefNum = 0;
	  }
#endif /* __RELEASE__ */
  }

  APPSTART_TRACE_SHOW_INT(14);

  // Vrifier la prsence de la navigation 5-ways
  Error = FtrGet(sysFtrCreator,
                 sysFtrNumFiveWayNavVersion,
                 &value);
#ifdef APPSTART_NO_5_WAYS_NAVIGATION
  Error = ~errNone;
#endif /* APPSTART_NO_5_WAYS_NAVIGATION */
  if (Error == errNone)
  {
    Navigation5WaysActive = 1;
  }

  APPSTART_TRACE_SHOW_INT(15);

  // Register to receive various OS notifications
  regError = false;
  if (SysNotifyRegister(cardNo,
                        dbID,
                        sysNotifyVolumeUnmountedEvent,
                        NULL,
                        sysNotifyNormalPriority,
                        NULL))
    regError=true;
  if (SysNotifyRegister(cardNo,
                        dbID,
                        sysNotifyVolumeMountedEvent,
                        NULL,
                        sysNotifyNormalPriority,
                        NULL))
    regError=true;
#ifdef APPSTART_BAD_SYSNOTIFYREGISTER
  regError = true;
#endif /* APPSTART_BAD_SYSNOTIFYREGISTER */
  if (regError == true)
  {
    FrmAlert(UnableToRegisterAlert);
    return sysErrNoFreeResource;
  }

  APPSTART_TRACE_SHOW_INT(16);

  // Vrifier la prsence du VFS (Virtual File System)
  Error = FtrGet(sysFileCVFSMgr,
                 vfsFtrIDVersion,
                 &value);
#ifdef APPSTART_NO_VFS_NAVIGATION
  Error = ~errNone;
#endif /* APPSTART_NO_VFS_NAVIGATION */
  if (Error != errNone)
  {
    FrmAlert(NoVirtualFileSystemAlert);
    return sysErrNotAllowed;
  }

  APPSTART_TRACE_SHOW_INT(17);

  return errNone;
}
/*----------------------------------------------------------------------------*/


static void AppStop(void)
/***********************************************************************
 *
 *  AppStop
 *
 ***********************************************************************/
{
UInt16 cardNo;
LocalID dbID;
UInt32 Ticks;
UInt16 oldCoordSys;

  // Begin display duration
  Ticks = TimGetTicks() + (SysTicksPerSecond() * SHOWSHUTDOWNTIME_S) + 1;

  // Display shutdown screen
  oldCoordSys = WinSetCoordinateSystem(kCoordinatesNative);
  DisplayShutdownScreen();
  WinSetCoordinateSystem(oldCoordSys);

  // Save CPC context before forms close
  if (AllowSaveSession)
  {
    if ( (prefP->AutomaticContextSave) && (prefP != NULL) )
    {
      tCPCSaveReturnCode ReturnCode = SaveCPC(DEFAULT_CAPRICE_PATH,
                                              LASTSESSION_FILENAME,
                                              contextP);

      if (ReturnCode == File_Not_Created)
      {
        FrmAlert(CPCSaveCreationAlert);
      }
      else if (ReturnCode == Write_Error)
      {
        FrmAlert(BadCPCSaveAlert);
      }
    }
  }

  StopKeyboard(&DIAKeyboardP);
  StopKeyboard(&MiniKeyboardP);

  if (NativeCPC != NULL)
  {
    SoundStop(NativeCPC); // !! Should be done before CPCStop !!
    SaveAndEjectBoth(NativeCPC);
  }
  
  CPCStop();
  CPCLastStop();

  // Wait for shutdown screen minimum duration.
  while (TimGetTicks() < Ticks);
  
  // Close all open forms.
  FrmCloseAllForms();

  // Sauvegarder les prfrences
  if (prefP != NULL)
  {
    if (prefP->PreferencesChanged)
    {
      EcrirePreferences(prefP);
    }

    MemPtrFree(prefP);
  }

  // Librer la palette
  if (originalPaletteP != NULL)
  {
    MemPtrFree(originalPaletteP);
  }

  // Librer lles index des couleurs
  if (colorIndexP != NULL)
  {
    MemPtrFree(colorIndexP);
  }


  //
  // Context restoration
  //
  WinScreenMode(winScreenModeSet,
                NULL,
                NULL,
                &oldScreenDepth,
                NULL);

  SysCurAppDatabase(&cardNo,
                    &dbID);
  SysNotifyUnregister(cardNo,
                      dbID,
                      sysNotifyVolumeUnmountedEvent,
                      sysNotifyNormalPriority);
  SysNotifyUnregister(cardNo,
                      dbID,
                      sysNotifyVolumeMountedEvent,
                      sysNotifyNormalPriority);

  if (collapsibleIA)
  {
    SysNotifyUnregister(cardNo,
                        dbID,
                        sysNotifyDisplayResizedEvent,
                        sysNotifyNormalPriority);

    RestoreDIAState(&oldDIAState);

#ifdef __RELEASE__
    // Close Display Extent Library
    if (DexLibRefNum)
	  {
      SysLibClose(DexLibRefNum);
      SysLibRemove(DexLibRefNum);
	  }
#endif /* __RELEASE__ */
  }

  // Restore Auto Off Time
  SysSetAutoOffTime(oldAutoOffTime);

  // Restaurer le mask des touches matrielles
  KeySetMask(oldHardKeyMask);

  // Free CmdToGo Params
  if (CmdToGoParamsP != NULL)
  {
    MemPtrFree((MemPtr)CmdToGoParamsP);
  }
}
/*----------------------------------------------------------------------------*/


static Boolean AppHandleEvent(EventPtr eventP)
/***********************************************************************
 *
 *  AppHandleEvent
 *
 ***********************************************************************/
{
UInt16 formId;
FormPtr frmP;

  if (eventP->eType == frmLoadEvent)
  {
    // Load the form resource.
    formId = eventP->data.frmLoad.formID;
    frmP = FrmInitForm(formId);
    FrmSetActiveForm(frmP);

    // Set the event handler for the form.  The handler of the currently
    // active form is called by FrmHandleEvent each time is receives an
    // event.
    switch (formId)
    {
      case MainForm:
        FrmSetEventHandler(frmP,
                           MainFormHandleEvent);
        break;

      case AboutForm:
        FrmSetEventHandler(frmP,
                           AboutFormHandleEvent);
        break;

      case DriveForm:
      case LoadForm:
        FrmSetEventHandler(frmP,
                           FilesFormHandleEvent);
        break;

      case DeviceSettingsForm:
        FrmSetEventHandler(frmP,
                           DeviceSettingsFormHandleEvent);
        break;

      case EmulatorSettingsForm:
        FrmSetEventHandler(frmP,
                           EmulatorSettingsFormHandleEvent);
        break;

      case HardKeysForm:
        FrmSetEventHandler(frmP,
                           HardKeysFormHandleEvent);
        break;

#ifndef __RELEASE__
      default:
        ErrDisplay("Bad Form ID");
#endif /* __RELEASE__ */
    }

    return true;
  }

  return false;
}
/*----------------------------------------------------------------------------*/


static Boolean HandleSpecialEvent(EventPtr eventP)
/***********************************************************************
 *
 *  HandleSpecialEvent
 *
 ***********************************************************************/
{
const UInt16 HardKeys[] = { hard1Chr, hard2Chr, hard3Chr, hard4Chr };

  //
  // keyHoldEvent
  //
  if (eventP->eType == keyHoldEvent)
  {
    // Prevent Attention List display
    if ( (eventP->data.keyHold.chr == vchrRockerCenter) ||
         (eventP->data.keyHold.chr == vchrHardRockerCenter) )
    {
      return true;
    }
  }
  //
  // keyDownEvent
  //
  else if (eventP->eType == keyDownEvent)
  {
    UInt8 HardKeyIndex = 0xff;

    if (eventP->data.keyUp.chr == hard1Chr)
    {
      HardKeyIndex = prefP->HardKey1Index;
    }
    else if (eventP->data.keyUp.chr == hard2Chr)
    {
      HardKeyIndex = prefP->HardKey2Index;
    }
    else if (eventP->data.keyUp.chr == hard3Chr)
    {
      HardKeyIndex = prefP->HardKey3Index;
    }
    else if (eventP->data.keyUp.chr == hard4Chr)
    {
      HardKeyIndex = prefP->HardKey4Index;
    }

    // Not a hard key
    if ( (HardKeyIndex == 0xff) ||
         (HardKeyIndex == HARDKEY_NOT_USED_INDEX) )
    {
      // Let system performing hard key operation
      return false;
    }
    // Hard key is "Menu"
    else if (HardKeyIndex == HARDKEY_MENU_INDEX)
    {
      // Hard key = "Menu"
      EventType newEvent;
      newEvent.eType = keyDownEvent;
      newEvent.data.keyDown.chr = vchrMenu;
      newEvent.data.keyDown.keyCode = 0;
      newEvent.data.keyDown.modifiers = commandKeyMask;
      EvtAddEventToQueue(&newEvent);
    }
    // Hard key = "Mini Keyboard"
    else if (HardKeyIndex == HARDKEY_MINI_KEYBOARD_INDEX)
    {
      // Toggle mini keyboard activation by sending menu event
      EventType newEvent;
      newEvent.eType = menuEvent;
      newEvent.data.menu.itemID = DisplayMiniKeyboardMenu;
      EvtAddEventToQueue(&newEvent);
    }
    // Hard key = "Fullscreen"
    else if (HardKeyIndex == HARDKEY_FULLSCREEN_INDEX)
    {
      // Toggle fullscreen activation by sending menu event
      EventType newEvent;
      newEvent.eType = menuEvent;
      newEvent.data.menu.itemID = DisplayFullscreenMenu;
      EvtAddEventToQueue(&newEvent);
    }
    // Hard key = "Decathlon"
    else if (HardKeyIndex == HARDKEY_DECATHLON_INDEX)
    {
      // Toggle Decathlon toggle by sending menu event
      EventType newEvent;
      newEvent.eType = menuEvent;
      newEvent.data.menu.itemID = PlayToggleDecathlonMenu;
      EvtAddEventToQueue(&newEvent);
    }

    // Hard key = "Rocker Center"
    // Hard key = "CPC keycodes"

    // Do not perform what system expects from this hard key.
    return true;
  }
  //
  // menuCmdBarOpenEvent
  //
  else if (eventP->eType == menuCmdBarOpenEvent)
  {
    // Prevent Command Bar to be openned while in Fullscreen
    if (FullscreenActive)
      return true;
  }

  return false;
}
/*----------------------------------------------------------------------------*/


static void AppEventLoop(void)
/***********************************************************************
 *
 *  AppEventLoop
 *
 ***********************************************************************/
#undef APPEVENTLOOP_TRACE_ENABLED
//#define APPEVENTLOOP_TRACE_ENABLED
#undef APPEVENTLOOP_CPCEXECUTE_TRACE_ENABLED
//#define APPEVENTLOOP_CPCEXECUTE_TRACE_ENABLED

#ifdef APPEVENTLOOP_TRACE_ENABLED
#  define APPEVENTLOOP_TRACE_SHOW_INT(value) TRACE_SHOW_INT("AppEventLoop", value)
#else /* APPEVENTLOOP_TRACE_ENABLED */
#  define APPEVENTLOOP_TRACE_SHOW_INT(value)
#endif /* APPEVENTLOOP_TRACE_ENABLED */
{
Err error;
EventType event;
UInt32 Condition;
UInt32 Ticks;
UInt32 NextCycle = 0;
UInt32 TicksPerSecond;
UInt32 TicksPerCycle;
UInt32 NextSecond;
UInt32 NextAutoToggle = 0;
UInt32 CycleCount = 0;
#ifdef PATCH_CPC_TRUE_SPEED
UInt32 LongCycleCount = 0;
#endif /* PATCH_CPC_TRUE_SPEED */
UInt32 Percentage = 0;
UInt32 HardKeysState;
UInt32 oldHardKeyState = 0;
UInt16 oldCoordSys;
UInt8 RunEmulation;
UInt8 oldFDCLed = 0;
UInt8 DiskDisplayCounter = 0;
UInt8 VideoFrameDelay;
#ifdef _DEBUG
UInt32 SoundSamples = 0;
UInt32 SoundCbCount = 0;
UInt32 SoundCbSamples = 0;
UInt8 StepByStep = 0;
#endif /* _DEBUG */
#ifdef _SHOW_FPS
UInt32 FPS;
UInt32 FPSCount;
#endif /* _SHOW_FPS */
char* indexAutoStartP;
UInt32 AutoStartDuration = 0;
UInt8 AutoStartStatus = 0;
UInt8 AutoStartActive = 0;
tUShort oldSHIFTandCTRLState;

  TicksPerSecond = SysTicksPerSecond();
  TicksPerCycle = TicksPerSecond / CYCLES_PER_SECOND;
  NextSecond = TimGetTicks() + TicksPerSecond;
  FDCLedOffScreenDisplay = 0;
  VideoFrameDelay = VideoFrameInitialDelay;
  WestPhaserTriggerActiveDelay = 0;

  do
  {
    // Emulator is running
#ifndef _DEBUG
    if (!SystemHalt)
#else /* _DEBUG */
    if (!SystemHalt || StepByStep)
#endif /* _DEBUG */
    {
      APPEVENTLOOP_TRACE_SHOW_INT(1);

      do
      {
        // Get pending event
        EvtGetEvent(&event, 0);

        Ticks = TimGetTicks();

        RunEmulation = 1;

        APPEVENTLOOP_TRACE_SHOW_INT(2);

        //
        // CPC True speed
        //
        if (prefP->CPCTrueSpeed)
        {
          APPEVENTLOOP_TRACE_SHOW_INT(3);

          if (!NextCycle)
          {
            NextCycle = Ticks + TicksPerCycle;
          }
          // Sound buffer has been filled by engine.
          else if ( (Condition == EC_SOUND_BUFFER) &&
                    (prefP->SoundEnabled) ) // Prevent Freeze when sound turned off while Condition = EC_SOUND_BUFFER
          {
            // Do not allow emulation while sound buffer not entirely read by sound stream callback
            if (IsBufferRead(contextP) == false)
            {
              RunEmulation = 0;
            }
          }
          // CPC True speed limitation
          else if (Condition == EC_CYCLE_COUNT)
          {
            if (Ticks < NextCycle)
            {
              // Wait for next true speed step
              RunEmulation = 0;
#ifdef PATCH_CPC_TRUE_SPEED
              LongCycleCount = 0; // Short cycle detected
#endif /* PATCH_CPC_TRUE_SPEED */
            }
            else
            {
              // Maintain true speed steps
              NextCycle += TicksPerCycle;
#ifdef PATCH_CPC_TRUE_SPEED
              LongCycleCount++;

              // Shorten too long delay
              /*if (Ticks > NextCycle)
              {
                NextCycle = Ticks + TicksPerCycle;
              }*/
#endif /* PATCH_CPC_TRUE_SPEED */
            }
          }
        }
        else
        {
          NextCycle = 0;
#ifdef PATCH_CPC_TRUE_SPEED
          LongCycleCount = 0;
#endif /* PATCH_CPC_TRUE_SPEED */
        }

        // Perform emulation
        if (RunEmulation)
        {
          APPEVENTLOOP_TRACE_SHOW_INT(4);

          //
          // Auto toggle
          //
          if (!AutoToggleActive)
          {
            if (NextAutoToggle)
            {
              NextAutoToggle = 0;

              SetCursorAndJoystickState(KeyCurrentState(),
                                        &oldHardKeyState);
            }
          }
          else
          {
            if ( (Ticks >= NextAutoToggle) || (!NextAutoToggle) )
            {
              ToggleAutoToggle();
              NextAutoToggle = Ticks + AutoToggleDurationInTicks;

              SetCursorAndJoystickState(KeyCurrentState(),
                                        &oldHardKeyState);
            }
          }

          APPEVENTLOOP_TRACE_SHOW_INT(5);

#if defined(_TRACE) && defined(APPEVENTLOOP_CPCEXECUTE_TRACE_ENABLED)
          NativeCPC->TraceInstruction = 1;
          NativeCPC->TraceDisplay = 1;
#endif /* _TRACE && APPEVENTLOOP_CPCEXECUTE_TRACE_ENABLED */

          // =================
          // Run Emulation
          // =================
          Condition = CPCExecute();

          APPEVENTLOOP_TRACE_SHOW_INT(6);

#ifdef _TRACE
          if (NativeCPC->TraceStop)
          {
            APPEVENTLOOP_TRACE_SHOW_INT(7);

            // Emulation paused
            EmulatorFreeze();
            EmulatorKeys[KEYINDEX_PAUSE].KeyStatus = KeyPressed;
            FrmUpdateForm(MainForm,
                          emulatorRedrawUpdateCode);
            NativeCPC->TraceStop = 0;
          }
#endif /* _TRACE */

#ifdef _TESTU
          if (Condition >= appErrorClass)
          {
            APPEVENTLOOP_TRACE_SHOW_INT(8);

            TRACE_SHOW_HEX("TestU Native_CPCExecute\nSystem Halted", Condition-testUErrorClass);
            // Emulation paused
            EmulatorFreeze();
            EmulatorKeys[KEYINDEX_PAUSE].KeyStatus = KeyPressed;
            FrmUpdateForm(MainForm,
                          emulatorRedrawUpdateCode);
            // Create debug event to stop emulation
            event.eType = debugBreakpointEvent;
          }
#endif /* _TESTU */

#ifdef _DEBUG
          // ================
          // Breakpoint
          // ================
          if (Condition == EC_BREAKPOINT)
          {
            tZ80* Z80 = (tZ80*)EndianSwap32((tULong)NativeCPC->Z80);
            Z80->Regs.breakpoint = 0xffffffff;

            APPEVENTLOOP_TRACE_SHOW_INT(9);

            // Emulation paused
            EmulatorFreeze();
            EmulatorKeys[KEYINDEX_PAUSE].KeyStatus = KeyPressed;
            FrmUpdateForm(MainForm,
                          emulatorRedrawUpdateCode);

            // Create debug event to stop emulation
            event.eType = debugBreakpointEvent;
          }
          else
#endif /* _DEBUG */

          // =======================
          // VDU Frame completed
          // =======================
          if (Condition == EC_FRAME_COMPLETE)
          {
            tCRTC* CRTC = (tCRTC*)EndianSwap32(NativeCPC->CRTC);

            APPEVENTLOOP_TRACE_SHOW_INT(10);

            CRTC->stop_rendering = VideoFrameDelay > 1 ? 1 : 0;

            if (!(--VideoFrameDelay))
            {
#ifdef _SHOW_FPS
              FPSCount++;
#endif /* _SHOW_FPS */

              APPEVENTLOOP_TRACE_SHOW_INT(11);

              VideoFrameDelay = VideoFrameInitialDelay;

              // Stay out of Native coordinate system
              if (ScreenshotRequested)
              {
                APPEVENTLOOP_TRACE_SHOW_INT(12);

                ScreenshotRequested = 0;
                // Allow 5-ways navigation
                KeySetMask(oldHardKeyMask);
                // Save screenshot
                SaveScreenshot(NativeCPC);
                // Restore key mak for emulation
                KeySetMask(emulatorHardKeyMask);
              }

              oldCoordSys = WinSetCoordinateSystem(kCoordinatesNative);

              //
              // All display before DisplayCPCScreen should be performed Offscreen !!
              //
              if (OnScreenRockerHelpDisplay)
              {
                DisplayOffScreenRockerHelp();
              }

#ifdef _DEBUG
              if (SoundSamples)
              {
                DisplayOffscreenSoundData(SoundSamples,
                                          SoundCbCount,
                                          SoundCbSamples);
              }
#endif /* _DEBUG */

#if defined(_DEBUG) || defined(_SHOW_KEYS)
              DisplayOffscreenKeyState();
#endif /* _DEBUG || _SHOW_KEYS */

#ifdef _SHOW_FPS
              DisplayOffscreenFPS(FPS);
#endif /* _SHOW_FPS */

              // Priority to show message
              if (showMessageTick > Ticks)
              {
                DisplayOffscreenMessage(messageToShowP);
              }
              // Update Speed display
              else if (DisplayEmuSpeed)
              {
                DisplayOffScreenEmulatorSpeed(Percentage); // Done offscreen before DisplayCPCScreen !!
              }

              if (AutoToggleActive)
              {
                DisplayOffScreenAutoToggleSpeed(AutoToggleDurationIndex);
              }
              else if (AdjustSoundVolumeActive)
              {
                DisplayOffScreenSoundVolumeAdjust(prefP->SoundVolume);
              }

              // Update FDC Led Display
              if (!DiskDisplayCounter)
              {
                if (NativeCPC->drive_led != oldFDCLed)
                {
                  oldFDCLed = NativeCPC->drive_led;

                  if (oldFDCLed)
                  {
                    DiskDisplayCounter = DISK_DISPLAY_COUNT;
                  }

                  if (!FDCLedOffScreenDisplay)
                  {
                    DisplayOnScreenFDCLed(oldFDCLed);
                  }
                }
              }
              else
              {
                DiskDisplayCounter--;

                if (FDCLedOffScreenDisplay)
                {
                  DisplayOffScreenDisk();
                }
              }

              // Display Key Status
              DisplayOffScreenKeyStatus();

#ifdef PATCH_CPC_TRUE_SPEED
              DISPLAY_OFFSCREEN_DATA("LongCycle", LongCycleCount, 0, 0);
#endif /* PATCH_CPC_TRUE_SPEED */

              APPEVENTLOOP_TRACE_SHOW_INT(13);
              
              // Transfer offscreen to draw window
              DisplayCPCScreen(prefP->ScreenDisplayOffsetX,
                               prefP->ScreenDisplayOffsetY);

              APPEVENTLOOP_TRACE_SHOW_INT(14);

              WinSetCoordinateSystem(oldCoordSys);
            }
          } /* if (Condition == EC_FRAME_COMPLETE) */

          // ===============
          // Cycle end
          // ===============
          else if (Condition == EC_CYCLE_COUNT)
          {
            APPEVENTLOOP_TRACE_SHOW_INT(15);

            // Update speed measure
            CycleCount++;
            if (Ticks >= NextSecond)
            {
              Percentage = !DisplayEmuSpeed ? 0 : CycleCount * 100 / CYCLES_PER_SECOND;
              NextSecond += TicksPerSecond;
              CycleCount = 0;
#ifdef _DEBUG
              {
                // Get Sound samples
                tPSG* PSG = (tPSG*)EndianSwap32(NativeCPC->PSG);
                SoundSamples = EndianSwap32(PSG->SampleCount);
                PSG->SampleCount = 0;

                SoundCbCount = GetSoundDebugInfo_Count(contextP);
                SoundCbSamples = GetSoundDebugInfo_Samples(contextP);
                ResetSoundDebugInfo(contextP);
              }
#endif /* _DEBUG */

#ifdef _SHOW_FPS
              FPS = FPSCount;
              FPSCount = 0;
#endif /* _SHOW_FPS */

            }

            // if Autostart enable
            if ( (prefP->AutoStartEnable) || (AutoStartCommand[0]) )
            {
              if (!AutoStartActive)
              {
                AutoStartDuration++;
                if (AutoStartDuration > AUTOSTARTRESET_CYCLES)
                {
                  AutoStartActive = 1;
                }
              }
              // Start Autostart command
              else if (AutoStartCommand[0])
              {
                switch (AutoStartStatus)
                {
                  case 0: // Start
                  {
                    indexAutoStartP = AutoStartCommand;
                    // Prepare Hold Duration
                    AutoStartDuration = AUTOSTARTHOLD_CYCLES;
                    // Save SHIFT and CTRL state before autostart
                    oldSHIFTandCTRLState = GetKeySHIFTandCTRLState();
                    // Press fist key
                    KeyboardSetAsciiKeyDown(*indexAutoStartP);
                    // Next Step
                    AutoStartStatus = 1;
                  }
                  break;

                  case 1: // Hold Period
                  {
                    if (!(--AutoStartDuration))
                    {
                      // Release key
                      KeyboardSetAsciiKeyUp(*indexAutoStartP);
                      // Prepare Release Duration
                      AutoStartDuration = AUTOSTARTRELEASE_CYCLES;
                      // Next Step
                      AutoStartStatus = 2;
                    }
                  }
                  break;

                  case 2: // Release key
                  {
                    if (!(--AutoStartDuration))
                    {
                      // If another key to press
                      if (*(indexAutoStartP+1))
                      {
                        // Prepare Hold Duration
                        AutoStartDuration = AUTOSTARTHOLD_CYCLES;
                        // Press next key
                        KeyboardSetAsciiKeyDown(*(++indexAutoStartP));
                        AutoStartStatus = 1;
                      }
                      else // End of string
                      {
                        // Stop and Prepare for next Autostart
                        AutoStartCommand[0] = 0;
                        AutoStartStatus = 0;

                        // Restore SHIFT and CTRL state before autostart
                        SetKeySHIFTandCTRLState(oldSHIFTandCTRLState);
                      }
                    }
                  }
                  break;
                }
              }
            }

            ReleasePressedKey();

            // Cursor/Joystick detection
            SetCursorAndJoystickState(KeyCurrentState(),
                                      &oldHardKeyState);
          } /* if (Condition == EC_CYCLE_COUNT) */

          if (WestPhaserTriggerActiveDelay)
          {
            if (TimGetTicks() > WestPhaserTriggerActiveDelay)
            {
            	RELEASE_KEY(WEST_PHASER_TRIG_KEY);
              WestPhaserTriggerActiveDelay = 0;
              NativeCPC->lightgun_beam_detect = 1;
            }
          }

#ifdef _TRACE
          if (NativeCPC->TraceInstruction)
          {
            UInt16 oldCoordSys = WinSetCoordinateSystem(kCoordinatesNative);

            APPEVENTLOOP_TRACE_SHOW_INT(16);

            DisplayTraceInfo(NativeCPC);
            WinSetCoordinateSystem(oldCoordSys);
          }
#endif /* _TRACE */
        }
      }
#ifndef _DEBUG
      while (event.eType == nilEvent);
#else /* _DEBUG */
      while ( (event.eType == nilEvent) && !StepByStep );
      StepByStep = 0;
#endif /* _DEBUG */
    }
    // Emulator is halted
    else
    {
      // Emulation stopped, typical event loop
      EvtGetEvent(&event, evtWaitForever);

      // Update speed measure
      Ticks = TimGetTicks();
      NextCycle = 0;
      NextSecond = Ticks + TicksPerSecond;
      CycleCount = 0;
      Percentage = 0;
      NextAutoToggle = 0;
#ifdef _SHOW_FPS
      FPS = 0;
      FPSCount = 0;
#endif /* _SHOW_FPS */
    }

    //
    // Events management
    //
#ifdef _DEBUG
    if (event.eType == keyDownEvent)
    {
      APPEVENTLOOP_TRACE_SHOW_INT(17);

      // Hard Key 1 = Pas  Pas en Pause
      if (event.data.keyHold.chr == hard1Chr)
      {
        StepByStep = 1;
        continue;
      }
      else if (event.data.keyHold.chr == hard2Chr)
      {
        continue;
      }
    }
#endif

    // Allow 5-Ways navigation
    KeySetMask(oldHardKeyMask);

    //APPEVENTLOOP_TRACE_SHOW_INT(18);

    PROFILE_ADD_68K(PROFILE_HandleSpecialEvent);
    if (HandleSpecialEvent(&event) == false)
    {
      //APPEVENTLOOP_TRACE_SHOW_INT(19);
      PROFILE_ADD_68K(PROFILE_SysHandleEvent);
      if (SysHandleEvent(&event) == false)
      {
        //APPEVENTLOOP_TRACE_SHOW_INT(20);
        PROFILE_ADD_68K(PROFILE_MenuHandleEvent);
        if (MenuHandleEvent(0,
                            &event,
                            &error) == false)
        {
          //APPEVENTLOOP_TRACE_SHOW_INT(21);
          PROFILE_ADD_68K(PROFILE_AppHandleEvent);
          if (AppHandleEvent(&event) == false)
          {
            //APPEVENTLOOP_TRACE_SHOW_INT(22);
            PROFILE_ADD_68K(PROFILE_FrmDispatchEvent);
            FrmDispatchEvent(&event);
          }
        }
      }
      
      if (WinGetDrawWindow() != FrmGetWindowHandle(FrmGetFormPtr(MainForm)))
      {
        SystemHalt = 1;
      }
    }

    // Mask hard keys events for emulation
    // Mask may have been changed by EmulatorFreeze/EmulatorUnfrezze
    // called by FrmDispatchEvent
    KeySetMask(emulatorHardKeyMask);
  }
  while (event.eType != appStopEvent);

  APPEVENTLOOP_TRACE_SHOW_INT(23);
}
/*----------------------------------------------------------------------------*/


tVoid ShowMessage(tUChar noMessage)
/***********************************************************************
 *
 *  ShowMessage
 *
 ***********************************************************************/
{
  messageToShowP = EmulatorMessages[noMessage];

  showMessageTick = TimGetTicks();
  showMessageTick += SysTicksPerSecond() * SHOWMESSAGETIME_S;
}
/*----------------------------------------------------------------------------*/
tVoid HideMessage(tVoid)
/***********************************************************************
 *
 *  HideMessage
 *
 ***********************************************************************/
{
  showMessageTick = 0;
}
/*----------------------------------------------------------------------------*/


tVoid UpdateFDCLedOffScreenDisplay(tVoid)
/***********************************************************************
 *
 *  UpdateFDCLedOffScreenDisplay
 *
 ***********************************************************************/
{
  FDCLedOffScreenDisplay = 1;

  do
  {
    if (SubPanelActive)
    {
      continue;
    }

    if ( MiniKeyboardActive && !prefP->Display320x480 )
    {
      continue;
    }

    if (FullscreenActive)
    {
      continue;
    }

    FDCLedOffScreenDisplay = 0;
  }
  while (0);
}
/*----------------------------------------------------------------------------*/


tUChar GetShowCPCKeycodes(tVoid)
/***********************************************************************
 *
 *  GetShowCPCKeycodes
 *
 ***********************************************************************/
{
  return ShowCPCKeycodes;
}
/*----------------------------------------------------------------------------*/
tVoid SetShowCPCKeycodes(tUChar show)
/***********************************************************************
 *
 *  SetShowCPCKeycodes
 *
 ***********************************************************************/
{
  ShowCPCKeycodes = show;
}
/*----------------------------------------------------------------------------*/
