/*
    CaPriCe for Palm OS - Amstrad CPC 464/664/6128 emulator for Palm devices
    Copyright (C) 2009  Frdric Coste

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <PalmOS.h>
#include <ByteOrderUtils.h>

#include "CaPriCe.h"
#include "Routines.h"
#include "Resources.h"
#include "Keyboard.h"
#include "CPC.h"
#include "sound.h"
#include "trace.h"
#include "Files.h"
#include "Display.h"


//===================
// PATCH begin
#ifdef _PATCH_ENABLE

#endif /* _PATCH_ENABLE */
// PATCH end
//===================


// Large PNO Native_CPCExecute
#define PNO_CPCEXECUTE_SIZE           ((tULong)117076)
#define PNO_CPCEXECUTE_CRC32          ((tULong)0xc795025c)
#define PNO_CPCEXECUTE_FIRSTRESID     5000

MemPtr PNOCPCExecuteP = NULL;

WinHandle OffScreen = NULL;

tUChar* contextP = NULL;
tNativeCPC* NativeCPC = NULL;

//FC!! will be placed into resources area
static MemHandle colours_rgbH = NULL;
static MemHandle colours_greenH = NULL;
static MemPtr FDCCommandTableP = NULL;
static MemHandle DAATableH = NULL;


/***********************************************************************
 *
 *  Internal Functions
 *
 ***********************************************************************/
static void ShowTraceAlert(tULong Number)
/***********************************************************************
 *
 *  ShowTraceAlert
 *
 ***********************************************************************/
{
#ifdef _TRACE
  UInt16 oldCoordSys = WinSetCoordinateSystem(kCoordinatesNative);
  DisplayNativeTrace(Number);
  WinSetCoordinateSystem(oldCoordSys);
#endif /* _TRACE */
}
/*----------------------------------------------------------------------------*/


static void* PalmOS_MemPtrNewLarge(tULong size)
/***********************************************************************
 *
 *  PalmOS_MemPtrNewLarge
 *
 ***********************************************************************/
{
void* ptrP;

  if (MemPtrNewLarge(size, &ptrP) != errNone)
    return NULL;

  return ptrP;
}
/*----------------------------------------------------------------------------*/


static Err PalmOS_WinPalette(RGBColorType* tableP,
                             Int16 startIndex,
                             UInt16 paletteEntries,
                             UInt8 operation)
/***********************************************************************
 *
 *  PalmOS_WinPalette
 *
 ***********************************************************************/
{
	return WinPalette(operation,
	                  startIndex,
	                  paletteEntries,
	                  tableP);
}
/*----------------------------------------------------------------------------*/


Err CPCFirstStart(void)
/***********************************************************************
 *
 *  CPCFirstStart
 *
 ***********************************************************************/
#undef CPCFIRSTSTART_TRACE_ENABLED
//#define CPCFIRSTSTART_TRACE_ENABLED

#ifdef CPCFIRSTSTART_TRACE_ENABLED
#  define CPCFIRSTSTART_TRACE_SHOW_INT(value) TRACE_SHOW_INT("CPCFirstStart", value)
#else /* CPCFIRSTSTART_TRACE_ENABLED */
#  define CPCFIRSTSTART_TRACE_SHOW_INT(value)
#endif /* CPCFIRSTSTART_TRACE_ENABLED */
{
tContextResources* resourcesP;
tContextHeader* ContextHeaderP;
BitmapType* bmpOffScreen;
MemHandle resH;
MemPtr resP;
Err ReturnCode = errNone;
Err Error;
UInt16 oldCoordSys;
UInt32 Size;
#ifndef __RELEASE__
Coord OffscreenWidth;
Coord OffscreenHeight;
#endif /* __RELEASE__ */

  CPCFIRSTSTART_TRACE_SHOW_INT(1);
  
  oldCoordSys = WinSetCoordinateSystem(kCoordinatesNative);

  CPCFIRSTSTART_TRACE_SHOW_INT(2);
  
  do
  {
    //
    // Allocate context memory
    //
    MemPtrNewLarge(SIZETAB_CONTEXT,
                   (tVoid**)&contextP);
    if (contextP == NULL)
    {
      FrmAlert(NotEnoughMemoryAlert);
      ReturnCode = memErrNotEnoughSpace;
      continue;
    }

    MemSet(contextP,
           SIZETAB_CONTEXT,
           0);
    
    CPCFIRSTSTART_TRACE_SHOW_INT(3);

    // Prepare context header
    ContextHeaderP = (tContextHeader*)(contextP + CONTEXT_OFFSET_HEADER);
    
    // Get Version string resource and add it to string
    resH = DmGetResource(verRsc,
                         1000);
#ifndef __RELEASE__
    if (resH == NULL)
      ErrDisplay("resH == NULL");
#endif /* __RELEASE__ */

    resP = MemHandleLock(resH);
#ifndef __RELEASE__
    if (resP == NULL)
      ErrDisplay("resP == NULL");
#endif /* __RELEASE__ */

    StrCopy(ContextHeaderP->version,
            resP);
    MemHandleUnlock(resH);
    DmReleaseResource(resH);

    CPCFIRSTSTART_TRACE_SHOW_INT(4);

    // Pointer affectation into context memory
    resourcesP = (tContextResources*)(contextP + CONTEXT_OFFSET_RESOURCES);
    
    //
    // Crer le buffer d'cran
    //
    OffScreen = WinCreateOffscreenWindow(CPC_SCR_WIDTH,
                                         CPC_SCR_HEIGHT,
                                         nativeFormat,
                                         &Error);
    if ( (OffScreen == NULL) || (Error != errNone) )
    {
      FrmAlert(NoOffscreenWindowsAlert);
      ReturnCode = sysErrNotAllowed;
      continue;
    }
    
    bmpOffScreen = WinGetBitmap(OffScreen);
#ifndef __RELEASE__
    if (bmpOffScreen == NULL)
      ErrDisplay("bmpOffScreen == NULL");
#endif /* __RELEASE__ */

    resourcesP->BmpOffScreenBits = BmpGetBits(bmpOffScreen);

    CPCFIRSTSTART_TRACE_SHOW_INT(5);

    //
    // Bitmap size
    //
    BmpGetDimensions(bmpOffScreen,
#ifdef __RELEASE__
                     NULL,
                     NULL,
#else /* __RELEASE__ */
                     &OffscreenWidth,
                     &OffscreenHeight,
#endif /* __RELEASE__ */
                     &resourcesP->BmpOffScreenBytesRow);

#ifndef __RELEASE__
    if (OffscreenWidth != CPC_SCR_WIDTH)
      ErrDisplay("OffscreenWidth != CPC_SCR_WIDTH");
    if (OffscreenHeight != CPC_SCR_HEIGHT)
      ErrDisplay("OffscreenHeight != CPC_SCR_HEIGHT");
#endif /* __RELEASE__ */

    CPCFIRSTSTART_TRACE_SHOW_INT(6);

    //
    // Lire les tables de constantes
    //
    // colours_rgb table
    colours_rgbH = DmGetResource(TABLE_ID_TYPE,
                                 TABLE_ID_COLOURS_RGB);
    resourcesP->colours_rgb = (colours_rgb_entry*)MemHandleLock(colours_rgbH);
#ifndef __RELEASE__
    if (colours_rgbH == NULL)
      ErrDisplay("colours_rgbH == NULL");
    if (resourcesP->colours_rgb == NULL)
      ErrDisplay("resourcesP->colours_rgb == NULL");
#endif /* __RELEASE__ */

    CPCFIRSTSTART_TRACE_SHOW_INT(7);

    // colours_green table
    colours_greenH = DmGetResource(TABLE_ID_TYPE,
                                   TABLE_ID_COLOURS_GREEN);
    resourcesP->colours_green = (colours_rgb_entry*)MemHandleLock(colours_greenH);
#ifndef __RELEASE__
    if (colours_greenH == NULL)
      ErrDisplay("colours_greenH == NULL");
    if (resourcesP->colours_green == NULL)
      ErrDisplay("resourcesP->colours_green == NULL");
#endif /* __RELEASE__ */

    CPCFIRSTSTART_TRACE_SHOW_INT(8);

    // FDCCommandTable table
    resH = DmGetResource(TABLE_ID_TYPE,
                         TABLE_ID_FDC_CMD_TABLE);
    resP = MemHandleLock(resH);
#ifndef __RELEASE__
    if (resH == NULL)
      ErrDisplay("resH == NULL");
    if (resP == NULL)
      ErrDisplay("resP == NULL");
#endif /* __RELEASE__ */
    Size = MemHandleSize(resH);
    FDCCommandTableP = (fdc_cmd_table_def*)MemPtrNew(Size);
    if (FDCCommandTableP == NULL)
    {
      FrmAlert(NotEnoughMemoryAlert);
      MemHandleUnlock(resH);
      DmReleaseResource(resH);
      ReturnCode = memErrNotEnoughSpace;
      continue;
    }
    MemMove(FDCCommandTableP,
            resP,
            Size);
    MemHandleUnlock(resH);
    DmReleaseResource(resH);
    resourcesP->FDCCommandTableP = FDCCommandTableP;

    CPCFIRSTSTART_TRACE_SHOW_INT(9);

    // DAATable table
    DAATableH = DmGetResource(TABLE_ID_TYPE,
                              TABLE_ID_DAATABLE);
    resourcesP->DAATable = (tUShort*)MemHandleLock(DAATableH);
#ifndef __RELEASE__
    if (DAATableH == NULL)
      ErrDisplay("DAATableH == NULL");
    if (resourcesP->DAATable == NULL)
      ErrDisplay("resourcesP->DAATable == NULL");
#endif /* __RELEASE__ */

    CPCFIRSTSTART_TRACE_SHOW_INT(10);

    //
    // Build CPC_Execute largePNO
    //
    PNOCPCExecuteP = DmCreateLargeResource(appFileCreator,
                                           CPCExecuteFtrNum,
                                           'tbin',
                                           PNO_CPCEXECUTE_FIRSTRESID,
                                           PNO_CPCEXECUTE_SIZE);
    if (PNOCPCExecuteP == NULL)
    {
      FrmAlert(NotEnoughMemoryAlert);
      ReturnCode = memErrNotEnoughSpace;
      continue;
    }

    CPCFIRSTSTART_TRACE_SHOW_INT(11);

    //
    // Check CPCExecute largePNO integrity
    //
    if (ComputeCRC32((tUChar*)PNOCPCExecuteP,
                     PNO_CPCEXECUTE_SIZE) != PNO_CPCEXECUTE_CRC32)
    {
      FrmAlert(LargePNOCorruptedAlert);
      ReturnCode = sysErrNotAllowed;
      continue;
    }

    CPCFIRSTSTART_TRACE_SHOW_INT(12);
  }
  while (0);

  WinSetCoordinateSystem(oldCoordSys);

  CPCFIRSTSTART_TRACE_SHOW_INT(13);

  return ReturnCode;
}
/*----------------------------------------------------------------------------*/


static Err CPCStart(void)
/***********************************************************************
 *
 *  CPCStart
 *
 ***********************************************************************/
#undef CPCSTART_TRACE_ENABLED
//#define CPCSTART_TRACE_ENABLED

#ifdef CPCSTART_TRACE_ENABLED
#  define CPCSTART_TRACE_SHOW_INT(value) TRACE_SHOW_INT("CPCStart", value)
#else /* CPCSTART_TRACE_ENABLED */
#  define CPCSTART_TRACE_SHOW_INT(value)
#endif /* CPCSTART_TRACE_ENABLED */
{
UInt32 Result;
Err ReturnCode = errNone;
MemHandle resH;
MemPtr resP;
tContextResources* resourcesP;

  CPCSTART_TRACE_SHOW_INT(1);

  // Pointer affectation into context memory
  NativeCPC = (tNativeCPC*)(contextP + CONTEXT_OFFSET_NATIVECPC);
  sessionFilenameP = (char*)(contextP + CONTEXT_OFFSET_SESSIONFILENAME);
  resourcesP = (tContextResources*)(contextP + CONTEXT_OFFSET_RESOURCES);

  CPCSTART_TRACE_SHOW_INT(2);

  //
  // Initialiser la structure
  //
  NativeCPC->prefP = prefP;
  NativeCPC->contextP = contextP;
  NativeCPC->BmpOffScreenBits = resourcesP->BmpOffScreenBits;
  NativeCPC->BmpOffScreenBytesRow = resourcesP->BmpOffScreenBytesRow;

#ifndef __RELEASE__
    if (NativeCPC->prefP == NULL)
      ErrDisplay("NativeCPC->prefP == NULL");
    if (NativeCPC->contextP == NULL)
      ErrDisplay("NativeCPC->contextP == NULL");
#endif /* __RELEASE__ */

  NativeCPC->TraceAlertPtr = (tULong)ShowTraceAlert;
  NativeCPC->MemPtrNewPtr = (tULong)PalmOS_MemPtrNewLarge;
  NativeCPC->WinPalettePtr = (tULong)PalmOS_WinPalette;
  NativeCPC->SoundCalculateLevelTablesPtr = (tULong)Sound_Calculate_Level_Tables;
  NativeCPC->HardKeyCPCKeyCodeA = prefP->CPCKeycodeA;
  NativeCPC->HardKeyCPCKeyCodeB = prefP->CPCKeycodeB;
  NativeCPC->HardKeyCPCKeyCodeC = prefP->CPCKeycodeC;
  NativeCPC->HardKeyCPCKeyCodeD = prefP->CPCKeycodeD;

  CPCSTART_TRACE_SHOW_INT(3);

  // Lire les tables de constantes
  NativeCPC->colours_rgb = resourcesP->colours_rgb;
  NativeCPC->colours_green = resourcesP->colours_green;
  NativeCPC->FDCCommandTable = resourcesP->FDCCommandTableP;
  NativeCPC->DAATable = resourcesP->DAATable;

  CPCSTART_TRACE_SHOW_INT(4);

  do
  {
    //
    // Initialisation de l'mulateur
    //
    Result = PceNativeCallResource(Native_CPCStart,
#ifdef __SIMU__
                                   "Native_CPCStart.dll\0PNOMain",
#endif /* __SIMU__ */
                                   NativeCPC);
#ifdef _TESTU
    if (Result >= appErrorClass)
    {
      TRACE_SHOW_HEX("TestU Native_CPCStart", Result-testUErrorClass);
    }
    else
#endif
    if (Result != errNone)
    {
      FrmAlert(NotEnoughMemoryAlert);
      ReturnCode = memErrNotEnoughSpace;
      continue;
    }

    CPCSTART_TRACE_SHOW_INT(5);

    CPCSetColor();

    CPCSTART_TRACE_SHOW_INT(6);

    CPCReset();

    CPCSTART_TRACE_SHOW_INT(7);

    //
    // CmdToGo Image Disk Loading
    //
    if (CmdToGoParamsP)
    {
    	tUChar* dataP;

      SetDriveFilename(NativeCPC->DriveA,
                       (tUChar*)CmdToGoParamsP + CmdToGoParamsP->descriptionOffset);
                       
      // Allocate memory for whole disk data
      MemPtrNewLarge(CmdToGoParamsP->fileSize,
                     (void**)&dataP);
      if (dataP == NULL)
      {
        FrmAlert(NotEnoughMemoryAlert);
      }
      else
      {
        // Copy Disk Image Content
        MemMove(dataP,
                (tUChar*)CmdToGoParamsP + CmdToGoParamsP->fileOffset,
                CmdToGoParamsP->fileSize);
        
    	  LoadDiskImageFromMemory(NativeCPC->DriveA,
    	                          NativeCPC,
    	                          EndianSwap32(CmdToGoParamsP->fileSize),
    	                          (tUChar*)EndianSwap32(dataP));

        // Autostart enable
        if (prefP->AutoStartEnable)
        {
        	// Get command to execute
        	DiskAutostart(NativeCPC,
        		            NativeCPC->DriveA,
        		            AutoStartCommand);
        }
      }
                       
      // Free CmdToGo Params
      MemPtrFree((MemPtr)CmdToGoParamsP);
      CmdToGoParamsP = NULL;
    }

    CPCSTART_TRACE_SHOW_INT(8);
  }
  while (0);

  CPCSTART_TRACE_SHOW_INT(9);

  return ReturnCode;
}
/*----------------------------------------------------------------------------*/


void CPCStop(void)
/***********************************************************************
 *
 *  CPCStop
 *
 ***********************************************************************/
#undef CPCSTOP_TRACE_ENABLED
//#define CPCSTOP_TRACE_ENABLED

#ifdef CPCSTOP_TRACE_ENABLED
#  define CPCSTOP_TRACE_SHOW_INT(value) TRACE_SHOW_INT("CPCStop", value)
#else /* CPCSTOP_TRACE_ENABLED */
#  define CPCSTOP_TRACE_SHOW_INT(value)
#endif /* CPCSTOP_TRACE_ENABLED */
{
	CPCSTOP_TRACE_SHOW_INT(1);
	
  if (NativeCPC != NULL)
  {
	  CPCSTOP_TRACE_SHOW_INT(2);
	  
    // Arrter l'mulation.
    PceNativeCallResource(Native_CPCStop,
#ifdef __SIMU__
                          "Native_CPCStop.dll\0PNOMain",
#endif /* __SIMU__ */
                          NativeCPC);

    NativeCPC = NULL;
  }

	CPCSTOP_TRACE_SHOW_INT(3);
}
/*----------------------------------------------------------------------------*/


void CPCLastStop(void)
/***********************************************************************
 *
 *  CPCLastStop
 *
 ***********************************************************************/
#undef CPCLASTSTOP_TRACE_ENABLED
//#define CPCLASTSTOP_TRACE_ENABLED

#ifdef CPCLASTSTOP_TRACE_ENABLED
#  define CPCLASTSTOP_TRACE_SHOW_INT(value) TRACE_SHOW_INT("CPCLastStop", value)
#else /* CPCLASTSTOP_TRACE_ENABLED */
#  define CPCLASTSTOP_TRACE_SHOW_INT(value)
#endif /* CPCLASTSTOP_TRACE_ENABLED */
{
tContextResources* resourcesP;

  CPCLASTSTOP_TRACE_SHOW_INT(1);
  
  // Pointer affectation into context memory
  resourcesP = (tContextResources*)(contextP + CONTEXT_OFFSET_RESOURCES);

  CPCLASTSTOP_TRACE_SHOW_INT(2);

  if (PNOCPCExecuteP != NULL)
  {
    FtrPtrFree(appFileCreator, CPCExecuteFtrNum);
    PNOCPCExecuteP = NULL;
  }

  CPCLASTSTOP_TRACE_SHOW_INT(3);
  
  if (OffScreen != NULL)
  {
    WinDeleteWindow(OffScreen, false);
    OffScreen = NULL;
  }

  CPCLASTSTOP_TRACE_SHOW_INT(4);
  
  if (colours_rgbH != NULL)
  {
    MemHandleUnlock(colours_rgbH);
    DmReleaseResource(colours_rgbH);
  }

  CPCLASTSTOP_TRACE_SHOW_INT(5);
  
  if (colours_greenH != NULL)
  {
    MemHandleUnlock(colours_greenH);
    DmReleaseResource(colours_greenH);
  }

  CPCLASTSTOP_TRACE_SHOW_INT(6);

  if (DAATableH != NULL)
  {
    MemHandleUnlock(DAATableH);
    DmReleaseResource(DAATableH);
  }
  
  CPCLASTSTOP_TRACE_SHOW_INT(7);
  
  if (FDCCommandTableP != NULL)
  {
    MemPtrFree(FDCCommandTableP);
    FDCCommandTableP = NULL;
  }

  CPCLASTSTOP_TRACE_SHOW_INT(8);
  
  if (contextP != NULL)
  {
    MemPtrFree(contextP);
    contextP = NULL;
  }

  CPCLASTSTOP_TRACE_SHOW_INT(9);
}
/*----------------------------------------------------------------------------*/


void CPCReset(void)
/***********************************************************************
 *
 *  CPCReset
 *
 ***********************************************************************/
{
Err Result;

  // ARM Reset
  Result = (Err)PceNativeCallResource(Native_CPCReset,
#ifdef __SIMU__
                                      "Native_CPCReset.dll\0PNOMain",
#endif /* __SIMU__ */
                                      NativeCPC);

#ifdef _TESTU
  if (Result >= appErrorClass)
  {
    TRACE_SHOW_HEX("TestU Native_CPCReset", Result);
  }
#endif /* _TESTU */

#ifdef _TRACE
  {
    tZ80* Z80 = (tZ80*)EndianSwap32((tULong)NativeCPC->Z80);
    Z80->Regs.breakpoint = NativeCPC->TraceInstruction ? EndianSwap32(TRACE_BREAKPOINT) : 0xffffffff;
  }
#endif /* _TRACE */

  SetRenderingParameters(prefP->Rendering);
  SetLightGunParameters(prefP,
                        NativeCPC);
}
/*----------------------------------------------------------------------------*/


void CPCResetWithConfirmation(void)
/***********************************************************************
 *
 *  CPCResetWithConfirmation
 *
 ***********************************************************************/
{
UInt32 hardKeyMask;

  // Allow 5-way rocker navigation
  hardKeyMask = KeySetMask(oldHardKeyMask);
  
  if (FrmAlert(ConfirmResetAlert) == 1)
  {
    CPCReset();
  }

  KeySetMask(hardKeyMask);
}
/*----------------------------------------------------------------------------*/


Err CPCColdReset(const char* pathP,
                 const char* filenameP,
                 tAutomaticLoadType Load)
/***********************************************************************
 *
 *  CPCColdReset
 *
 ***********************************************************************/
#undef CPCCOLDRESET_TRACE_ENABLED
//#define CPCCOLDRESET_TRACE_ENABLED

#ifdef CPCCOLDRESET_TRACE_ENABLED
#  define CPCCOLDRESET_TRACE_SHOW_INT(value) TRACE_SHOW_INT("CPCColdReset", value)
#else /* CPCCOLDRESET_TRACE_ENABLED */
#  define CPCCOLDRESET_TRACE_SHOW_INT(value)
#endif /* CPCCOLDRESETTRACE_ENABLED */
{
Err Result;
tUChar RestoreOK = 0;

  CPCCOLDRESET_TRACE_SHOW_INT(1);

  if (CPCStarted)
  {
    SoundStop(NativeCPC); // !! Should be done before CPCStop !!
  }

  CPCCOLDRESET_TRACE_SHOW_INT(2);

  while (!RestoreOK)
  {
    RestoreOK = 1;

    if (CPCStarted)
    {
      CPCCOLDRESET_TRACE_SHOW_INT(3);

      CPCStop();
      CPCStarted = 0;
    }

    CPCCOLDRESET_TRACE_SHOW_INT(4);

    Result = CPCStart();

    if (Result != errNone)
    {
      EventType newEvent;

#ifndef __RELEASE__
      TRACE_SHOW_TEXT("Fatal error", "CPCStart() != errNone");
#endif /* __RELEASE__ */

      // Terminate application
      newEvent.eType = appStopEvent;
      EvtAddEventToQueue(&newEvent);
    }
    else
    {
      CPCStarted = 1;
      
      CPCCOLDRESET_TRACE_SHOW_INT(5);

      StopKeyboard(&DIAKeyboardP);

      CPCCOLDRESET_TRACE_SHOW_INT(6);

      StartDIAKeyboard();

      CPCCOLDRESET_TRACE_SHOW_INT(7);

      if (IsFileExist(pathP, filenameP) == cTrue)
      {
        if ( (Load == AutomaticLoad) ||
             ( (Load == LoadWithConfirmation) && (FrmAlert(RestorationRequestAlert) == 0) ) )
        {
          tCPCRestoreReturnCode RestoreReturnCode;
          tPreferences RestorePref;

          CPCCOLDRESET_TRACE_SHOW_INT(8);

          // Restore previously saved CPC context
          RestoreReturnCode = RestoreCPC(pathP,
                                         filenameP,
                                         &EmulatorKeysStatus,
                                         &RestorePref,
                                         contextP);

          //
          // Restore_OK
          //
          if (RestoreReturnCode == Restore_OK)
          {
            CPCCOLDRESET_TRACE_SHOW_INT(9);

            // Save current session filename
            StrCopy(sessionFilenameP, filenameP);
            
            CPCCOLDRESET_TRACE_SHOW_INT(10);

            // Update colors due to NativeCPC->colours replacement.
            CPCSetColor();

            CPCCOLDRESET_TRACE_SHOW_INT(11);
            
            if (prefP->CPCKeycodeA != NativeCPC->HardKeyCPCKeyCodeA)
            {
            	prefP->CPCKeycodeA = NativeCPC->HardKeyCPCKeyCodeA;
              prefP->PreferencesChanged = 1;
            }
            if (prefP->CPCKeycodeB != NativeCPC->HardKeyCPCKeyCodeB)
            {
            	prefP->CPCKeycodeB = NativeCPC->HardKeyCPCKeyCodeB;
              prefP->PreferencesChanged = 1;
            }
            if (prefP->CPCKeycodeC != NativeCPC->HardKeyCPCKeyCodeC)
            {
            	prefP->CPCKeycodeC = NativeCPC->HardKeyCPCKeyCodeC;
              prefP->PreferencesChanged = 1;
            }
            if (prefP->CPCKeycodeD != NativeCPC->HardKeyCPCKeyCodeD)
            {
            	prefP->CPCKeycodeD = NativeCPC->HardKeyCPCKeyCodeD;
              prefP->PreferencesChanged = 1;
            }
            prefP->ScreenType = RestorePref.ScreenType;
            prefP->UseParados = RestorePref.UseParados;
            prefP->Use64kMemoryExtension = RestorePref.Use64kMemoryExtension;
            prefP->Mode2AntiAliasing = RestorePref.Mode2AntiAliasing;
            prefP->PreferencesChanged = 1;
          }
          //
          // Wrong_CPC_Model
          //
          else if (RestoreReturnCode == Wrong_CPC_Model)
          {
            CPCCOLDRESET_TRACE_SHOW_INT(12);

            prefP->CPCModel = RestorePref.CPCModel;

            RestoreOK = 0;  // Request start with new CPC Model
          }
          //
          // Error
          //
          else
          {
            DisplayRestoreError(RestoreReturnCode,
                                &RestorePref);
          }
        }
      }
    }
  } /* while (!RestoreOK) */

  CPCCOLDRESET_TRACE_SHOW_INT(13);

  if (CPCStarted)
  {
    EmulatorKeysStatus.SoundKeyStatus = KeyReleased;

    CPCCOLDRESET_TRACE_SHOW_INT(14);

    Result = SoundStart(contextP);
    if (Result == errNone)
    {
      if (prefP->SoundEnabled == true)
      {
        EmulatorKeysStatus.SoundKeyStatus = KeyPressed;
      }
    }

    RestoreEmulatorKeysStatus(&EmulatorKeysStatus);
  }

  CPCCOLDRESET_TRACE_SHOW_INT(15);

  return (Result);
}
/*----------------------------------------------------------------------------*/


#ifndef _USE_MACROS
UInt32 CPCExecute(void)
/***********************************************************************
 *
 *  CPCExecute
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (NativeCPC == NULL)
    ErrDisplay("NativeCPC == NULL");
#endif /* __RELEASE__ */

  return PceNativeCallMemPtr(PNOCPCExecuteP,
#ifdef __SIMU__
                             "Native_CPCExecute.dll\0PNOMain",
#endif /* __SIMU__ */
                             NativeCPC);
}
/*----------------------------------------------------------------------------*/
#endif /* _USE_MACROS */


UInt32 CPCSetColor(void)
/***********************************************************************
 *
 *  CPCSetColor
 *
 ***********************************************************************/
{
#ifndef __RELEASE__
  if (NativeCPC == NULL)
    ErrDisplay("NativeCPC == NULL");
#endif /* __RELEASE__ */

  NativeCPC->night_mode = prefP->NightModeActive;

  return PceNativeCallResource(Native_CPCSetColor,
#ifdef __SIMU__
                               "Native_CPCSetColor.dll\0PNOMain",
#endif /* __SIMU__ */
                               NativeCPC);
}
/*----------------------------------------------------------------------------*/


Boolean CPCHandleEvent(EventPtr Event)
/***********************************************************************
 *
 *  CPCHandleEvent
 *
 ***********************************************************************/
{
  if (Event->eType == penDownEvent)
  {
    return PenDownHandleEvent(Event);
  }
  else if (Event->eType == penUpEvent)
  {
    return PenUpHandleEvent(Event);
  }
  else if (Event->eType == penMoveEvent)
  {
    return PenMoveHandleEvent(Event);
  }

  if (SystemHalt)
    return false;

  if (Event->eType == keyDownEvent)
  {
    return KeyDownHandleEvent(Event);
  }

  return false;
}
/*----------------------------------------------------------------------------*/


void EmulatorFreeze(void)
/***********************************************************************
 *
 *  EmulatorFreeze
 *
 ***********************************************************************/
{
  SystemHalt = 1;

  SoundPause(NativeCPC);

  // Restaurer le mask des touches matrielles
  emulatorHardKeyMask = oldHardKeyMask;
  KeySetMask(emulatorHardKeyMask);
}
/*----------------------------------------------------------------------------*/


void EmulatorUnfreeze(void)
/***********************************************************************
 *
 *  EmulatorUnfreeze
 *
 ***********************************************************************/
{
  SystemHalt = 0;

  SoundPlay(NativeCPC);
  
  // Appliquer le masque des touches matrielles pour l'mulation
  SetHardKeysMask(prefP,
                  oldHardKeyMask);
}
/*----------------------------------------------------------------------------*/


void CPCSwapDrive(void)
/***********************************************************************
 *
 *  CPCSwapDrive
 *
 ***********************************************************************/
{
	SwapDrive(NativeCPC);
}
/*----------------------------------------------------------------------------*/


void CPCEjectBoth(void)
/***********************************************************************
 *
 *  CPCEjectBoth
 *
 ***********************************************************************/
{
	if (SaveAndEjectBoth(NativeCPC) == true)
	{
		// Refresh Drive Panel
    FrmUpdateForm(MainForm,
                  SubPanelRedrawUpdateCode);
	}
}
/*----------------------------------------------------------------------------*/


void SetRenderingParameters(tUChar Rendering)
/***********************************************************************
 *
 *  SetRenderingParameters
 *
 ***********************************************************************/
{
	switch (Rendering)
	{
		case 4: // Most realistic
		  VideoFrameInitialDelay = 1; // Render and Display video frame at each CPC frame completed
		  break;
		  
		case 3:
		  VideoFrameInitialDelay = 2; // Render and Display video frame each 2 CPC frames completed
		  break;
		  
		case 2:
		  VideoFrameInitialDelay = 3; // Render and Display video frame each 3 CPC frames completed
		  break;
		  
		case 1:
		  VideoFrameInitialDelay = 4; // Render and Display video frame each 4 CPC frames completed
		  break;
		  
		default: // 0 = fastest
		  VideoFrameInitialDelay = 5; // Render and Display video frame each 5 CPC frames completed
	}
}
/*----------------------------------------------------------------------------*/


void SetLightGunParameters(tPreferences* prefP,
                           tNativeCPC* NativeCPC)
/***********************************************************************
 *
 *  SetLightGunParameters
 *
 ***********************************************************************/
{
	NativeCPC->lightgun_random_crtc = prefP->OnScreenMagnumGunActive;
}
/*----------------------------------------------------------------------------*/
